/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include "include/support.h"
#include "include/environment.h"

#ifndef PACKAGE
#define PACKAGE "littlewizard"
#endif

GtkWidget *LWEnvironment::widget=NULL;
GtkWidget *LWEnvironment::program_paned=NULL;
GtkWidget *LWEnvironment::world_paned=NULL;
GtkWidget *LWEnvironment::both_paned=NULL;
LWPixmapSet *LWEnvironment::pixmapset=NULL;
LWBoardSet *LWEnvironment::icons=NULL;
LWBoardSet *LWEnvironment::commands=NULL;
LWProject *LWEnvironment::project=NULL;
xmlNode *LWEnvironment::wizard_xml=NULL;
LWDesign LWEnvironment::mode=LW_DESIGN_PROGRAM;
GData *LWEnvironment::symbols=NULL;
LWProgram *LWEnvironment::program=NULL;
gboolean LWEnvironment::enabled_grid=FALSE;

/*! \brief Tworzy obszar z suwakiem

	Tworzy obszar z suwakiem, modyfikując dodatkowo kolor tła obszaru.
	\param Kontrolka, która zostanie umieszczona w obszarze
*/
static GtkWidget *create_scrolled (GtkWidget *child)
{
	const GdkColor color = {0, 0x8000, 0x8000, 0x8000};

	GtkWidget *vp = gtk_viewport_new (NULL, NULL);
	gtk_widget_modify_bg (vp, GTK_STATE_NORMAL, &color);
	gtk_container_add (GTK_CONTAINER (vp), child);
	gtk_widget_show (vp);

	GtkWidget *sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					  GTK_POLICY_AUTOMATIC,
					  GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (sw), vp);
	gtk_widget_show (sw);

	return sw;
}

/*! \brief Inicjalizacja środowiska

	Inicjuje środowisko, odczytując ustawienia z pliku
*/
void LWEnvironment::init (GData *the_symbols)
{
static gboolean inited=FALSE;
xmlDoc *doc=NULL;
	
	g_assert (inited == FALSE);
	g_assert (the_symbols != NULL);
	
	inited = TRUE;
	symbols = the_symbols;
	
	xmlNode *root_node = initializeXMLTree (doc);
	g_assert (root_node != NULL);
	
	g_assert (root_node->children != NULL);
	pixmapset = new LWPixmapSet ();
	pixmapset->restoreFromXML (root_node->children);

	g_assert (root_node->children->next != NULL);
	wizard_xml = xmlCopyNode (root_node->children->next, 1);
	g_assert (wizard_xml != NULL);

	world_paned = gtk_vpaned_new ();
	g_object_ref (G_OBJECT (world_paned));
	gtk_widget_show (world_paned);

	g_assert (root_node->children->next->next != NULL);
	icons = new LWBoardSet (LW_TYPE_ICONS);
	icons->restoreFromXML (root_node->children->next->next);
	gtk_paned_add1 (GTK_PANED (world_paned), icons->getWidget());

	program_paned = gtk_vpaned_new ();
	g_object_ref (G_OBJECT (program_paned));
	gtk_widget_show (program_paned);

	g_assert (root_node->children->next->next->next != NULL);
	commands = new LWBoardSet (LW_TYPE_COMMANDS);
	commands->restoreFromXML (root_node->children->next->next->next);
	gtk_paned_add1 (GTK_PANED (program_paned), commands->getWidget());

	xmlFreeDoc (doc);

	widget = gtk_event_box_new ();
	
	setDesignMode (mode);
	gtk_widget_show (widget);
	
	both_paned = gtk_hpaned_new ();
	gtk_paned_set_position (GTK_PANED (both_paned), 400);
	g_object_ref (G_OBJECT (both_paned));
	gtk_widget_show (both_paned);
}

/*! \brief Deinicjalizacja środowiska

	Deinicjuje środowisko, zwalniając wszelkie zasoby przydzielone 
	podczas inicjalizacji.
*/
void LWEnvironment::uninit ()
{
	/* Niszczenie programu obecnego w środowisku */
	if (NULL != program)
		delete program;
	
	/* Niszczenie planszy z ikonami */
	delete icons;
	
	/* Niszczenie planszy z poleceniami */
	delete commands;
	
	/* Niszczenie projektu */
	if (NULL != project)
		delete project;

	/* Niszczenie zestawu ikon */
	delete pixmapset;
	
	/* Niszczenie stworzonych symboli, 
	które nie zostały zniszczone wcześniej */
	LWSymbol::destroyAll ();

	/* Niszczenie zbioru konstruktorów symboli języka */
	g_datalist_clear (&symbols);
	
	/* Niszczenie wzorca czarodzieja w XML */
	xmlFreeNode (wizard_xml);
	
	/* Zminiejszenie licznika referencji */
	g_object_unref (G_OBJECT (world_paned));
	g_object_unref (G_OBJECT (program_paned));
	g_object_unref (G_OBJECT (both_paned));
	
	/* Niszczenie głównego okna, wraz z pozostałymi jeszcze kontrolkami */
	gtk_widget_destroy (widget);
}

/*! \brief Inicjalizacja drzewa XML

	Ładuje plik konfiguracyjny oraz tworzy na jego podstawie drzewo
	\param doc DOM
	\return Korzeń drzewa XML
*/
xmlNode *LWEnvironment::initializeXMLTree (xmlDocPtr &doc)
{
	/*
     * inicjuje bibliotekę i sprawdza poptencjalne braki w ABI
     * pomiędzy wersją z którą został skompilowany program a
     * wersją użytą
     */
    LIBXML_TEST_VERSION

#ifdef _WIN32
	gchar *t = g_win32_get_package_installation_subdirectory(NULL, NULL, "share\\littlewizard");
	gchar *filename = g_build_filename (t, "littlewizard.xml", NULL);
	g_free (t);
#else
	gchar *filename = g_build_filename (PACKAGE_DATA_DIR, PACKAGE, "littlewizard.xml", NULL);
#endif
	
    /* Włącz opcje ignorowania białych znaków */
	xmlKeepBlanksDefault (0);
	
	/* przetwórz plik i pobierz DOM */
    doc = xmlParseFile(filename);
	
	g_free (filename);

	g_assert(doc != NULL);

    /* Pobierz węzeł korzenia */
    return xmlDocGetRootElement(doc);
}

/*! \brief Czyszczenie środowiska

	Wyjmuje z kontrolki środowiska, kontrolkę świata lub programu
*/
void LWEnvironment::clear()
{
	GtkWidget *child = gtk_bin_get_child (GTK_BIN (widget));

	if (child != NULL) {
		if (child == both_paned) {
			gtk_container_remove (GTK_CONTAINER(program_paned->parent), program_paned);
			gtk_container_remove (GTK_CONTAINER(world_paned->parent), world_paned);
		}

	gtk_container_remove (GTK_CONTAINER (widget), child);
	}
}

/*! \brief Pobranie zbioru ikon

	Pobiera zbiór ikon
	\return Pobrany zbiór
*/
LWPixmapSet *LWEnvironment::getPixmapSet ()
{
	return pixmapset;
}

/*! \brief Poranie informacji o czarodzieju

	Pobiera informację o czarodzieju w postaci węzła drzewa XML
	\return Pobrany węzeł
*/
xmlNode *LWEnvironment::getWizardXML ()
{
	return wizard_xml;	
}

/*! \brief Ustalenie trybu projektowego

	Ustawia tryb projektowy
	\param a_mode Ustawiany tryb
*/
void LWEnvironment::setDesignMode (LWDesign a_mode)
{
	mode = a_mode;

	clear();
	
	switch (mode) {
		case LW_DESIGN_WORLD:
			gtk_container_add (GTK_CONTAINER (widget), world_paned);
			break;
		
		case LW_DESIGN_PROGRAM:
			gtk_container_add (GTK_CONTAINER (widget), program_paned);
			break;
		
		case LW_DESIGN_MIXED:
			gtk_paned_add1 (GTK_PANED (both_paned), world_paned);
			gtk_paned_add2 (GTK_PANED (both_paned), program_paned);
			gtk_container_add (GTK_CONTAINER (widget), both_paned);
	}
	
}

/*! \brief Przydzielenie projektu

	Przydziela projekt, który będzie kontrolowany przez środowisko
	\param a_project Projekt
*/
void LWEnvironment::setProject (LWProject *a_project)
{
	g_return_if_fail (a_project != NULL);

	project = a_project;

	enableGrid (enabled_grid);
	
	gtk_paned_add2 (GTK_PANED (program_paned), 
		create_scrolled (project->getProgram()->getWidget()));
	gtk_paned_add2 (GTK_PANED (world_paned), 
		create_scrolled (project->getWorld()->getWidget()));
	
	setDesignMode (mode);
}

/*! \brief Zdjęcie projektu

	Informuje środowisko, że projekt nie będzie już używany przez
	środowisko
*/
void LWEnvironment::unsetProject ()
{
	g_return_if_fail (project != NULL);

	clear();
	
	gtk_widget_destroy (gtk_paned_get_child2 (GTK_PANED(program_paned)));
	gtk_widget_destroy (gtk_paned_get_child2 (GTK_PANED(world_paned)));
	
	project = NULL;
}

/*! \brief Pobranie projektu

	Pobranie projektu, nad którym ma kontrolę środowisko
	\return Pobrany projekt
*/
LWProject *LWEnvironment::getProject ()
{
	return project;
}

/*! \brief Zarajestrownie programu

	Rejestruj program, który jest wykonywany przez środowisko.
	Dotychczasowy program (jeśli obecny) jest zwalniany.
	\param a_program Przydzielony program
*/
void LWEnvironment::registerProgram (LWProgram *a_program)
{
	if (program != NULL)
		delete program;
	
	program = a_program;
}

/*! \brief Odrejestrownie programu

	Odrejstrowuje program. Program po zakończeniu a przed zniszczeniem
	samego siebie woła tę metodę
*/
void LWEnvironment::unregisterProgram ()
{
	program = NULL;
}

LWProgram *LWEnvironment::getProgram ()
{
	return program;
}

/*! \brief Budowanie nazwy pliku zawierającego ikonę

	Buduje nazwę pliku wraz ze ścieżką bezwzględną do pliku
	zawierającego ikonę
	\param file Nazwa pliku
	\return Nazwa pliku wraz ze ścieżką bezwzględną
*/
gchar *LWEnvironment::buildPixmapFilename (const gchar *file)
{
#ifdef _WIN32
	gchar *t = g_win32_get_package_installation_subdirectory(NULL, NULL, "share\\pixmaps");
	gchar *f = g_build_filename (t, file, NULL);
	g_free (t);
	return f;		
#else
	return g_build_filename (PACKAGE_PIXMAPS_DIR, file, NULL);
#endif
}

/*! \brief Pobranie symboli języka

	Pobierz opis symboli dostępnych w języku programowania
	obsługiwanym przez środowisko

	\return Tablica asocjacyjna z poleceniami języka
*/
GData *LWEnvironment::getSymbols ()
{
	if (symbols == NULL)
		g_warning ("Can't get symbols!");

	return symbols;
}

/*! \brief Włącza siatkę

	Włącz lub wyłącz siatkę w środowisku
	\param enable Jeśli prawda, siatka zostanie włączona. Jeśli nie
	siatka zostanie wyłączona
*/
void LWEnvironment::enableGrid (gboolean enable)
{
	enabled_grid = enable;
	
	if (project != NULL) {
		icons->enableGrid(enable);
		commands->enableGrid(enable);
		
		project->getWorld()->enableGrid(enable);
		project->getProgram()->enableGrid(enable);
	}		
}

/*! \brief Pobranie kontrolki
	
	Pobiera kontrolkę środowiska
	\return Pobrana kontrolka
*/
GtkWidget *LWEnvironment::getWidget ()
{
	return widget;
}
