# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import gobject
import gtk
import pango
from time import time
import threading

import utils

from player import Player
from helper import Dispatcher
from library import ListenDB, ListenDBQuery
from source import Source,SourceItem
from widget.misc import ScrolledWindow
from widget.view import MultiDragTreeview
from widget.song_view import SimpleSongView
from widget.song_menu import SongMenu
from cover_manager import CoverManager

CONTEXT_SONG_NUMBER = 20

def make_label(text):
    label = gtk.Label()
    label.set_use_markup(True)
    label.set_markup("<span size=\"large\"><b>"+text+"</b></span>")
    label.set_alignment(0,0.5)
    label.set_size_request(-1,22)
    label.set_ellipsize(pango.ELLIPSIZE_END)
    box = gtk.HBox(False,6)
    box.pack_start(gtk.Arrow(gtk.ARROW_RIGHT,gtk.SHADOW_NONE),False,False)
    box.pack_start(label,True,True)
    return box

class ContextBox(gtk.HBox):
    def __init__(self):
        super(ContextBox,self).__init__(True,6)
        
        #LASTEST
        self.lastest_view = ContextSongView("info_lastplayed","lastplayed")
        #PREFERED
        self.prefered_song_view = ContextSongView("info_current","playcount")
        
        self.albums_view = AlbumView(True)
        
        
        box = gtk.VBox(False,6)
        box.pack_start(make_label(_("Favorite albums")),False,False)
        box.pack_start(ScrolledWindow(self.albums_view),True,True)
        self.pack_start(box,True,True)
        
        box = gtk.VBox(False,6)
        box.pack_start(make_label(_("Favorite songs")),False,False)
        box.pack_start(ScrolledWindow(self.prefered_song_view),True,True)
        box.pack_start(make_label(_("Last played songs")),False,False)
        box.pack_start(ScrolledWindow(self.lastest_view),True,True)
        self.pack_start(box,True,True)
        
        
        self.update = False
        
        self.condition = threading.Condition()
        t = threading.Thread(target=self.thread)
        t.setDaemon(True)
        t.start()

        self.__db_query = ListenDBQuery("local")
        
        if ListenDB.isloaded():
            self.__library_loaded(ListenDB)
        else:
            ListenDB.connect("loaded",self.__library_loaded)
        
    def __library_loaded(self,db):
        self.condition.acquire()
        self.__db_query.set_query()
        Player.connect("new-song",self.refresh)
        self.refresh()
        self.condition.release()

    def refresh(self,*data):
        gobject.timeout_add(1500,self.idle_refresh)
        
    def idle_refresh(self):
        self.condition.acquire()    
        self.update = True
        self.condition.notify()
        self.condition.release()
        
        
    def thread(self):
        while True:
            self.condition.acquire()
            while not self.update:
                self.condition.wait()
            self.update = False
            self.condition.release()    
            
            all_songs = list(self.__db_query.get_songs())
            songs = all_songs[:]
            songs = [(song.get("#lastplayed"),song) for song in songs if song.get("#lastplayed")]
            songs.sort()
            songs.reverse()
            songs_lastest = [song[1] for song in songs[:CONTEXT_SONG_NUMBER]]

            songs = all_songs[:]
            songs = [(song.get("#playcount"),song) for song in songs if song.get("#playcount")]
            songs.sort()
            songs.reverse()
            songs = songs[:CONTEXT_SONG_NUMBER]
            songs_prefered = [song[1] for song in songs[:CONTEXT_SONG_NUMBER]]  
        
            def build_row(infos):
                _playcount, duration, album, songs = infos
                if album=="":
                    title = _("Unknown")
                else:
                    title = utils.xmlescape(album)
                nb_song = len(songs)
                if nb_song<=1:
                    value = _("song")
                else:
                    value = _("songs")
    
                duration  = " - "+utils.duration_to_string(duration)
    
                songs.sort()
                if not songs[0].get("artist"):
                    title += " - <i>"+_("Unknown")+"</i>"
                else:
                    title += " - <i>"+songs[0].get_str("artist",True)+"</i>"
    
                markup = "<b>"+title + "</b>\n%d"%nb_song+" "+value+duration
                
                try:pixbuf = CoverManager.get_pixbuf_from_song(songs[0], 40, 40, False) 
                except gobject.GError: pixbuf = CoverManager.DEFAULT_COVER_PIXBUF
                return (songs,markup,pixbuf)
            
            data = self.__db_query.get_info("album",extended_info=True)
            albums_prefered = [(playcount,duration,album,songs) for _key, (album,playcount,duration,songs) in data.iteritems() ]
            albums_prefered.sort()
            albums_prefered.reverse()
            
            albums_prefered = [build_row(info) for info in albums_prefered[:CONTEXT_SONG_NUMBER]]
                
            self.condition.acquire()
            if not self.update:
                gobject.idle_add(self.refresh_album,albums_prefered)
                if self.lastest_view.get_model():
                    gobject.idle_add(self.lastest_view.get_model().fill,songs_lastest) 
                if self.prefered_song_view.get_model():
                    gobject.idle_add(self.prefered_song_view.get_model().fill,songs_prefered)   
            self.condition.release()
            
    def refresh_album(self,albums):
        self.albums_view.model_song.clear()
        for album in albums:
            self.albums_view.model_song.append(album)
        
class CurrentInfoBox(gtk.HBox):
    def __init__(self):
        super(CurrentInfoBox,self).__init__(False,6)
        
        #LASTEST
        self.lastest_view = ContextSongView("info_lastplayed","lastplayed")
        #PREFERED
        self.prefered_song_view = ContextSongView("info_current","playcount")
        self.albums_view = AlbumView(True)
        
        self.albums_label = make_label(utils.xmlescape(_("Albums by")))
        self.prefered_label = make_label(utils.xmlescape(_("Favorite songs of")))
        self.lastest_label = make_label(utils.xmlescape(_("Last played songs of")))
        
        song_box = gtk.VBox(False,6)
        song_box.pack_start(self.albums_label,False,False)
        song_box.pack_start(ScrolledWindow(self.albums_view),True,True)
        self.pack_start(song_box,True,True)
        
        song_box = gtk.VBox(False,6)
        song_box.pack_start(self.prefered_label,False,False)
        song_box.pack_start(ScrolledWindow(self.prefered_song_view),True,True)
        song_box.pack_start(self.lastest_label,False,False)
        song_box.pack_start(ScrolledWindow(self.lastest_view),True,True)
        self.pack_start(song_box,True,True)
        
        
        self.cur_song = None

        Player.connect("new-song",self.refresh)

        self.condition = threading.Condition()
        t = threading.Thread(target=self.thread)
        t.setDaemon(True)
        t.start()
       
        self.__update = False

        self.__db_query = ListenDBQuery("local")
        
        if ListenDB.isloaded():
            self.__library_loaded(ListenDB)
        else:
            ListenDB.connect("loaded",self.__library_loaded)

    def __library_loaded(self,db):
        self.__db_query.set_query()
        self.condition.acquire()    
        self.__update = True
        self.condition.notify()
        self.condition.release()

        
    def refresh(self,db,song):
        if not song: return     
        if ListenDB.song_has_capability(song,"current"):
            self.condition.acquire()    
            self.cur_song = song
            self.condition.notify()
            self.condition.release()
            self.albums_label.get_children()[1].set_markup("<span size=\"large\"><b>"+utils.xmlescape(_("Albums by")) +" "+ song.get_str("artist",True)+"</b></span>")
            self.prefered_label.get_children()[1].set_markup("<span size=\"large\"><b>"+utils.xmlescape(_("Favorite songs of")) +" "+ song.get_str("artist",True)+"</b></span>")
            self.lastest_label.get_children()[1].set_markup("<span size=\"large\"><b>"+utils.xmlescape(_("Last played songs of")) +" "+ song.get_str("artist",True)+"</b></span>")
            
        else:   
            self.cur_song = song
            print "W:Context Refresh Not implemented for this Song type",self.cur_song.get_type()
            return
        
        
    def thread(self):
        while True:
            self.condition.acquire()
            while not self.cur_song or not self.__update:
                self.condition.wait()
            cur_song = self.cur_song
            self.cur_song = None
            self.condition.release()
            
            all_songs = list(self.__db_query.get_songs(artists=[cur_song.get_sortable("artist")]))
            songs = all_songs[:]
            songs = [(song.get("#lastplayed"),song) for song in songs if song.get("#lastplayed")]
            songs.sort()
            songs.reverse()
            songs_lastest = [song[1] for song in songs[:CONTEXT_SONG_NUMBER]]

            songs = all_songs[:]
            songs = [(song.get("#playcount"),song) for song in songs if song.get("#playcount")]
            songs.sort()
            songs.reverse()
            songs_prefered = [song[1] for song in songs[:CONTEXT_SONG_NUMBER]]  
            
            def build_row(infos):
                _playcount, duration, album, songs = infos
                if album=="":
                    title = _("Unknown")
                else:
                    title = utils.xmlescape(album)
                nb_song = len(songs)
                if nb_song<=1:
                    value = _("song")
                else:
                    value = _("songs")
    
                duration  = " - "+utils.duration_to_string(duration)
    
                songs.sort()
                markup = "<b>"+title + "</b>\n%d"%nb_song+" "+value+duration
                try:pixbuf = CoverManager.get_pixbuf_from_song(songs[0], 40, 40, False)
                except gobject.GError: pixbuf = CoverManager.DEFAULT_COVER_PIXBUF
                return (songs,markup,pixbuf)
        
            data = self.__db_query.get_info("album",artists=[cur_song.get_sortable("artist")],extended_info=True)
            albums_prefered = [(playcount,duration,album,songs) for _key, (album,playcount,duration,songs) in data.iteritems() ]
            albums_prefered.sort()
            albums_prefered.reverse()
        
            albums_prefered = [build_row(infos) for infos in albums_prefered]
        
            self.condition.acquire()
            if not self.cur_song:
                gobject.idle_add(self.prefered_song_view.get_model().fill,songs_prefered) 
                gobject.idle_add(self.lastest_view.get_model().fill,songs_lastest) 
                gobject.idle_add(self.refresh_album,albums_prefered)    
            self.condition.release()
        
        
        
    def refresh_album(self,albums):
        self.albums_view.model_song.clear()
        for album in albums:
            self.albums_view.model_song.append(album)
        
class ContextSongView(SimpleSongView):
    show_all_columns = True
    def __init__(self,conf_prefix,info_to_display,show_album=True):
        
        self.info_to_display = info_to_display
        self.show_album = show_album
        
        kwargs = { "title":(_("Title"),gobject.TYPE_STRING)}
        super(ContextSongView,self).__init__(conf_prefix,** kwargs)
        
        self.set_menu(SongMenu())

        self.set_headers_visible(False)
        
        c = self.get_column(0)
        c.set_cell_data_func(c.get_cell_renderers()[0], self.cell_data_func)

    def cell_data_func(self, column, cell, model, iter):
        song = model.get_value(iter, 0)
        text = "<span><b>"+song.get_str("title",True)+"</b></span><span size=\"small\">"
        if self.info_to_display=="playcount":
            text +=  " <i>"+_("played")+" "+song.get_str("#playcount",True)+" "+_("times")+"</i>"
        elif self.info_to_display=="lastplayed":
            if False and song.get("#lastplayed")!=None:
                text += " <i>"+_("played")+" "+utils.duration_to_string(time()-song.get("#lastplayed"),"00:00",1)+" "+_("ago")+"</i>"
                
        sep = ""
        if song.get("artist"):
            sep = " - "
            text += "\n<i>"+song.get_str("artist",True)+"</i>"
        if self.show_album and song.get("album"):
            text += sep+song.get_str("album",True)
        text += "</span>"

        cell.set_property("markup",text )

class AlbumView(MultiDragTreeview):
    def __init__(self,show_artist=False):
        super(AlbumView,self).__init__()

        self.show_artist = show_artist
        self.model_song = gtk.ListStore(object,str,gtk.gdk.Pixbuf)

        self.renderer_cover = gtk.CellRendererPixbuf()
        self.renderer_cover.set_property("height",50)
        self.renderer_cover.set_property("width",50)
        
        self.renderer_album = gtk.CellRendererText()
        self.renderer_album.set_property("height",50)
        self.renderer_album.set_property("ellipsize",pango.ELLIPSIZE_END)
        #self.renderer_album.set_property("yalign",0)

        targets = [("text/listen-songs", gtk.TARGET_SAME_APP, 1),("text/uri-list", 0, 2)]
        self.enable_model_drag_source(
            gtk.gdk.BUTTON1_MASK|gtk.gdk.SHIFT_MASK|gtk.gdk.CONTROL_MASK, targets,
            gtk.gdk.ACTION_COPY)

        self.column_cover = gtk.TreeViewColumn("",self.renderer_cover,pixbuf=2)
        self.column_album = gtk.TreeViewColumn("",self.renderer_album,markup=1)
        self.column_album.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)

        self.append_column(self.column_cover)
        self.append_column(self.column_album)
        self.set_headers_visible(False)
        self.set_rules_hint(True)
        self.connect("row-activated", self.on_row_song_activated)
        self.connect("drag-data-get", self.on_song_drag_data_get)
        self.connect("popup-menu",self.__popup_menu)

        self.set_model(self.model_song)
        self.menu = SongMenu()
        self.menu.disable(["edit","delete","deletedisk"])


    def __popup_menu(self,widget):
        model,rows = self.get_selection().get_selected_rows()
        if len(rows)<0:
            return
        else:
            songs = []
            for row in rows: 
                songs.extend(model[row[0]][0])
            songs.sort()
            if self.menu and songs:
                self.menu.popup(songs)
                
    def on_row_song_activated(self, treeview, path, view_column):
        model, rows = self.get_selection().get_selected_rows()
        for row in rows:
            songs = model.get_value(model.get_iter(row[0]),0)
            songs.sort()
            Dispatcher.cur_playlist_play(songs)


    def on_song_drag_data_get(self,treeview, context, selection, info, timestamp):

        model,rows = treeview.get_selection().get_selected_rows()
        if len(rows)<0:
            return
        else:
            songs = []
            for row in rows: 
                songs.extend(model[row[0]][0])
        songs.sort()
        list_uri = list([ song.get("uri") for song in songs])
        selection.set("text/listen-songs", 8, "\n".join(list_uri))
        selection.set_uris(list_uri)



        


class ContextItem(SourceItem):
    widget_klass = ContextBox
    config_code = "context"
    stock = gtk.STOCK_HOME
    label = _("Context")

class CurrentInfo(SourceItem):
    widget_klass = CurrentInfoBox
    config_code = "current"
    label = _("Current")
    stock = gtk.STOCK_INFO

class InfoSource(Source):
    PLUGIN_NAME = "Context"
    PLUGIN_DESC = _("Display Statistics about most played track")

    categorie = "info"
    
    display_index = 0

    def __init__(self):
        Source.__init__(self)
        self.items = [ ContextItem() ]

class CurrentSource(Source):
    PLUGIN_NAME = "Current Info"
    PLUGIN_DESC = _("Display information about current played artist")

    categorie = "info"
    display_index = 1
    def __init__(self):
        Source.__init__(self)
        self.items = [ CurrentInfo() ]

