#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>

#ifndef WIN32
#include <unistd.h>
#endif

#include "core.h"
#include "list.h"
#include "config.h"
#include "smtp.h"
#include "user.h"
#include "flag.h"
#include "fileapi.h"
#include "variables.h"
#include "mystring.h"
#include "compat.h"
#include "submodes.h"

int inbody;
int request_type;

int user_read(FILE *userfile, struct list_user *user)
{
    char tempbuf[HUGE_BUF];
    char address[BIG_BUF];
    char flags[BIG_BUF];
    int assigned;
    int validate = get_bool("validate-users");

    while(read_file(&tempbuf[0], BIG_BUF, userfile)) {
       if(tempbuf[strlen(tempbuf)-1] == '\n')
           tempbuf[strlen(tempbuf)-1]='\0'; /* get rid of \n */
       assigned = sscanf(&tempbuf[0], "%s : %s", &address[0], &flags[0]);
       if(assigned != 2) {
           log_printf(0, "Invalid userfile line '%s'\n", tempbuf);
           continue;
       }
       if(validate) {
           if(check_address(address) == 0) {
               log_printf(0, "BOGUS email address '%s'\n", address);
               /* Only log the line, don't through it away */
               /* continue; */
           }
       }
       strcpy(user->address,address);
       strcpy(user->flags,flags);
       return 1;
    }
    return 0;
}

int user_find_list(const char *listname, const char *addy,
                   struct list_user *user)
{
    char buffer[BIG_BUF];
    char *listdir;
    listdir = list_directory(listname);
    sprintf(buffer,"%s/users",listdir);
    free(listdir);
    return(user_find(&buffer[0],addy,user));
}

int user_find(const char *listusers, const char *addy, struct list_user *user)
{
    FILE *userfile;
    int found;
    int loose = get_bool("no-loose-domain-match");

    if (!exists_file(listusers)) return 0;

    if((userfile = open_file(listusers,"r")) == NULL) return 0;
 
    found = 0;
    while(!found) {
        if(!user_read(userfile,user)) break;
        if(loose) {
            if (!strcasecmp(user->address,addy))
                found = 1;
        } else {
            if(address_match(addy, user->address))
                found = 1;
        }
    }

    close_file(userfile);
    return found;
}

int user_add(const char *listusers, const char *fromaddy)
{
    struct list_user user;
    const char *userflags = NULL;

    strcpy(&(user.address[0]),fromaddy);
    if(get_var("submodes-mode"))
        userflags = get_submode_flags(get_string("submodes-mode"));
    if(!userflags)
        userflags = get_submode_flags("default");
    if(!userflags)
        userflags = get_var("default-flags");
    sprintf(user.flags, "%s", userflags);
    return user_write(listusers, &user);
}


int user_remove(const char *listusers, const char *fromaddy)
{
    FILE *myfile;

    if (!exists_file(listusers)) return ERR_LISTOPEN;

    if ((myfile = open_exclusive(listusers, "r+")) == NULL) {
        return ERR_LISTOPEN;
    } else {
        char buf[BIG_BUF];
        FILE *newfile;
        char temp[BIG_BUF];

        sprintf(buf, "%s.new", listusers);
        if ((newfile = open_exclusive(buf, "w+")) == NULL) {
            close_file(myfile);
            return ERR_LISTCREATE;
        } else {
            while(read_file(buf,BIG_BUF,myfile)) {
                sscanf(buf, "%s", &temp[0]);
                if (strcasecmp(temp,fromaddy)) {
                    if (!write_file(newfile,"%s", buf)) {
                        close_file(newfile);
                        close_file(myfile);
                        return ERR_LISTWRITE;
                    }
                }
            }
            rewind_file(newfile);
            rewind_file(myfile);
            truncate_file(myfile,0);
            while(read_file(buf,BIG_BUF,newfile)) {
               write_file(myfile,"%s", buf);
            }
            close_file(myfile);
            close_file(newfile);
            sprintf(buf, "%s.new", listusers);
            (void)unlink_file(buf);
            return 0;
        }
    }
}

int user_write(const char *listusers, struct list_user *user)
{
    FILE *myfile;
    char buf[BIG_BUF];
    FILE *newfile;
    char temp[BIG_BUF];
    int didwrite;

    if ((myfile = open_exclusive(listusers, "r+")) == NULL)
        return ERR_LISTOPEN;

    didwrite = 0;

    sprintf(buf, "%s.new", listusers);
    if ((newfile = open_file(buf, "w+")) == NULL) {
       close_file(myfile);
       return ERR_LISTCREATE;
    }

    while(read_file(buf,BIG_BUF,myfile)) {
        sscanf(buf, "%s", &temp[0]);
        if (strcasecmp(&temp[0],user->address)) {
            write_file(newfile,"%s", buf);
        } else {
            didwrite = 1;
            write_file(newfile,"%s : %s\n", user->address, user->flags);
        }
    }
    if (!didwrite) {
        write_file(newfile, "%s : %s\n", user->address, user->flags);
    }
    rewind_file(newfile);
    rewind_file(myfile);
    truncate_file(myfile, 0);
    while(read_file(buf, BIG_BUF, newfile)) {
       write_file(myfile, "%s", buf);
    }
    close_file(myfile);
    close_file(newfile);
    sprintf(buf,"%s.new",listusers);
    unlink_file(buf);
    return 0;
}

int user_hasflag(struct list_user *user, const char *flag)
{
    char tbuf[64];

    sprintf(tbuf, "|%s|", flag);

    return ((strstr(user->flags,tbuf)) != NULL);
}

int user_setflag(struct list_user *user, const char *flag, int admin)
{
    char tbuf[HUGE_BUF];
    struct listserver_flag *flagrec;

    flagrec = get_flag(flag);

    if (!flagrec) return ERR_NOSUCHFLAG;

    if (flagrec->admin) {
        if ((!admin) && 
           !((flagrec->admin & ADMIN_SAFESET) && user_hasflag(user,"ADMIN")))
           return ERR_NOTADMIN;

        if (flagrec->admin & ADMIN_UNSETTABLE)
           return ERR_UNSETTABLE;
    }

    if (user_hasflag(user,flag)) return ERR_FLAGSET;

    if (user->flags[strlen(user->flags) - 1] != '|') {
        sprintf(tbuf,"%s|%s|",user->flags,flag);
    } else {
        sprintf(tbuf,"%s%s|",user->flags,flag);
    }

    sprintf(user->flags,"%s",tbuf);

    return 0;   
}

int user_unsetflag(struct list_user *user, const char *flag, int admin)
{
    struct listserver_flag *flagrec;
    char tbuf[HUGE_BUF];
    char fbuf[64];
    char *fptr;

    flagrec = get_flag(flag);

    if (!flagrec) return ERR_NOSUCHFLAG;

    if (flagrec->admin) {
        if ((!admin) &&
           !((flagrec->admin & ADMIN_SAFEUNSET) && user_hasflag(user,"ADMIN")))
           return ERR_NOTADMIN;
    }

    if (!user_hasflag(user,flag)) return ERR_FLAGNOTSET;
    sprintf(fbuf,"%s|",flag);

    sprintf(tbuf,"%s",user->flags);

    fptr = strstr(tbuf,fbuf);
    *fptr++ = '\0';
    while(*fptr && (*fptr != '|')) {
        fptr++;
    }
    sprintf(user->flags,"%s%s",tbuf,++fptr);

    return 0;
}
