package Lire::WWW::Filename;

use strict;

use vars qw/$REVISION $VERSION/;

use Lire::WWW::Filename::Attack;

use Carp;

BEGIN {
    $REVISION  = '$Id: Filename.pm,v 1.4 2002/02/03 13:17:20 vanbaal Exp $';
    $VERSION   = "0.1.0";
}

my %customAttackHash = ();
my %compiledAttackHash = ();
my %notInAttackHash = ();

# -----------------------------------------------------------------------------
# Functions
# -----------------------------------------------------------------------------

sub new {
     my ($class) = @_;

     my $self = {};
     bless $self, $class;

     return $self;
}

sub setFilename {
    my ($self, $filename) = @_;

    if (defined $filename) {
        $self->{filename} = lc($filename);
    }
}

sub getAttack {
    my ($self) = @_;

    my $filename = $self->{filename};
    if (defined $filename) {
        # for filenames this speed up might not be useful (FIXME)
        return "Unknown/No Attack" if ($notInAttackHash{$filename});

        if (defined (my $attack = Lire::WWW::Filename::Attack::getAttack($filename))) {
            return $attack;
        } else {
            # check custom hash
            foreach my $key (keys %customAttackHash) {
		my $re = $compiledAttackHash{$key};
                if ($filename =~ /$re/) {
                    return $customAttackHash{$key};
                }
            }
            $notInAttackHash{$filename} = "1";
            return "Unknown/No Attack";
        }
    }
    return undef;
}

sub addAttack {
    my ($self, $code, $value) = @_;

    croak "Must supply both code and value!\n"
      unless $code eq "" || $value eq "";

    $code = lc $code;
    $customAttackHash{$code} = $value;
    $compiledAttackHash{$code} = qr/$code/;
}

sub addAttackHash {
    my ($self, %Attackhash) = @_;

    foreach my $code (keys %Attackhash) {
        $self->addAttack($code, $Attackhash{$code});
    }
}

1;

__END__

=pod

=head1 NAME

Lire::WWW::Filename - analyze filenames

=head1 SYNOPSIS

 use Lire::WWW::Filename;

 my $test_string = "/default.ida?######...";

 my $analyzer = new Lire::WWW::Filename();
 $analyzer->setFilename($test_string);

 print "  Attack: " . $analyzer->getAttack() . "\n";

=head1 DESCRIPTION

This module offers an interface to databases with known information useful in
analyzing filenames as found in Apache logfiles.  One could use it to analyze
attacks on known webserver vulnerabilities, consisting of specially crafted
HTTP GET requests.

Optionally, the user can add custom information to the database with
the addX() and addXHash() functions.

=head1 SEE ALSO

Lire::Extensions::WWW::AttackSchema(3)

=head1 VERSION

$Id: Filename.pm,v 1.4 2002/02/03 13:17:20 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2001  Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHORS

Egon Willighagen <egonw@logreport.org>

=cut

