package tests::CUPSTest;

use strict;

use base qw/Lire::Test::DlfConverterTestCase/;

use Lire::DlfConverters::CUPS;
use File::Basename;

sub schemasdir {
    return dirname( __FILE__ ) . "/../..";
}

sub create_converter {
    return new Lire::DlfConverters::CUPS();
}

sub registration_file {
    return dirname( __FILE__ ) . "/../../cups_init";
}

sub test_init_dlf_converter {
    my $self = $_[0];

    my $converter = $self->create_converter();
    $self->assert_null( $converter->{'dlf'} );
    $converter->init_dlf_converter( undef, undef );
    $self->assert_deep_equals( {}, $converter->{'dlf'} );
}

sub test_process_log_line {
    my $self = $_[0];

    my $process = $self->create_mock_process();
    my $converter = $process->converter();

    $converter->process_log_line( $process, 'P450 kurt 117 [10/Aug/2001:21:51:53 +0100] 5 1' );

    my $dlf = { 'job-id' => 117,
                'user' => 'kurt',
                'printer' => 'P450',
                'num_pages' => '5',
                'num_copies' => '1',
                'billing' => undef,
                'time' => 997476713,
                'duration' => 0,
              };
    $self->assert_deep_equals( $dlf, $converter->{'dlf'} );
    $self->assert_num_equals( 0, $process->dlf_count() );

    $converter->process_log_line( $process, 'P450 kurt 117 [10/Aug/2001:21:51:55 +0100] 6 1' );

    $dlf->{'duration'} = 2;
    $dlf->{'num_pages'} = 6;
    $self->assert_deep_equals( $dlf, $converter->{'dlf'} );
    $self->assert_num_equals( 0, $process->dlf_count() );

    $converter->process_log_line( $process, 'raw  137 [19/Aug/2001:16:58:58 +0100] 1 1' );
    $self->assert_deep_equals( [ $dlf ], $process->dlf( 'print' ) );
    $self->assert_deep_equals( { 'job-id' => 137,
                                 'user' => undef,
                                 'time' => 998236738,
                                 'duration' => 0,
                                 'billing' => undef,
                                 'printer' => 'raw',
                                 'num_pages' => 1,
                                 'num_copies' => 1 }, $converter->{'dlf'} );
}

sub test_process_log_line_errors {
    my $self = $_[0];

    my $process = $self->create_mock_process();
    my $converter = $process->converter();

    $converter->process_log_line( $process, 'P450' );
    $self->assert_num_equals( 1, $process->error_count() );
    $self->assert_deep_equals( [ 'invalid page_log line', 'P450' ], $process->errors()->[0] );

    $converter->process_log_line( $process, 'P450 kurt 117 [10/XXX/2001:21:51:55 +0100] 6 1' );
    $self->assert_num_equals( 2, $process->error_count() );
    $self->assert_matches( qr/syslog2tm cannot get month/,
                           $process->errors()->[1][0] );
}

sub test_finish_conversion {
    my $self = $_[0];

    my $process = $self->create_mock_process();
    my $converter = $process->converter();

    $converter->finish_conversion( $process );
    $self->assert_num_equals( 0, $process->dlf_count() );

    $converter->{'dlf'}{'job-id'} = 'An ID';
    $converter->finish_conversion( $process );
    $self->assert_num_equals( 1, $process->dlf_count() );
    $self->assert_deep_equals( [ { 'job-id' => 'An ID' } ],
                               $process->dlf( 'print' ) );
}


1;
