package tests::FunctionalTestCaseTest;

use strict;

use base qw/Test::Unit::TestSuite/;

sub name {
    return "Lire::Test::FunctionalTestCase Tests";
}

sub include_tests {
    return qw/tests::FunctionalTestCaseTest::init
              tests::FunctionalTestCaseTest::api
            /;
}

package tests::FunctionalTestCaseTest::base;

use strict;

use base qw/Lire::Test::TestCase/;

use Lire::Config::Build qw/ ac_info ac_path /;
use Lire::Utils qw/ tempdir create_file file_content /;

use Cwd 'realpath';
use File::Basename;
use File::Path;

sub set_up_fake_lire_installation {
    my $self = $_[0];

    $self->{'testdir'} = realpath( dirname( __FILE__));
    $self->{'lire_root'} = tempdir( "lire_XXXXXX", 'CLEANUP' => 1 );
    $self->{'saved_build_infos'} = \%Lire::Config::Build::ac;
    %Lire::Config::Build::ac =
      ( %{$self->{'saved_build_infos'}},
        'bindir' => "$self->{'lire_root'}/bin",
        'sysconfdir' => "$self->{'lire_root'}/etc",
        'PACKAGE' => 'lire',
        'datadir' => "$self->{'lire_root'}/share",
        'LR_PERL5LIB' => "$self->{'testdir'}/..",
      );

    mkpath( [ "$self->{'lire_root'}/etc/lire",
              "$self->{'lire_root'}/bin",
              "$self->{'lire_root'}/libexec/lire",
              "$self->{'lire_root'}/share/lire/config-spec",
              "$self->{'lire_root'}/share/lire/schemas",
              "$self->{'lire_root'}/share/lire/filters",
              "$self->{'lire_root'}/share/lire/reports",
            ], 0, 0755 );

    create_file( "$self->{'lire_root'}/bin/lr_xml2report",
                 "#!/bin/sh -e\ntrue" );
    chmod 0755, "$self->{'lire_root'}/bin/lr_xml2report"
      or $self->error( "chmod on lr_xml2report failed: $!\n" );

    create_file( "$self->{'lire_root'}/etc/lire/defaults",
                 "#!/bin/sh -e\ntrue" );
    create_file( "$self->{'lire_root'}/share/lire/config-spec/lire.xml",
                 file_content(  "$self->{'testdir'}/../config-spec/lire.xml" ));

    return;
}

sub tear_down_fake_lire_installation {
    my $self = $_[0];

    %Lire::Config::Build::ac = %{$self->{'saved_build_infos'}};

    return;
}

package tests::FunctionalTestCaseTest::init;

use base qw/tests::FunctionalTestCaseTest::base/;

use Lire::Config;
use Lire::PluginManager;
use Lire::Test::FunctionalTestCase;
use Lire::Utils qw/tempfile/;

sub set_up {
    my $self = $_[0];

    $self->SUPER::set_up();
    $self->set_up_fake_lire_installation();

    return;
}

sub tear_down {
    my $self = $_[0];

    $self->SUPER::tear_down();
    $self->tear_down_fake_lire_installation();

    return;
}

sub test_new {
    my $self = $_[0];

    my $test = new Lire::Test::FunctionalTestCase();
    $self->assert_isa( 'Lire::Test::FunctionalTestCase', $test );

    chmod 0000, "$self->{'lire_root'}/bin/lr_xml2report"
      or $self->error( "failed to remove permissions on lr_xml2report: $!" );
    $self->assert_dies( qr/isn't an executable. Was Lire installed?/,
                        sub { new Lire::Test::FunctionalTestCase() });

    chmod 0755, "$self->{'lire_root'}/bin/lr_xml2report"
      or $self->error( "failed to put back permissions on lr_xml2report: $!" );

    chmod 0000, "$self->{'lire_root'}/etc/lire/defaults"
      or $self->error( "failed to remove permissions on defaults: $!" );
    $self->assert_dies( qr/isn't present. Was Lire installed\?/,
                      sub { new Lire::Test::FunctionalTestCase() } );
    chmod 0664, "$self->{'lire_root'}/etc/lire/defaults"
      or $self->error( "failed to put back permission on defaults: $!" );

    foreach my $d ( "$self->{'lire_root'}/etc/lire",
                    "$self->{'lire_root'}/share/lire" )
    {
        chmod 0555, $d
          or $self->error( "failed to remove permissions on $d: $!" );
        $self->assert_dies( qr/you need write access to Lire's.*$d/,
                            sub { new Lire::Test::FunctionalTestCase() } );
        chmod 0755, "$d"
          or $self->error( "failed to remove permissions on $d: $!" );
    }
}

sub test_set_up {
    my $self = $_[0];

    my @or_inc = @INC;
    my $test = new Lire::Test::FunctionalTestCase();
    $test->set_up();
    foreach my $a ( qw/_rundir _stdout_file _stderr_file _homedir/ ) {
        $self->assert_not_null( $test->{$a}, "set_up() should set $a" );
    }

    foreach my $d ( qw/_rundir _homedir/ ) {
        $self->assert( -d $test->{$d}, "$d '$test->{$d}' wasn't created" );
    }

    foreach my $d ( qw/ reports schemas filters config converters plugins / ) {
        my $dir = $test->{'_homedir'} . "/.lire/$d";
        $self->assert( -d $dir,  "directory '$dir' wasn't created" );
    }

    $self->assert( -f "$test->{'_homedir'}/.lire/config/func_test_sendmail.xml",
                   "Configuration file for fake sendmail not created." );

    my ( $fh, $filename ) = tempfile( "testXXXXXX" );
    push @{$test->{'_test_files'}}, $filename;

    $self->assert_num_equals(  @or_inc + 1, scalar @INC );
    $self->assert_str_equals( "$self->{'testdir'}/..", $INC[0] );

    $test->tear_down();
    foreach my $d ( qw/_rundir _homedir/ ) {
        $self->assert( !-d $test->{$d}, "$d '$test->{$d}' wasn't deleted by tear_down" );
    }
    $self->assert( !-f $filename, "$filename should have been deleted" );
    $self->assert_deep_equals( \@or_inc, \@INC );
}

sub test_set_up_lire_globals {
    my $self = $_[0];

    my $home = $ENV{'HOME'};
    my $config = $Lire::Config::SINGLETON;
    my $mgr = $Lire::PluginManager::instance;

    my $test = new Lire::Test::FunctionalTestCase();
    $test->{'_homedir'} = 'mydir';
    $test->_set_up_lire_globals();
    $self->assert_isa( 'Lire::Config::XMLFilesConfig',
                       $Lire::Config::SINGLETON );
    $self->assert_isa( 'Lire::PluginManager',
                       $Lire::PluginManager::instance );
    $self->assert_str_equals( $mgr, $test->{'_old_mgr'} );
    $self->assert_str_equals( $config, $test->{'_old_config'} );
    $self->assert_str_not_equals( $config, $Lire::Config::SINGLETON );
    $self->assert_str_not_equals( $mgr,
                                  $Lire::PluginManager::instance );
    $self->assert_str_equals( 'mydir', $ENV{'HOME'} );
    $self->assert_str_equals( $home, $test->{'_old_homedir'} );

    $test->_tear_down_lire_globals();
    $self->assert_str_equals( $config, $Lire::Config::SINGLETON );
    $self->assert_str_equals( $mgr, $Lire::PluginManager::instance );
    $self->assert_str_equals( $home, $ENV{'HOME'} );
}

package tests::FunctionalTestCaseTest::api;

use base qw/tests::FunctionalTestCaseTest::base/;

use Lire::Config::Build qw/ ac_info ac_path /;
use Lire::Utils qw/file_content/;
use Lire::Test::FunctionalTestCase;

sub set_up  {
    my $self = $_[0];

    $self->set_up_fake_lire_installation();

    $self->{'functional'} = new Lire::Test::FunctionalTestCase();
    $self->{'functional'}->set_up();
}

sub tear_down {
    my $self = $_[0];

    $self->tear_down_fake_lire_installation();

    $self->{'functional'}->tear_down();
}

sub test_lire_run {
    my $self = $_[0];

    $self->assert_dies( qr/missing 'cmd' parameter/,
                        sub { $self->{'functional'}->lire_run() } );

    my $result = $self->{'functional'}->lire_run( "echo This is a test" );
    $self->assert_isa( "Lire::Test::CommandResult", $result );

    $self->assert_equals( 0, $result->status() );
    $self->assert_equals( "This is a test\n", $result->stdout() );
    $self->assert( !$result->stderr(), "stderr should be empty" );

    $result = $self->{'functional'}->lire_run( 'echo $PATH' );
    $self->assert_equals( 0, $result->status() );

    my $bindir = ac_info( 'bindir' );
    $self->assert_matches( qr/^$bindir/, $result->stdout() );
 }

sub test_locale {
    my $self = $_[0];

    $self->assert_str_equals( 'C', $self->{'functional'}{'_locale'} );
    my $rv = $self->{'functional'}->locale( 'fr' );
    $self->assert_str_equals( 'C', $rv );
    $self->assert_str_equals( 'fr', $self->{'functional'}{'_locale'} );

    $rv = $self->{'functional'}->locale();
    $self->assert_str_equals( 'fr', $rv );
    $self->assert_str_equals( 'C', $self->{'functional'}{'_locale'} );
}

sub test_lire_run_locale {
    my $self = $_[0];

    my $cmd = 'echo LANG=$LANG LC_ALL=$LC_ALL LANGUAGE=$LANGUAGE';
    my $result = $self->{'functional'}->lire_run( $cmd );
    $self->assert_str_equals( "LANG= LC_ALL=C LANGUAGE=\n",
                              $result->stdout() );

    $self->{'functional'}->locale( 'fr' );
    $result = $self->{'functional'}->lire_run( $cmd );
    $self->assert_str_equals( "LANG= LC_ALL=fr LANGUAGE=\n",
                              $result->stdout() );
}

sub test_create_test_file {
    my $self = $_[0];

    my $file = $self->{'functional'}->rundir() . "/test_file";
    my $content = <<EOF;
This is the content of the file.

That was created.
EOF
    $self->{'functional'}->create_test_file( $file, $content );
    $self->assert( -f $file, "file $file doesn't exist" );
    $self->assert( grep { $_ eq $file } @{$self->{'functional'}{'_test_files'}},
                   "$file wasn't added to _test_files" );

    $self->assert_equals( $content, file_content( $file ) );

    ;
    $self->assert_dies( qr/file already exists:/,
                        sub { $self->{'functional'}->create_test_file( $file, $content ) });

    ;
    $self->assert_dies( qr/missing 'filename' parameter/,
                        sub { $self->{'functional'}->create_test_file() } );

    my $empty_file = $self->{'functional'}->rundir() . "/empty_file";
    $self->{'functional'}->create_test_file( $empty_file );
    $self->assert( -f $empty_file, "file $empty_file wasn't created" );
    $self->assert( ! -s $empty_file, "$empty_file isn't empty" );
}

sub test_install_xml_spec {
    my $self = $_[0];

    my $mock_file = $self->{'functional'}->rundir() . "/test-mock.xml";
    open my $fh, "> $mock_file"
      or $self->error( "can't create $mock_file in rundir: $!" );
    print $fh <<EOF;
Mock file
EOF
    close $fh;

    $self->assert_dies( qr/invalid spec_type parameter/,
                        sub { $self->{'functional'}->install_xml_spec( "bad_param" ) } );

    my $base_site = ac_path( "datadir", "PACKAGE" );
    $self->{'functional'}->install_xml_spec( "site_filter", "func_test",
                                           $mock_file );
    $self->assert( -f $base_site . "/filters/func_test/test-mock.xml",
                   "file wasn't installed properly as site_filter" );
    $self->{'functional'}->install_xml_spec( "site_report", "func_test",
                                           $mock_file );
    $self->assert( -f $base_site . "/reports/func_test/test-mock.xml",
                   "file wasn't installed properly as site_report" );
    $self->{'functional'}->install_xml_spec( "site_schema", "test", $mock_file );
    $self->assert( -f $base_site . "/schemas/test-mock.xml",
                   "file wasn't installed properly as site_schema" );
    foreach my $f ( qw!filters/func_test/test-mock.xml
                       reports/func_test/test-mock.xml
                       schemas/test-mock.xml ! )
    {
        my $file = "$base_site/$f";
        my $found = grep { $_ eq $file } @{$self->{'functional'}{'_test_files'}};
        $self->assert( $found, "file $file wasn't added to list of files to remove" );
    }

    my $base_home = $self->{'functional'}->homedir() . "/.lire";
    $self->{'functional'}->install_xml_spec( "filter", "func_test", $mock_file );
    $self->assert( -f $base_home . "/filters/func_test/test-mock.xml",
                   "file wasn't installed properly as filter" );
    $self->{'functional'}->install_xml_spec( "report", "func_test", $mock_file );
    $self->assert( -f $base_home . "/reports/func_test/test-mock.xml",
                   "file wasn't installed properly as report" );
    $self->{'functional'}->install_xml_spec( "schema", "tes", $mock_file );
    $self->assert( -f $base_home . "/schemas/test-mock.xml",
                   "file wasn't installed properly as schema" );

    foreach my $f ( qw!filters/func_test/test-mock.xml
                       reports/func_test/test-mock.xml
                       schemas/test-mock.xml ! )
    {
        my $file = "$base_home/$f";
        my $found = grep { $_ eq $file } @{$self->{'functional'}{'_test_files'}};
        $self->assert( $found, "file $file wasn't added to list of files to remove" );
    }
}

sub test_config_spec {
    my $self = $_[0];

    my $spec = $self->{'functional'}->config_spec();
    $self->assert_isa( 'Lire::Config::ConfigSpec', $spec );
}

sub test_setup_sendmail_path {
    my $self = $_[0];

    my $config_file = $self->{'functional'}->homedir() . "/.lire/config/func_test_sendmail.xml";
    $self->assert_not_null( $self->{'functional'}{'_config_files'}{'func_test_sendmail'},
                            "no config file 'func_test_sendmail' was created"
                          );

    my $conf = $self->{'functional'}{'_config_files'}{'func_test_sendmail'};
    my $e_sendmail = $self->{'functional'}->rundir() . "/func_test_sendmail";
    my $sendmail_path = $conf->global->get( 'sendmail_path' )->as_value();
    $self->assert_not_null( $sendmail_path, "sendmail_path wasn't set" );
    $self->assert_equals( $e_sendmail, $sendmail_path );
    $self->assert( -x $sendmail_path, "$sendmail_path isn't executable" );

    # Make sure only one var was set
    foreach my $var ( $conf->global()->spec()->components() ) {
        next if $var->name eq 'sendmail_path';
        $self->assert( ! $conf->global()->is_set( $var->name() ),
                       $var->name() . " was set in $config_file" );
    }

    my $maildir = $self->{'functional'}->rundir() . "/.func_test_mail";
    $self->assert( -d $maildir, "directory to store sent mail doesn't exists: $maildir" );
}

sub test_sent_mail {
    my $self = $_[0];

    my $content = <<EOM;
From: me
To: you
Subject: Test mail

Test
EOM
    my $sendmail_path = $self->{'functional'}->rundir() . "/func_test_sendmail";
    open my $fh, "| $sendmail_path test1 test.2 test2"
      or $self->error( "open $sendmail_path failed: $!" );
    print $fh $content;
    close $fh
      or $self->error( "close failed: $!" );

    open $fh, "| $sendmail_path recipient1 recipient2"
      or $self->error( "open $sendmail_path failed: $!" );
    print $fh $content;
    close $fh
      or $self->error( "close failed: $!" );

    my $msg = $self->{'functional'}->sent_mail();
    $self->assert_equals( 2, scalar @$msg );
    for my $idx ( 0..1 ) {
        $self->assert_not_null( $msg->[$idx], "Message $idx structure is undef");
        $self->assert_not_null( $msg->[$idx]{'recipients'},
                                "Message $idx 'recipients' is undef");
        $self->assert_not_null( $msg->[$idx]{'message'},
                                "Message $idx 'message' is undef");
        $self->assert_equals( 2, scalar (keys %{$msg->[$idx]}) );
    }
    $self->assert_deep_equals( [ qw/test1 test.2 test2/ ],
                               $msg->[0]{'recipients'} );
    $self->assert_deep_equals( [ qw/recipient1 recipient2/ ],
                               $msg->[1]{'recipients'} );
    $self->assert_equals( $content, $msg->[0]{'message'} );
    $self->assert_equals( $content, $msg->[1]{'message'} );

    $self->assert_deep_equals( $msg, $self->{'functional'}->sent_mail() );

    my $result = $self->{'functional'}->lire_run( "true" );
    $self->assert_equals( 0, scalar @{$result->sent_mail() } );
    $self->assert_deep_equals( $result->sent_mail(),
                               $self->{'functional'}->sent_mail() );
}

sub test_create_cfg_file {
    my $self = $_[0];

    $self->assert_dies( qr/missing 'name' parameter/,
                        sub { $self->{'functional'}->create_test_cfg_file() } );

    $self->assert_dies( qr/'name' parameter should only contains digits, letters/,
                        sub { $self->{'functional'}->create_test_cfg_file( "../test" ) } );

    my $cfg = $self->{'functional'}->create_test_cfg_file( "test" );
    $self->assert_isa( 'Lire::Config::ConfigFile', $cfg );

    my $cfg_file = $self->{'functional'}->homedir() . "/.lire/config/test.xml";
    $self->assert_equals( $cfg_file, $cfg->filename() );
    $self->assert( ! -f $cfg_file, "$cfg_file shouldn't exist" );
    $self->assert_not_null( $cfg->global(), "config's global section wasn't initialzied" );
    $self->assert_equals($cfg->global()->spec(), $self->{'functional'}->config_spec() );

    $self->assert_equals( $cfg, $self->{'functional'}->create_test_cfg_file( "test" ));

    my $result = $self->{'functional'}->lire_run( "echo Hi" );
    $self->assert( -f $cfg_file, "$cfg_file wasn't created" );
}

1;
