package Lire::ReportParser::RowColHandler;

use strict;

use base qw/ Lire::ReportParser /;

=pod

=head1 NAME

Lire::ReportParser::RowColHandler - Lire::ReportParser subclass
which synthetize row

=head1 SYNOPSIS

In XML Report processors:

    package MyParser;

    use base qw/Lire::ReportParser::RowColHandler/;

    sub handle_header_row {
        my ( $self, $row ) = @_;

        ...
    }

    sub handle_row {
        my ( $self, $row ) = @_;

        ...
    }

    sub handle_table_summary {
        my ( $self, $nrecords, $row ) = @_;

        ....
    }

=head1 DESCRIPTION

The Lire::ReportParser::RowColHandler module is a Lire::ReportParser
subclass which will synthetize handle_row() events. This makes it
easier to write subclases which onlypurpose is to format the report.
Instead of having to reconstruct the table row from the various
entry_start, group_start, handle_name, events, the subclass only has
to process handle_row() events.

=head1 USING Lire::ReportParser::RowColHandler

Client only have to inherit from Lire::ReportParser::RowColHandler.
After that, they can define handle_row(), handle_header_row() and
handle_table_summary() method.

There is a parameter that the RowColHandler accepts and its
C<summary_when> which specify when the handle_table_summary() event
will be synthetised ( C<before> or C<after> the table's body.) When
it's C<before> the event will be generated after the header_row events
but before any handle_row() events. Otherwise, it will be generated
once all handle_row() events are processed. The default is C<before>.

=cut

sub new {
    my ( $class, %params ) = @_;

    my $self = $class->SUPER::new( %params );

    $self->{'lrc_summary_when'} =
      $params{'summary_when'} eq 'before' ? 'before' : 'after';

    return $self;
}

sub subreport_start {
    my ( $self, $name, $attr ) = @_;

    $self->SUPER::subreport_start( $name, $attr );
    $self->{'lrc_row_idx_stack'} = [];

    return;
}

sub subreport_end {
    my ( $self, $name ) = @_;

    $self->SUPER::subreport_end( $name );
    die "subreport_end(): assertion failed: row_idx_stack should be empty"
      if @{$self->{'lrc_row_idx_stack'}};

    delete $self->{'lrc_row_idx_stack'};
    delete $self->{'lrc_table_summary'};
    delete $self->{'lrc_in_table_summary'};

    return;
}

sub table_end {
    my ( $self, $name ) = @_;

    $self->handle_table_summary( $self->{'lrc_table_summary'}{'nrecords'},
                                 $self->{'lrc_table_summary'}{'row'} )
      if $self->{'lrc_summary_when'} eq 'after';

    $self->SUPER::table_end( $name );

    return;
}

sub table_info_end {
    my ( $self, $name ) = @_;

    $self->SUPER::table_info_end( $name );

    foreach my $row ( @{ $self->current_table_info()->header_rows() } ) {
        $self->handle_header_row( $row );
    }

    return;
}

sub group_summary_start {
    my ( $self, $name, $attr ) = @_;

    $self->SUPER::group_summary_start( $name, $attr );

    if ( defined $attr->{'row-idx'} ) {
        push @{$self->{'lrc_row_idx_stack'}}, $attr->{'row-idx'};
    } elsif ( ! $self->within_element( "lire:group" ) ) {
        $self->{'lrc_in_table_summary'} = 1;
        $self->{'lrc_table_summary'} = { 'nrecords' => $attr->{'nrecords'},
                                         'row'      => [],
                                       };
    }
    return;
}

sub group_summary_end {
    my ( $self, $name ) = @_;

    $self->SUPER::group_summary_end( $name );

    if ( $self->{'lrc_in_table_summary'} ) {
        $self->{'lrc_in_table_summary'} = 0;
        $self->handle_table_summary( $self->{'lrc_table_summary'}{'nrecords'},
                                     $self->{'lrc_table_summary'}{'row'} )
          if $self->{'lrc_summary_when'} eq 'before';
    } else {
        pop @{$self->{'lrc_row_idx_stack'}};
    }
}

sub entry_start {
    my ( $self, $name, $attr ) = @_;

    $self->SUPER::entry_start( $name, $attr );
    push @{$self->{'lrc_row_idx_stack'}},  $attr->{'row-idx'};

    $self->{'lrc_curr_entry'} = {}
      unless ( $self->within_element( 'lire:group' ) );
}

sub entry_end {
    my ( $self, $name) = @_;

    $self->SUPER::entry_end( $name );
    my $row_idx = pop @{$self->{'lrc_row_idx_stack'}};

    # Skip extra entry
    return unless defined $row_idx;

    # We generate row events only when the entry of the first level
    # is closed. We are sure that all the row spanned by that entry
    # are complete
    return if ( $self->within_element( 'lire:group' ) );

    foreach my $idx ( sort { $a <=> $b } keys %{$self->{'lrc_curr_entry'}} ) {
        $self->handle_row( $self->{'lrc_curr_entry'}->{$idx} );
    }

    return;
}

sub handle_name {
    my ( $self, $name ) = @_;

    # When row_idx is undefined, that cell is part of an
    # entry which shouldn't be displayed.
    my $row_idx = $self->curr_row_idx();
    return unless defined $row_idx;

    my $col = $name->{'col_info'}->col_start();
    $self->{'lrc_curr_entry'}{$row_idx}[$col] = $name;

    return;
}

sub handle_value {
    my ( $self, $value ) = @_;

    # When row_idx is undefined, that cell is part of an
    # entry which shouldn't be displayed.
    my $row_idx = $self->curr_row_idx();
    return unless defined $row_idx;

    my $col = $value->{'col_info'}->col_start();
    $self->{'lrc_curr_entry'}{$row_idx}[$col] = $value;

    return;
}

sub handle_summary_value {
    my ( $self, $value ) = @_;

    # Mark it as a summary value
    $value->{'is_summary'} = 1;
    my $col = $value->{'col_info'}->col_start();
    if ( $self->{'lrc_in_table_summary'} ) {
        $self->{'lrc_table_summary'}{'row'}[$col] = $value;
    } else {
        # When row_idx is undefined, that cell is part of an
        # entry which shouldn't be displayed.
        my $row_idx = $self->curr_row_idx();
        return unless defined $row_idx;
        $self->{'lrc_curr_entry'}{$row_idx}[$col] = $value;
    }

    return;
}

sub curr_row_idx {
    $_[0]{'lrc_row_idx_stack'}[$#{$_[0]{'lrc_row_idx_stack'}}];
}

=pod

=head2 handle_table_summary( $nrecords, $row )

Called after the table summary is available (and depending on the
C<summary_when> initialisation parameter). $nrecords contains the
number of records used to compute the report and $row is an array
reference to the table summary value. The $row array contains as much
elements as there is columns in the table. Only the numerical column
will have a value in there, others will be set to undef.

The summary value are hash reference containing the same values than
is available in the Lire::ReportParser's handle_summary_value() event.

=cut

sub handle_table_summary {
    my ( $self, $nrecords, $row ) = @_;
}

=pod

=head2 handle_header_row( $row )

This will called once for each header row there is. The first header
row contains all the numerical columns as well as the main categorical
column. 

$row is an array reference containing as much element as there is
columns in the table. Column's labels which shouldn't appear on this
row are undef. Other elements will contain the related
Lire::Report::ColumnInfo object.

=cut

sub handle_header_row {
    my ( $self, $row ) = @_;
}

=pod

=head2 handle_row( $row )

This event will be called for each row to display in the table. 

$row is an array reference containing the data that should appear in
the row. It contains as many elements as there are columns defined in
the table. Empty column will have undef as content. Other elements
will be an hash reference identical to what would be received in the
handle_name() or handle_value() method of Lire::ReportParser. The only
difference is that summary value will have a key C<is_summary> set to
1.

In the case of spanning columns, the data element will be in the
col_start() element. The other cells will be undef.

=cut

sub handle_row {
    my ( $self, $row ) = @_;
}

1;

__END__


=head1 SEE ALSO

Lire::ReportParser(3pm) Lire::Report::ColumnInfo(3pm)

=head1 VERSION

$Id: RowColHandler.pm,v 1.12 2004/08/08 21:55:55 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
