package Lire::Config::PluginSpec;

use strict;

=pod

=head1 NAME

Lire::Config::PluginSpec - SelectSpec with variable properties.

=head1 DESCRIPTION

Lire::Config::TypeSpec that represents a configuration variable which
defines additional properties based on the value which is currently
selected. This can be used for extensible type like DlfConverter or
OutputFormat.

=cut

use base qw/Lire::Config::SelectSpec/;

use Lire::Config::Plugin;
use Lire::Config::RecordSpec;
use Carp;

=pod

=head2 new()

Create a Lire::Config::SectionSpec object.

=cut

sub new {
    my $self = shift->SUPER::new( @_ );

    $self->{'_no_properties_spec'} =
      new Lire::Config::RecordSpec( 'name' => $self->name() . "_no_properties" );

    return $self;
}

sub instance_class {
    return 'Lire::Config::Plugin';
}

=pod

=head2 get_properties_spec( $option )

Returns the TypeSpec that specifies the properties of the $option.
This method will return an empty RecordSpec when the plugin didn't
define any properties. One can also use the has_properties() method to
check if the Plugin defined configuration properties.

=cut

sub get_properties_spec {
    my ( $self, $option ) = @_;

    return $self->{'_no_properties_spec'}
      unless defined $option;

    my $name = $self->normalize( $option );
    croak "no option '$option' in this specification"
      unless defined $name;

    $name .=  '_properties';
    my $spec = Lire::Config->config_spec();
    return $spec->get( $name ) if $spec->has_component( $name );

    return $self->{'_no_properties_spec'};
}

1;

__END__

=pod

=head1 VERSION

$Id: PluginSpec.pm,v 1.5 2004/06/23 20:28:19 flacoste Exp $

=head1 AUTHORS

  Francis J. Lacoste <flacoste@logreport.org>
  Wolfgang Sourdeau <wolfgang@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2002-2004  Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
