package Lire::Config::DlfStreamSpec;

use strict;

use base qw/Lire::Config::CompoundSpec/;

use Lire::PluginManager;
use Lire::Config::IntegerSpec;
use Lire::Config::DlfAnalyserSpec;
use Locale::TextDomain 'lire';

=pod

=head1 NAME

Lire::Config::DlfStreamSpec - Defines DlfStream configuration.

=head1 DESCRIPTION

This Lire::Config::TypeSpec is a CompoundSpec which hold the
configuration related to a DlfStream..

=head2 new( 'name' => $dst_stream_name )

The name of the type specification is the name of the DlfStream which
is configured.

A DlfStreamSpec will define an integer component named 'keep_days'
which hold the number of days of Dlf to keep. One component for each
available extension schema is also defined.

=cut

sub new {
    my $self = shift->SUPER::new( @_ );

    my $int =
      new Lire::Config::IntegerSpec( 'name' => 'keep_days',
                                     'summary' => N__( 'Days of DLF to keep' ),
                                     'description' =>
                                     '<para>' . join( "", N__( 'This parameter states how many days of DLF records you want to keep in the DlfStore. Records older than this number of days will be deleted. Use 0 to never expires records.' ) ) . '</para>' );
    $int->default( $int->instance( 'value' => 60 ) );
    $self->add( $int );
    foreach my $a ( @{ $self->_extension_schemas() } ) {
        $self->add( new Lire::Config::DlfAnalyserSpec( 'name' => $a,
                                                       'src_schema' => $self->name() ) );
    }
    return $self;
}

sub _extension_schemas {
    my $self = $_[0];

    my %schemas = ();
    foreach my $a ( @{Lire::PluginManager->analysers_by_src( $self->name() )} )
    {
        $schemas{$a->dst_schema()} = 1;
    }
    return [ sort keys %schemas ];
}

sub summary {
    return __x( 'Configuration for the {schema} DlfStream',
                'schema' => $_[0]->name() );
}

sub description {
    return '<para>' . __x( 'This configures the analysers that should be run on this DlfStream as well as the number of days of data to keep.',
                           'schema' => $_[0]->name() ) . '</para>';
}

sub instance_class {
    return 'Lire::Config::Dictionary';
}

1;

__END__

=pod

=head2 SEE ALSO

  Lire::Config::DlfStreamsSpec(3pm), Lire::Config::DlfAnalyserSpec(3pm)

=head1 VERSION

$Id: DlfStreamSpec.pm,v 1.3 2004/06/04 21:36:05 flacoste Exp $

=head1 AUTHORS

  Francis J. Lacoste <flacoste@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2004  Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
