package Lire::Config::BasePluginSpec;

use strict;

=pod

=head1 NAME

Lire::Config::BasePluginSpec - PluginManager based PluginSpec

=head1 DESCRIPTION

This is Lire::Config::PluginSpec that determines its valid list of
options from the Lire::PluginManager.

=cut

use base qw/Lire::Config::PluginSpec/;

use Lire::Config::OptionSpec;
use Lire::PluginManager;
use Carp;

=pod

=head2 new()

Create a Lire::Config::BasePluginSpec object.

=head2 plugin_type()

Subclasses should defines a plugin_type() method that would be use
to select the available options.

=cut

sub plugin_type {
    croak ref shift, '::plugin_type() unimplemented';
}

sub is_valid {
    my ( $self, $val ) = @_;

    return !$self->required() unless defined $val;

    return Lire::PluginManager->has_plugin( $self->plugin_type(), $val );
}

sub normalize {
    my ( $self, $val ) = @_;

    return $self->is_valid( $val ) ? $val : undef;
}

sub add {
    croak( ref shift, ' cannot contain any components' );
}

sub options {
    my $self = $_[0];

    my @options;

    my $mgr = Lire::PluginManager->instance();
    foreach my $name ( @{$mgr->plugin_names( $self->plugin_type() ) } ) {
        my $plugin = $mgr->get_plugin( $self->plugin_type(), $name );
        push @options,
          new Lire::Config::OptionSpec( 'name' => $name,
                                        'summary' => $plugin->title(),
                                        'description' => $plugin->description()
                                      );
    }
    return sort { $a->name() cmp $b->name() } @options;
}

1;

__END__

=pod

=head1 SEE ALSO

 Lire::Config::DlfConverterSpec(3pm) Lire::Config::

=head1 VERSION

$Id: BasePluginSpec.pm,v 1.2 2004/07/21 21:28:30 flacoste Exp $

=head1 AUTHORS

  Francis J. Lacoste <flacoste@logreport.org>
  Wessel Dankers <wsl@logreport.org>
  Wolfgang Sourdeau <wolfgang@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2002-2004  Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
