// vim: nowrap

/* ***********

Module: Isapnpconf
Created: 15/05/2000
Last Modified: 19/06/2000
Author: Cristiano Otto Von Trompczynski
e-mail: cris@conectiva.com.br
Description: Functions to manipulate the 'isapnp' structure. This one contains all the 'isapnp.conf' file information
(extracted by the parser) organized by CARD, CONFIGURE, SETTING, IRQ, DMA and IO structures.
                                                                                                      "This module is for you, Penguin!"
   ***********
*/

#include "isapnpconf.h"
#include <stdlib.h>

//Set of functions to extract the number of DMAs, IRQs, IOs, SETTINGs, CONFIGUREs and CARDs from the structure
PUBLIC int MODULE_isapnpconf::get_n_dma (struct settings *s){
	return s->n_dma;
}

PUBLIC int MODULE_isapnpconf::get_n_irq (struct settings *s){
	return s->n_irq;
}

PUBLIC int MODULE_isapnpconf::get_n_io (struct settings *s){
	return s->n_io;
}

PUBLIC int MODULE_isapnpconf::get_n_settings (struct configure *c){
	return	c->n_settings;
}

PUBLIC int MODULE_isapnpconf::get_n_configure (struct pcard *c){
	return 	c->n_cfg;
}

PUBLIC int MODULE_isapnpconf::get_n_card (){
	if (cf_start)
		return cf_start->n_card;
	else 	return ERROR;
}

/*
struct s_dma* MODULE_isapnpconf::get_dma (struct settings *s, int n)
  Returns a pointer to the DMA structure. Receives as a parameter the address of a SETTING and the desired DMA number.
RETURN: NULL or a pointer to the struct
*/
PUBLIC struct s_dma* MODULE_isapnpconf::get_dma (struct settings *s, int n){
	if (s->dma){
		struct s_dma *d=s->dma;
		for (int i=0; i<n; i++, d=d->next_dma){
			if (d->next_dma==NULL)
				return	NULL;
		}
		return d;
	}else 	return NULL;
}

/*
struct s_irq* MODULE_isapnpconf::get_irq (struct settings *s, int n)
  Returns a pointer to the IRQ structure. Receives as a parameter the address of a SETTING and the desired IRQ number.
RETURN: NULL or a pointer to the struct
*/
PUBLIC struct s_irq* MODULE_isapnpconf::get_irq (struct settings *s, int n){
	if (s->irq){
		struct s_irq *irq=s->irq;
		for (int i=0; i<n; i++, irq=irq->next_irq){
			if (irq->next_irq==NULL)
				return	NULL;
		}
		return irq;
	}else 	return NULL;
}

/*
PUBLIC struct s_io* MODULE_isapnpconf::get_io (struct settings *s, int n){
  Returns a pointer to the IO structure. Receives as a parameter the address of a SETTING and the desired IO number.
RETURN: NULL or a pointer to the struct
*/
PUBLIC struct s_io* MODULE_isapnpconf::get_io (struct settings *s, int n){
	if (s->io){
		struct s_io *io=s->io;
		for (int i=0; i<n; i++, io=io->next_io){
			if (io->next_io==NULL)
				return	NULL;
		}
		return io;
	}else 	return NULL;
}

/*
struct settings* MODULE_isapnpconf::get_setting (struct configure *c, int n)
  Returns a pointer to the SETTING structure. Receives as a parameter the address of a CONFIGURE and the number of the desired SETTING.
RETURN: NULL or a pointer to the struct
*/
PUBLIC struct settings* MODULE_isapnpconf::get_setting (struct configure *c, int n){
	if (c->setting){
		struct settings *s=c->setting;
		for (int i=0; i<n; i++, s=s->next_setting){
			if (s->next_setting==NULL)
				return	NULL;
		}
		return s;
	}else 	return NULL;
}

/*
struct configure* MODULE_isapnpconf::get_configure (struct pcard *c, int n)
  Returns a pointer to the CONFIGURE structure. Receives as a parameter the address of a CARD and the number of the CONFIGURE desired.
RETURN: NULL or a pointer to the struct
*/
PUBLIC struct configure* MODULE_isapnpconf::get_configure (struct pcard *c, int n){
	if (c->cfg){
		struct configure *cf=c->cfg;
		for (int i=0; i<n; i++, cf=cf->next_cfg){
			if (cf->next_cfg==NULL)
				return	NULL;
		}
		return cf;
	}else 	return NULL;
}

/*
struct pcard* MODULE_isapnpconf::get_card (int n)
  Returns a pointer to the CARD structure. Receives as parameter the desired CARD number.
RETURN: NULL or a pointer to the struct
*/
PUBLIC struct pcard* MODULE_isapnpconf::get_card (int n){
	if (cf_start->card){
		struct pcard *s=cf_start->card;
		for (int i=0; i<n; i++,s=s->next_card){
			if (s->next_card==NULL)
				return 	NULL;
		}
		return s;
	}else 	return 	NULL;
}

/*
void MODULE_isapnpconf::clean_struct ()
  This function disallocates all the structure pointed by cf_start
RETURN: Nothing
*/
PUBLIC void MODULE_isapnpconf::clean_struct (){
	if (cf_start){
		struct pcard *s=NULL;
		struct configure *p=NULL;
		struct settings *d=NULL;
		struct s_io *o=NULL;
		struct s_dma *m=NULL;
		struct s_irq *i=NULL;
		void *pclean=NULL;
		
		s=cf_start->card;
		while (s!=NULL){
			p=(struct configure*)s->cfg;
			while (p!=NULL){
				d=(struct settings*)p->setting;
				while (d!=NULL){
					o=(struct s_io*)d->io;
					while (o!=NULL){
						free ((char *)o->io);
						pclean=(void*)o;
						o=o->next_io;
						free ((struct s_io*)pclean);
					}
					i=(struct s_irq*)d->irq;
					while (i!=NULL){
						free ((char *)i->irq);
						pclean=(void*)i;
						i=i->next_irq;
						free ((struct s_irq*)pclean);
					}
					m=(struct s_dma*)d->dma;
					while (m!=NULL){
						free ((char *)m->dma);
						pclean=(void*)m;
						m=m->next_dma;
						free ((struct s_dma*)pclean);
					}
					pclean=(void*)d;
					d=d->next_setting;
					free ((struct settings*)pclean);
				}
				free (p->description);
				pclean=(void*)p;
				p=(struct configure*)p->next_cfg;
				free ((struct configure*)pclean);
			}
			free (s->vendor_Id);
			free (s->serial);
			free (s->description);
			pclean=(void*)s;
			s=(struct pcard*)s->next_card;
			free ((struct pcard*)pclean);
		}
		free (cf_start->id);
		free (cf_start->version);
		free (cf_start->date);
		free(cf_start);
		cf_start=NULL;
	}
}

/*
void MODULE_isapnpconf::print_struct (void)
  Prints the structure on the screen. Prints only the CARD, CONFIGURE, SETTING and the IOs, DMAs and IRQs numbers of this SETTING.
  Only for data checking
RETURN: Nothing.
*/
PUBLIC void MODULE_isapnpconf::print_struct (void){
	int nCard=get_n_card();			
	xconf_notice ("Card number: %d",nCard);
	for (int i=0; i<nCard; i++){
		struct pcard *c= get_card (i);
		int nCfg=get_n_configure (c);
		xconf_notice ("CONFIGURE Card[%d] number: %d",i,nCfg);
		for (int j=0; j<nCfg; j++){
			struct configure *cfg= get_configure (c,j);
			int nSet=get_n_settings (cfg);
			xconf_notice ("SETTINGs number of CONFIGURE %d of the Card %d: %d",j,i,nSet);
			for (int k=0; k<nSet; k++){
				struct settings *set= get_setting (cfg,k);
				int nIrq=get_n_irq (set);
				int nIo=get_n_io (set);
				int nDma=get_n_dma (set);
				xconf_notice ("IRQ number: %d\nIO number: %d\nDMA number: %d\n of SETTING %d of CONFIGURE %d of the Card %d",nIrq,nIo,nDma,k,j,i);
			}
		}
	}
}

/*
void MODULE_isapnpconf::print_all_struct (void)
  Print all the data structure. Only for checking
RETURN: Nothing.
*/
PUBLIC void MODULE_isapnpconf::print_all_struct (void){
	if (cf_start){
		struct pcard *s=NULL;
		struct configure *p=NULL;
		struct settings *d=NULL;
		struct s_io *o=NULL;
		struct s_dma *m=NULL;
		struct s_irq *i=NULL;
		
		xconf_notice ("cf_isapnp\nid: %s\nversion: %s\ndate: %s\nn_card: %d\nstruct pcard *card: '%d'\n",cf_start->id,cf_start->version,cf_start->date,cf_start->n_card,cf_start->card);
		s=cf_start->card;
		int z=0;
		while (s!=NULL){
			xconf_notice ("CARD %d\nline_card: %d\nvendor_Id: %s\nserial: %s\ndescription: %s\nn_cfg: %d\nstruct configure *cfg: '%d'\nstruct pcard *next_card: '%d'",++z,s->line_card,s->vendor_Id,s->serial,s->description,s->n_cfg,s->cfg,s->next_card);
			
			p=(struct configure*)s->cfg;
			int x=0;
			while (p!=NULL){
				xconf_notice ("CONFIGURE %d\nsuccess: %d\nline_cfg_start: %d\nline_cfg_end: %d\ndescription: %s\nn_settings: %d\nld: %d\nstruct settings *setting: '%d'\nstruct configure *next_cfg: '%d'\n",++x,p->success,p->line_cfg_start,p->line_cfg_end,p->description,p->n_settings,p->ld,p->setting,p->next_cfg);
				d=(struct settings*)p->setting;
				int c=0;
				while (d!=NULL){
					xconf_notice ("SETTINGS %d\nline_setting: %d\ninUse: %d\n inUseCg: %d\nn_io: %d\nn_irq: %d\nn_dma: %d\nstruct s_io *io: '%d'\nstruct s_irq *irq: '%d'\nstruct s_dma *dma: '%d'\nstruct settings *next_setting: '%d'",c++,d->line_setting,d->inUse,d->inUseCg,d->n_io,d->n_irq,d->n_dma,d->io,d->irq,d->dma,d->next_setting);
					o=(struct s_io*)d->io;
					int no=0;
					while (o!=NULL){
						xconf_notice ("IO %d\nline_io: %d\nio: %s\nstruct s_io *next_io: '%d'\n",no++,o->line_io,o->io,o->next_io);
						o=o->next_io;
					}
					no=0;
					i=(struct s_irq*)d->irq;
					while (i!=NULL){
						xconf_notice ("IRQ %d\nline_irq: %d\nirq: %s\nstruct s_irq *next_irq: '%d'\n",no++,i->line_irq,i->irq,i->next_irq);
						i=i->next_irq;
					}
					no=0;
					m=(struct s_dma*)d->dma;
					while (m!=NULL){
						xconf_notice ("DMA %d\nline_dma: %d\ndma: %s\nstruct s_dma *next_dma: '%d'\n",no++,m->line_dma,m->dma,m->next_dma);
						m=m->next_dma;
					}
					d=d->next_setting;	
				}
				p=(struct configure*)p->next_cfg;
			}
			s=(struct pcard*)s->next_card;	
		}
		
	}
}

// Methods to work with the Structure
// Add a Card
PUBLIC struct pcard* MODULE_isapnpconf::add_card (){
	if (cf_start){
		struct pcard *c=cf_start->card;
		if (c){
			while (c->next_card){
				c=c->next_card;
			}
			if( (c->next_card=(struct pcard*)malloc(sizeof(struct pcard))) )
				c=c->next_card;
			else	return NULL;
			
		}else{
			if( (cf_start->card=(struct pcard*)malloc(sizeof(struct pcard))) )
				c=cf_start->card;
			else	return NULL;
		}
		cf_start->n_card++;
		c->line_card=-1;
		c->vendor_Id=NULL;
		c->serial=NULL;
		c->description=NULL;
		c->n_cfg=0;
		c->cfg=NULL;
		c->next_card=NULL;
		return c;
	}
	return NULL;
}

// Add a Configuration
PUBLIC struct configure* MODULE_isapnpconf::add_configure (){
	if (cf_start && cf_start->card){ // Have a cf_start or card
		struct pcard *c=cf_start->card;
		struct configure *conf=NULL;
		// Find last card
		while (c->next_card){ // Find last card
			c=c->next_card;
		}
		if (c->cfg){ // Find last configuration
			conf=c->cfg;
			while (conf->next_cfg){
				conf=conf->next_cfg;
			}
			if ( (conf->next_cfg=(struct configure*)malloc(sizeof(struct configure))) )
				conf=conf->next_cfg;
			else 	return NULL;
		}else{
			if ( (c->cfg=(struct configure*)malloc(sizeof(struct configure))) )
				conf=c->cfg;
			else 	return NULL;
		}
		// Set the last configuration
		c->n_cfg++;
		conf->success=0;
		conf->line_cfg_start=-1;
		conf->line_cfg_end=-1;
		conf->description=NULL;
		conf->n_settings=0;
		conf->ld=-1;
		conf->setting=NULL;
		conf->next_cfg=NULL;
		return conf;
	}
	return NULL; // We don't have a cf_start or a card
}

// Add a setting
PUBLIC struct settings* MODULE_isapnpconf::add_setting (){
	if (cf_start && cf_start->card){ // Have a cf_start, card and a configure?
		struct pcard *c=cf_start->card;
		struct configure *conf=NULL;
		struct settings *set=NULL; 
		while (c->next_card){ // Find last card
			c=c->next_card;
		}
		if (c->cfg){ // Last card have a configuration?
			conf=c->cfg;
			while (conf->next_cfg){ // Find last configuration
				conf=conf->next_cfg;
			}
			if (conf->setting){ // Last configuration have a setting?
				set=conf->setting;
				while (set->next_setting){ // Find last setting
					set=set->next_setting;
				}
				if ( (set->next_setting=(struct settings*)malloc(sizeof(struct settings)))  ) // Set last setting
					set=set->next_setting;
				else 	return NULL;
			}else{
				if( (conf->setting=(struct settings*)malloc(sizeof(struct settings))) ) // Set last setting
					set=conf->setting;
				else	return NULL;
			}
			conf->n_settings++;
			set->inUse=0;
			set->inUseCg=0;
			set->line_setting=-1;
			set->n_io=0;
			set->n_irq=0;
			set->n_dma=0;
			set->io=NULL;
			set->irq=NULL;
			set->dma=NULL;
			set->next_setting=NULL;
			return set;
		}
		else 	return NULL;	
	}
	return NULL; // We don't have a cf_start, card and a configure
}

// Add an io
PUBLIC struct s_io* MODULE_isapnpconf::add_io (){
	if (cf_start && cf_start->card){ // Have a cf_start, card, configure and a setting?
		struct pcard *c=cf_start->card;
		struct configure *conf=NULL;
		struct settings *set=NULL;
		struct s_io *io=NULL;
		while (c->next_card){ // Find last card
			c=c->next_card;
		}
		if (c->cfg){ // Last card have a configuration?
			conf=c->cfg;
			while (conf->next_cfg){ // Find last configuration
				conf=conf->next_cfg;
			}
			if (conf->setting){ // Last configuration have a setting?
				set=conf->setting;
				while (set->next_setting){ // Find last setting
					set=set->next_setting;
				}
				if (set->io){ // Last setting have an io?
					io=set->io;
					while (io->next_io){ // Find last io
						io=io->next_io;
					}
					if ( (io->next_io=(struct s_io*)malloc(sizeof(struct s_io)))  ) // Set last io
						io=io->next_io;
					else    return NULL;
				}else{
					if ( (set->io=(struct s_io*)malloc(sizeof(struct s_io))) )
						io=set->io;
					else 	return NULL;
				}
				set->n_io++;
				io->line_io=-1;
				io->io=NULL;
				io->next_io=NULL;
				return io;
			}else	return NULL;
											
		}else 	return NULL;		
	}
	return NULL; // We dont't have a cf_start, card, configure and a setting
}

// Add an irq
PUBLIC struct s_irq* MODULE_isapnpconf::add_irq (){
	if (cf_start && cf_start->card){ // Have a cf_start, card, configure and a setting?
		struct pcard *c=cf_start->card;
		struct configure *conf=NULL;
		struct settings *set=NULL;
		struct s_irq *irq=NULL;
		while (c->next_card){ // Find last card
			c=c->next_card;
		}
		if (c->cfg){ // Last card have a configuration
			conf=c->cfg; // Find last configuration
			while (conf->next_cfg){
				conf=conf->next_cfg;
			}
			if (conf->setting){ // Last configuration have a setting?
				set=conf->setting;
				while (set->next_setting){ // Find last setting
					set=set->next_setting;
				}
				if (set->irq){ // Last setting have an irq?
					irq=set->irq;
					while (irq->next_irq){ // Find last irq
						irq=irq->next_irq;
					}
					if ( (irq->next_irq=(struct s_irq*)malloc(sizeof(struct s_irq)))  ) // Set last irq
						irq=irq->next_irq;
					else    return NULL;
				}else{
					if ( (set->irq=(struct s_irq*)malloc(sizeof(struct s_irq))) )
						irq=set->irq;
					else 	return NULL;
				}
				set->n_irq++;
				irq->line_irq=-1;
				irq->irq=NULL;
				irq->next_irq=NULL;
				return irq;
			}else	return NULL;
											
		}else 	return NULL;		
	}
	return NULL; // We dont't have a cf_start, card, configure and a setting
}

// Add a dma
PUBLIC struct s_dma* MODULE_isapnpconf::add_dma (){
	if (cf_start && cf_start->card){ // Have a cf_start, card, configure and a setting?
		struct pcard *c=cf_start->card;
		struct configure *conf=NULL;
		struct settings *set=NULL;
		struct s_dma *dma=NULL;
		while (c->next_card){ // Find last card
			c=c->next_card;
		}
		if (c->cfg){ // Last card have a configuration?
			conf=c->cfg;
			while (conf->next_cfg){ // Find last configuration
				conf=conf->next_cfg;
			}
			if (conf->setting){ // Last configuration have a setting?
				set=conf->setting;
				while (set->next_setting){ // Find last setting
					set=set->next_setting;
				}
				if (set->dma){ // Last setting have an dma?
					dma=set->dma;
					while (dma->next_dma){ // Find last dma
						dma=dma->next_dma;
					}
					if ( (dma->next_dma=(struct s_dma*)malloc(sizeof(struct s_dma)))  ) // Set last dma
						dma=dma->next_dma;
					else    return NULL;
				}else{
					if ( (set->dma=(struct s_dma*)malloc(sizeof(struct s_dma))) )
						dma=set->dma;
					else 	return NULL;
				}
				set->n_dma++;
				dma->line_dma=-1;
				dma->dma=NULL;
				dma->next_dma=NULL;
				return dma;
			}else	return NULL;
											
		}else 	return NULL;		
	}
	return NULL; // We dont't have a cf_start, card, configure and a setting
}
