#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <limits.h>
#include <sys/stat.h>
#include <subsys.h>
#include <translat.h>
#include "../paths.h"
#include "internal.h"
#include "userconf.h"
#include "userconf.m"
#include <quota.h>
#include <fstab.h>
#include <dialog.h>
#include <netconf.h>
#include "usercomng.h"

/* #Specification: userconf / etc/passwd
	/etc/passwd is the user database. Its permission flags are
	always set to 0644 when rewritten.
*/
#define ETC_PTMP	"/etc/ptmp"
static USERCONF_HELP_FILE help_users ("users");
static USERCONF_HELP_FILE help_filter ("filter");

static int (*user_fct_del)(USER *, SHADOW *)=NULL;
static int (*user_fct_add)(USER *, SHADOW *, bool)=NULL;
static int (*user_fct_mod)(USER *, SHADOW *)=NULL;


/*
	record some hooks for distribution dependant account management methods 
*/
void users_sethook (
	int (*_user_fct_del)(USER *, SHADOW *),
	int (*_user_fct_add)(USER *, SHADOW *, bool),
	int (*_user_fct_mod)(USER *, SHADOW *))
{
	user_fct_del = _user_fct_del;
	user_fct_add = _user_fct_add;
	user_fct_mod = _user_fct_mod;
}

static CONFIG_FILE f_passwd (ETC_PASSWD,help_users
	,CONFIGF_MANAGED|CONFIGF_TMPLOCK
	,"root","root",0644,subsys_useraccounts);


PRIVATE void USERS::readusers()
{
	/* #Specification: /etc/passwd / strategy
		/etc/passwd is read "by hand" instead of using getpwent() to avoid
		getting all those NIS entries. This is done when editing local
		user account.
	*/
	nisentry = NULL;
	nis_at_end = true;
	FILE_CFG *fin = configf->fopen ("r");
	if (fin != NULL){
		char line[1000];
		while (fgets(line,sizeof(line)-1,fin)!=NULL){
			strip_end (line);
			if (line[0] != '\0'){
				USER *usr = new USER(line);
				if (strcmp(usr->getname(),"+")==0){
					delete nisentry;
					nisentry = usr;
					if (getnb()==0) nis_at_end = false;
				}else{
					add (usr);
				}
			}
		}
		fclose (fin);
	}
	shadows = NULL;
	rstmodified();
}

PUBLIC USERS::USERS()
{
	home.setfrom (policies_getdefhome());
	domain.setfrom ("/");
	root.setfrom ("/");
	baseuid = 1;
	configf = &f_passwd;
	readusers();
	if (shadow_exist()) shadows = new SHADOWS;
	maxusers = 0;
}

PUBLIC USERS::USERS(
	CONFIG_FILE &_file,
	CONFIG_FILE &_shadow,
	const char *_root,
	const char *_home,
	const char *_domain,
	int _baseuid)
{
	root.setfrom (_root);
	home.setfrom (_home);
	domain.setfrom (_domain);
	baseuid = _baseuid;
	configf = &_file;
	readusers();
	if (_shadow.exist()) shadows = new SHADOWS(_shadow);
	maxusers = 0;
}

/*
	Set the maximum users in this domain.
	0 means no limit.
*/
PUBLIC void USERS::setmaxusers(int mx)
{
	maxusers = mx;
}

/*
	Forget the current users and reload from file
	This is used after an override function has been called since
	it probably have modified the configuration files
*/
PRIVATE void USERS::reload()
{
	SHADOWS *tmp = shadows;
	shadows = NULL;
	remove_all();
	readusers();
	shadows = tmp;
	if (shadows != NULL) shadows->reload();
}

/*
	Does this USERS list has shadow password support
*/
PUBLIC int USERS::has_shadow () const
{
	return shadows != NULL;
}
/*
	Return the recommend standard HOME base directory for users.
*/
PUBLIC const char *USERS::getstdhome() const
{
	return home.get();
}
/*
	Return the root of the user base (root of the domain)
*/
PUBLIC const char *USERS::getroot() const
{
	return root.get();
}
/*
	The the domain we are editing (/ for the main domain)
*/
PUBLIC const char *USERS::getdomain() const
{
	return domain.get();
}
/*
	Return the recommend standard HOME base directory for a member of a group
*/
PUBLIC const char *USERS::getstdhome(const char *group) const
{
	const char *ret = home.get();
	const char *grouphome = group_gethomebase(group);
	if (grouphome != NULL) ret = grouphome;
	return ret;
}


/*
	This object will contain a copy (only pointers).
	It is used as a temporary holder for the normal object, allowing doing
	some sort.
*/
PRIVATE USERS::USERS(USERS *users)
{
	configf = NULL;
	shadows = NULL;
	nisentry = NULL;
	nis_at_end = true;
	int n = users->getnb();
	neverdelete();
	for (int i=0; i<n; i++) add (users->getitem(i));
}


PUBLIC USERS::~USERS()
{
	delete shadows;
	delete nisentry;
}
/*
	Get one USER specification of the table or NULL
*/
PUBLIC USER *USERS::getitem(int no)
{
	return (USER*)ARRAY::getitem(no);
}
/*
	Get one USER specification of the table or NULL from his login name
*/
PUBLIC USER *USERS::getitem(const char *name, USER *exclude)
{
	USER *ret = NULL;
	int nbu = getnb();
	for (int i=0; i<nbu; i++){
 		USER *usr = getitem(i);
		if (usr != exclude && strcmp(usr->getname(),name)==0){
			ret = usr;
			break;
		}
	}
	return ret;
}

/*
	Get one USER specification of the table or NULL from his login name
*/
PUBLIC USER *USERS::getitem(const char *name)
{
	return getitem (name,NULL);
}
/*
	Get one SHADOW specification of the table or NULL from his login name
*/
PUBLIC SHADOW *USERS::getshadow(USER *usr)
{
	SHADOW *ret = NULL;
	if (shadows != NULL) ret = shadows->getitem(usr->getname());
	return ret;
}
PUBLIC void USERS::addshadow (SHADOW *shadow)
{
	shadows->add (shadow);
}
/*
	Get one USER specification of the table or NULL from his UID
*/
PUBLIC USER *USERS::getfromuid(int uid, USER *exclude)
{
	USER *ret = NULL;
	int nbu = getnb();
	for (int i=0; i<nbu; i++){
 		USER *usr = getitem(i);
		if (usr != exclude && usr->getuid() == uid){
			ret = usr;
			break;
		}
	}
	return ret;
}
/*
	Get one USER specification of the table or NULL from his UID
*/
PUBLIC USER *USERS::getfromuid(int uid)
{
	return getfromuid(uid,NULL);
}
/*
	Get one unused User ID.
*/
PUBLIC int USERS::getnewuid()
{
	/* #Specification: userconf / automatic allocaion of uid
		We multiply gid by 500. From there we search in all
		user id and allocate the first uid in the range.

		We don't allocate into holes (unused uid between used one)
		to avoid uid reuse (and a security hole).

		This has shown problematic. Now linuxconf allocated the a new
		UID as the next one to the one with the highest uid number.
	*/
	#if 0
	int base = gid * 500;
	int maxu = base + 500;
	#else
	int base = baseuid;
	int maxu = 65530;	// Some special UID exist at the end of the range
						// such as nobody which is often 65535
	#endif
	int ret = base;
	int nbu = getnb();
	for (int i=0; i<nbu; i++){
 		USER *usr = getitem(i);
		int uid = usr->getuid();
		if (uid >= base && uid < maxu){
			if (uid >= ret) ret = uid + 1;
		}
	}
	return ret;	
}
/*
	Write the /etc/passwd file with proper locking
*/
PUBLIC int USERS::write(PRIVILEGE *priv)
{
	int ret = -1;
	//sortbygid();
	if (configf != NULL){
		FILE_CFG *fout = configf->fopen (priv,"w");
		if (fout != NULL){
			int nbu = getnb();
			if (nisentry != NULL && !nis_at_end) nisentry->write(fout);
			for (int i=0; i<nbu; i++){
				getitem(i)->write(fout);
			}
			if (nisentry != NULL && nis_at_end) nisentry->write(fout);
			ret = configf->fclose(fout);
			// ret = configf->relink_tmp();
			if (shadows != NULL) shadows->write(priv);
		}
	}
	return ret;
}

/*
	Write the user accounts only if override maintenance function
	are not in place, since these function do the modification themselves.

	If the user account override function are in place, this means that
	other processes have updated the passwd database, so our own
	copy if invalid. Instead of writing, we reload
*/
PUBLIC int USERS::writeif (PRIVILEGE *priv, USER *&usr)
{
	int ret = -1;
	if (user_fct_mod != NULL && may_override()){
		SSTRING id;
		char enabled =1;
		if (usr != NULL){
			id.setfrom (usr->getname());
			enabled = usr->enabled;
		}
		reload();
		if (usr != NULL){
			usr = getitem(id.get());
			if (usr == NULL){
				xconf_notice ("Programming error:\n"
					"userconf/users.cc:writeif usr==NULL");
			}else{
				usr->enabled = enabled;
			}
		}
		ret = 0;
	}else{
		ret = write (priv);
	}
	return ret;
}
PUBLIC int USERS::writeif (PRIVILEGE *priv)
{
	USER *pt = NULL;
	return writeif (priv,pt);
}

struct USRACC_FILTER{
	SSTRING prefix;		// Prefix for filtering the list
	SSTRING gecos;
	int gid;			// Group to filter or -1
	int from;
	int to;
	USRACC_FILTER(){
		gid = -1;
		from = 0;
		to = 65535;
	}
};


PRIVATE MENU_STATUS USERS::setselectprefix(
	struct USRACC_FILTER &filter,
	const USER *like,
	bool may_add)
{
	MENU_STATUS ret = MENU_ACCEPT;
	int nbsel = 0;
	int nbu = getnb();
	for (int i=0; i<nbu; i++){
		USER *usr = getitem(i);
		if (usr->is_like(like)) nbsel++;
	}
	int prefix_trig = linuxconf_getprefixtrig();
	if (prefix_trig != 0 && nbsel > prefix_trig){
		DIALOG dia;
		dia.newf_str (MSG_U(F_USERPREFIX,"Login Prefix"), filter.prefix);
		dia.newf_str (MSG_U(F_GECOSPREFIX,"Full name prefix"), filter.gecos);
		dia.newf_num(MSG_U(F_UID_FROM,  "UID From"),filter.from);
		dia.newf_num(MSG_U(F_UID_TO,    "UID To"),  filter.to);
		SSTRING group;
		GROUPS groups;
		if (like == NULL){
			GROUP *g = groups.getfromgid(filter.gid);
			if (g != NULL) group.setfrom (g->getname());
			FIELD_COMBO *grpl = dia.newf_list (MSG_R(F_GROUP),group);
			grpl->addopt ("");
			groups.setcombo (grpl);
		}
		int buts = MENUBUT_CANCEL|MENUBUT_ACCEPT;
		if (may_add) buts |= MENUBUT_ADD;
		int nof = 0;
		ret = dia.edit (MSG_U(T_USERPREFIX,"Filter control")
			,MSG_U(I_USERPREFIX
				,"The list of users is long, so you\n"
				 "may want to filter it a bit by providing\n"
				 "a prefix to search. An empty prefix means to show\n"
				 "all users.")
			,help_filter
			,nof,buts);
		if (ret == MENU_ACCEPT){
			if (!group.is_empty()){
				filter.gid = groups.getgid(group.get());
			}else{
				filter.gid = -1;
			}
		}
	}
	return ret;
}

static bool users_match (
	const char *str,
	const char *prefix,
	int len)
{
	bool ret = true;
	if (len != 0){
		if (prefix[0] != '*'){
			ret = strncasecmp(str,prefix,len)==0;
		}else{
			ret = strstr (str,prefix+1) != NULL;
		}
	}
	return ret;
}


/*
	Select one user from the list.
	May return NULL if no valid selection was done (escape). See code.
*/
PRIVATE USER *USERS::select_sorted(
	USER *like,	// Used to select which user to pick.
			// the function USER::islike() is called for
			// each.
	int may_add,	// Set the delete and add button
	MENU_STATUS &code,
	DIALOG_RECORDS &dia,
	int &choice,	// Will contain the selection. Not so useful
					// but help for the reentrancy of the list
					// (It reedit on the last item selected).
	struct USRACC_FILTER &filter)
{
	GROUPS groups;
	int nbu = getnb();
	int myfrom;
	int myto;
	
	/* do some primitive boundary checks */
	myfrom = (filter.from < 1)  ? 1 : filter.from;
	myto   = (filter.to < myfrom) ? myfrom: filter.to;
	/* #Specification: userconf / user account / root bin ...
		Some special account are simply left out of the configuration
		menu. These account are never edited. They make the list larger
		for no reason.

		Also account with special shells are not shown. This include
		accounts like uucp and slip. Theu are show in a separate menu.

		The same functionnality is used to edit those accounts, but
		the edition is trigerred from different menus.
	*/
	if (dia.getnb()==0){
		dia.newf_head ("",MSG_U(H_USERS,"Account\tName\tUid\tGroup"));
	}
	int len_prefix = filter.prefix.getlen();
	const char *prestr = filter.prefix.get();
	int len_gecos = filter.gecos.getlen();
	const char *gecos = filter.gecos.get();
	int lookup[nbu];
	int shown = 0;
	for (int i=0; i<nbu; i++){
		USER *usr = getitem(i);
		int uid = usr->getuid();
		if (usr->is_like(like)
		    && users_match(usr->getname(),prestr,len_prefix)
		    && users_match(usr->getgecos(),gecos,len_gecos)
			&& (filter.gid == -1 || filter.gid == usr->getgid())
		    && uid >= myfrom
		    && uid <= myto){
		        char buf[100];
			int gid = usr->getgid();
			GROUP *grp = groups.getfromgid(gid);
			char grpstr[100];
			if (grp == NULL){
				sprintf (grpstr,"%d",gid);
			}else{
				strcpy (grpstr,grp->getname());
			}
			char gecos[50];
			strncpy (gecos,usr->getgecos(),49);
			gecos[49] = '\0';
			sprintf (buf,"%s\t%d\t%s",gecos,usr->getuid(),grpstr);
			dia.set_menuitem (shown,usr->getname(),buf);
			lookup[shown++] = i;
		}
	}
	dia.remove_last (shown +1);
	if (may_add) dia.addwhat (MSG_R(I_TOADD));
	code = dia.editmenu (MSG_U(T_USERACCT,"Users accounts")
		,may_add
			? MSG_U(I_CANEDIT,"You can edit, add, or delete users")
			: MSG_U(I_SELECT,"You must select one of those users")
		,help_users
		,choice,0);
	// Locate the selected user in the list, given that not all
	// user where displayed.
	return choice >=0 && choice < shown
		? getitem(lookup[choice]) : (USER*)NULL;
}

/*
	Select one user from the list.
	May return NULL if no valid selection was done (escape). See code.
*/
PUBLIC USER *USERS::select(
	USER *like,	// Used to select which user to pick.
			// the function USER::islike() is called for
			// each.
	int may_add,	// Set the delete and add button
	MENU_STATUS &code,
	DIALOG_RECORDS &dia,
	int &choice,	// Will contain the selection. Not so useful
					// but help for the reentrancy of the list
					// (It reedit on the last item selected).
	struct USRACC_FILTER &filter)
{
	USERS sorted(this);
	sorted.sortbyname();
	return sorted.select_sorted(like,may_add,code,dia,choice,filter);
}

PUBLIC int USERS::dodel (
	USER *usr,
	USER_DELOPER oper,
	GROUPS &groups,
	PRIVILEGE *priv)
{
	int ret = 0;
	net_introlog (NETINTRO_DELETEUSER);
	net_prtlog (NETLOG_VERB,MSG_U(I_DELETING,"Delete account %s (%s)\n")
		,usr->getname(),usr->getgecos());
	GROUP *g = groups.getfromgid(usr->getgid());
	const char *group = g != NULL ? g->getname() : "";
	if (oper == DELOPER_ARCHIVE){
		ret = runcmd(policies_getarchivecmd(),usr,group);
	}else if (oper == DELOPER_DELETE){
		ret = runcmd(policies_getdeletecmd(),usr,group);
	}
	if (ret == 0){
		ret = rundeletecmd(usr,group);
	}
	if (ret == 0 && user_fct_del == NULL){
		if (groups.delmember (usr->getname())){
			groups.write (priv);
		}
	}
	return ret;
}

/*
	Check if distribution dependant methode may be used.
	They can't be used on virtual email domain and they can't be used
	in administration trees.
	Once these "linuxconf specifics" become widespread, this may change
*/
PUBLIC bool USERS::may_override ()
{
	return domain.cmp("/")==0 && context_isroot();
}

/*
	Return true if the distribution dependant mechanism is used to create
	user accounts
*/
PUBLIC bool USERS::has_add_override()
{
	return user_fct_add != NULL && may_override();
}

/*
	Use a distribution dependant method to update the user account database
	if available
*/
PRIVATE int USERS::override (
	PRIVILEGE *priv,
	USER *usr,
	int (*fct) (USER *, SHADOW *))
{
	int ret = -1;
	if (fct == NULL || ! may_override()){
		ret = 0;
	}else if (perm_access(priv,MSG_U(P_USERDB
		,"to maintain the user database"))){
		ret = (*fct)(usr,getshadow(usr));
		if (ret != 0){
			net_showlastlog();
		}
	}
	return ret;
}

/*
	Use a distribution dependant method to create a user account database
	if available
*/
PRIVATE int USERS::override_add (
	PRIVILEGE *priv,
	USER *usr,
	const char *group)
{
	int ret = -1;
	if (user_fct_add == NULL || ! may_override()){
		ret = 0;
	}else if (perm_access(priv,MSG_R(P_USERDB))){
		bool createdir = group_homeneeded(group);
		ret = (*user_fct_add)(usr,getshadow(usr),createdir);
		if (ret == -1){
			xconf_error (MSG_U(E_FAILCREATE,"Can't create user account"));
		}else if(createdir){
			chmod (usr->wrkdir.get(),group_getcreateperm(group));
		}
	}
	return ret;
}

PRIVATE int USERS::docreate (USER *usr, PRIVILEGE *priv, const char *group)
{
	net_introlog (NETINTRO_CREATEUSER);
	net_prtlog (NETLOG_VERB,MSG_U(I_CREATING
		,"Creating user account %s (%s)\n")
		,usr->getname(),usr->getgecos());
	add (usr);
	return override_add(priv,usr,group);
}


/*
	Edit/Commit a single account
	This is where we will call the modules one day so that they
	create the various objects (like the QUOTA_EDIT) which participate
	to the user account dialog
*/
PRIVATE int USERS::editone_precut (
	USER *usr,
	bool is_new,
	PRIVILEGE *priv,	// Privilege required to manage those accounts
						// or NULL if only root can do this
	unsigned may_edit)	// Which fields may be edited
{
	// #Specbeg: USERACCT_COMNG / dialog setup
	/*
		The following code define the context for the co-manager
		and enable them.
	*/
	USERACCT_COMNGS comngs;
	comngs.set_bool ("is_new",is_new);
	comngs.set_str  ("domain",domain.get());
	comngs.set_str  ("name",usr->getname());
	comngs.set_int ("categ",usr->getcateg());
	// This triggers the REGISTER_USERACCT_COMNG objets
	comngs.getall  ("user");
	// #Specend:
	QUOTA_EDIT qedit (is_new ? (const char*)NULL : usr->getname()
		, QUOTA_USER);
	GROUPS groups;
	comngs.set_obj ("groups",&groups);
	comngs.set_obj ("users",this);
	USER_DELOPER deloper = DELOPER_NONE;
	int code = usr->edit (*this,groups,is_new,priv,may_edit,qedit
		,comngs,deloper,domain.get());
	if (code != -1){
		GROUP *g = groups.getfromgid(usr->getgid());
		const char *group = g != NULL ? g->getname() : "";
		SSTRING id (usr->getname());	// We must take a copy
										// because the usr object may be deleted
										// later.
		const char *tbmessage[]={
			id.get(),domain.get()
		};
		if (code == 0){
			if (is_new){
				if (docreate(usr,priv,group) != -1
					&& writeif (priv,usr) != -1){
					module_sendmessage ("newuser",2,tbmessage);
					if (usr->enabled){
						char info[100];
						snprintf (info,sizeof(info)-1,"password:%s"
							,usr->getname());
						html_setcutinfo (info);
						usr->editpass(getshadow(usr),true,may_override()
							,domain.get());
					}
				}
			}else{
				net_introlog (NETINTRO_MODIFYUSER);
				if (override (priv,usr,user_fct_mod)!=-1){
					module_sendmessage ("chguser",2,tbmessage);
				}
			}
		}else if (code == 1){
			if (is_new){
				delete usr;
			}else if (dodel(usr,deloper,groups,priv)!=-1){
				module_sendmessage ("deluser",2,tbmessage);
				override (priv,usr,user_fct_del);
				remove_del (usr);
			}else{
				if (dialog_yesno (MSG_U(E_NOTDEL,"Account not deleted")
					,MSG_U(I_NOTDEL
						,"User account was not deleted\n"
						 "because the post-deletion command failed\n"
						 "\n"
						 "Do you want to see the log ?")
					,help_nil)
					==MENU_YES){
					net_showlog();
				}
				if (dialog_yesno(MSG_U(E_DELANYWAY,"Delete anyway")
					,MSG_U(I_DELANYWAY
						,"Would you like to delete this account\n"
						 "even if the post-deletion command failed ?")
					,help_nil)==MENU_YES){
					override (priv,usr,user_fct_del);
					remove_del (usr);
				}else{
					code = -1;
				}
			}
		}
		if (code != -1){
			USER tmp(usr);	// writeif may invalidate the usr object, take a copy
			writeif(priv);
			if (user_fct_mod == NULL) groups.write();
			// The qedit.save must be done after the /etc/passwd is saved
			if (code == 0){
				if (domain.cmp("/")==0) qedit.save(priv);
				if (is_new) runcreatecmd(&tmp,group);
				comngs.save (priv);
			}else if (code == 1){
				if (domain.cmp("/")==0) qedit.deluser(priv);
				comngs.deluser(priv);
			}
		}
	}
	return code;
}

/*
	Edit/Commit a single account
	This is where we will call the modules one day so that they
	create the various objects (like the QUOTA_EDIT) which participate
	to the user account dialog
*/
PUBLIC int USERS::editone (
	USER *usr,
	bool is_new,
	PRIVILEGE *priv,	// Privilege required to manage those accounts
						// or NULL if only root can do this
	unsigned may_edit)	// Which field may be edited
{
	int ret = -1;
	/* #Specification: user account / privilege / set default
		When editing a user account, we call the function perm_setdefprivi
		to register the default privilege. This is done to help co-manager
		do there tasks withoug being forced to pass PRIVILEGE pointers
		around all the time.

		This is annoying as it does not allow a sub-process to drop privilege
		easily (just by setting the PRIVILEGE pointer to NULL for example).
	*/
	PRIVILEGE *old = perm_setdefprivi (priv);
	const char *info = html_getcutinfo();
	if (is_new && info != NULL && strncmp(info,"password:",9)==0){
		usr = getitem (info+9);
		// fprintf (stderr,"Taking the cut for user :%s: %p\n",info+9,usr);
		if (usr != NULL){		
			ret = usr->editpass(getshadow(usr),true,may_override()
				,domain.get());
			if (ret != -1){
				writeif(priv,usr);
			}
		}
	}else{
		// fprintf (stderr,"Not taking the cut\n");
		ret = editone_precut (usr,is_new,priv,may_edit);
	}
	perm_setdefprivi (old);
	return ret;
}


/*
	Add one new user
	Return -1 if the user was not added.
*/
PUBLIC int USERS::addone (
	USER *special,
	const char *name,	// Proposed login name
	const char *fullname,	// gecos field suggested
	PRIVILEGE *priv,	// Privilege required to manage those accounts
						// or NULL if only root can do this
	unsigned may_edit)	// Which field may be edited
{
	int ret = -1;
	/* 
	   Don't add the user if it would exceed the objects maxusers limit.
	   Used for virtual domains with user limit (maxusers=0 means unlimited).
	*/
	if ((maxusers != 0) && (getnb() >= maxusers)) {
		xconf_error(MSG_U(E_MAXUSERS
		    ,"Adding user would exceed the userlimit for this domain\n(%d>%d) - aborting.")
			,getnb()+1,maxusers);
	}else{
		USER *user = new USER;
		user->setlike (special);
		user->setname (name);
		user->setgecos (fullname);
		ret = editone (user,true,priv,may_edit);
	}
	return ret;
}

PUBLIC void USERS::remove_del (USER *usr)
{
	SHADOW *sha = getshadow(usr);
	if (sha != NULL) shadows->remove_del (sha);
	ARRAY::remove_del (usr);
}
/*
	General edition (addition/deletion/correction) of /etc/passwd
*/
PUBLIC int USERS::edit(
	USER *special,		// Template for user creation
						// and selection.
	PRIVILEGE *priv,	// Privilege required to manage those accounts
						// or NULL if only root can do this
	unsigned may_edit)
{
	int ret = -1;
	if (perm_access (priv,MSG_U(P_USERACCOUTS,"To view and edit user accounts"))){
		/* predefined, we normally need these values... */
		USRACC_FILTER filter;
		MENU_STATUS selcod = setselectprefix(filter,special,true);
		if (selcod == MENU_ADD){
			if (perm_access(priv,MSG_R(P_USERDB))){
				addone (special,NULL,NULL,priv,may_edit);
			}
		}else if (selcod == MENU_ACCEPT){
			DIALOG_RECORDS dia;
			int choice = 0;
			while (1){
				MENU_STATUS code;
				USER *usr = select (special,1,code,dia,choice,filter);
				if (code == MENU_ESCAPE || code == MENU_QUIT){
					break;
				}else if (code == MENU_OK){
					if (usr != NULL){
						int status = editone(usr,false,priv,may_edit);
						if (status != -1){
							ret = 0;
						}
					}
				}else if (perm_access(priv,MSG_R(P_USERDB))){
					if (code == MENU_ADD){
						addone (special,NULL,NULL,priv,may_edit);
					}
				}
			}
		}
	}
	return ret;
}
static int cmpbyname (const ARRAY_OBJ *o1, const ARRAY_OBJ *o2)
{
	USER *g1 = (USER*) o1;
	USER *g2 = (USER*) o2;
	return strcmp(g1->getname(),g2->getname());
}
/*
	Sort the array of group by name
*/
PUBLIC void USERS::sortbyname()
{
	sort (cmpbyname);
}
static int cmpbygid (const ARRAY_OBJ *o1, const ARRAY_OBJ *o2)
{
	USER *g1 = (USER*) o1;
	USER *g2 = (USER*) o2;
	int ret = g1->getgid() - g2->getgid();
	if (ret == 0){
		ret = strcmp(g1->getname(),g2->getname());
	}
	return ret;
}
/*
	Sort the array of group by gid, and name
*/
PUBLIC void USERS::sortbygid()
{
	sort (cmpbygid);
}
/*
	General edition of users account and special account
	See USERS::edit()
*/
void users_edit(
	USER *special,	// Template for a special account.
					// Or NULL.
	PRIVILEGE *priv,	// Privilege required to manage those accounts
						// or NULL if only root can do this
	unsigned may_edit)	// Which field may be edited (USRACCT_EDIT____
{
	USERS users;
	users.edit(special,priv,may_edit);
}

/*
	Add one new special user.
	Return -1 if the user was not created
*/
int users_addone(
	const char *name,
	const char *group,		// Which group to use or NULL.
	const char *fullname)	// gecos field suggested
{
	
	int ret = -1;
	PRIVILEGE *priv = special_getpriv (group);
	if (perm_access(priv,MSG_R(P_USERDB))){
		USER *special;
		if (special_init (group,special) != -1){
			USERS users;
			ret = users.addone(special,name,fullname,priv
				,USRACCT_EDITSHELL|USRACCT_EDITGROUP|USRACCT_EDITSUPGRP);
		}
		delete special;	
	}
	return ret;
}

/*
	Return != 0 if a user account exist.
*/
int user_exist (const char *name)
{
	return getpwnam(name)!=NULL;
}
	

/*
	Edit one user spec.
	If the user does not exist, ask if we want to create it.

	Return -1 if the account was not created
*/
int users_editone(
	const char *name,
	const char *group,		// Group to use (or NULL) if the user
							// is created
	const char *fullname)	// gecos field suggested when creating a new
							// account
{
	int ret = -1;
	USERS users;
	PRIVILEGE *priv = special_getpriv (group);
	if (perm_access(priv,MSG_R(P_USERDB))){
		USER *user = users.getitem(name);
		if (user == NULL){
			char buf[300];
			sprintf (buf,MSG_U(I_USERCREATE,"User account %s does not exist\n"
				 "Do you want to create it"),name);
			if (dialog_yesno(MSG_U(Q_USERCREATE,"Account creation")
				,buf,help_users)==MENU_YES){
				ret = users_addone	(name,group,fullname);
			}
		}else{
			ret = users.editone(user,false,priv
				,USRACCT_EDITSHELL|USRACCT_EDITGROUP|USRACCT_EDITSUPGRP);
		}
	}
	return ret;
}

PRIVATE int USERS::runcmd(const char *cmd, USER *usr, const char *group)
{
	int ret = 0;
	if (cmd != NULL && cmd[0] != '\0'){
		char gecos_esc[1000];
		usr->getgecos_esc (gecos_esc,sizeof(gecos_esc)-1);
		char buf[strlen(cmd)+10+PATH_MAX];
		sprintf (buf,"%s --uid %s --name %s --basehome %s --home %s --domain %s --group \"%s\""
			,cmd,usr->getname(),gecos_esc
			,getstdhome(group)
			,usr->gethome()+1
			,domain.get()
			,group);
		net_resetnberr();
		int old = net_setshowmode(0);
		ret = netconf_system (60,buf);
		if (net_getnberr()>0) ret = -1;
		net_setshowmode(old);
	}
	return ret;
}

/*
	Run a user supplied post creation command
*/
PROTECTED int USERS::runcreatecmd(USER *usr, const char *group)
{
	return runcmd(policies_getcreatecmd(),usr,group);
}
/*
	Run a user supplied pre deletion command
*/
PROTECTED int USERS::rundeletecmd(USER *usr, const char *group)
{
	return runcmd(policies_getpostdeletecmd(),usr,group);
}

/*
	Add a user account, no question asked
*/
PUBLIC int USERS::addbatch (
	const char *id,
	const char *group,
	const char *name,
	const char *shell,
	bool locked)		// This account will be created locked
{
	int ret = -1;
	USER *u = getitem(id);
	GROUPS groups;
	GROUP *g = groups.getitem(group);
	if (u != NULL){
		fprintf (stderr,MSG_U(E_UEXIST,"User account %s already exists\n")
			,id);
	}else if (g == NULL){
		fprintf (stderr,MSG_U(E_GNOTEXIST,"Group %s does not exist\n")
			,group);
	}else{
		u = new USER;
		//u->setlike (special);
		u->setname (id);
		u->setgecos (name);
		u->setdefhome (*this,group);
		u->shell.setfrom (shell);
		u->uid = getnewuid();
		u->gid = g->getgid();
		SHADOW *shadow = NULL;
		if (has_shadow()){
			shadow = new SHADOW ();
			shadow->name.setfrom (id);
			addshadow (shadow);
		}
		if (u->sethome (NULL,group,false,*this) != -1){
			if (docreate (u,NULL,group) != -1
				&& writeif (NULL,u) != -1){
				if (locked){
					u->update_passwd (NULL,shadow,true,domain.get());
					writeif (NULL,u);
				}
				const char *tbmessage[]={
					u->getname(),domain.get()
				};
				module_sendmessage ("newuser",2,tbmessage);
				if (domain.cmp("/")==0) quota_applyone (id);
				runcreatecmd (u,group);
				ret = 0;
			}
		}
	}
	return ret;
}

/*
	Delete a user account, no question asked
*/
PUBLIC int USERS::delbatch (const char *id, USER_DELOPER oper)
{
	int ret = -1;
	USER *u = getitem(id);
	if (u == NULL){
		fprintf (stderr,MSG_U(E_UNOTEXIST,"User account %s does not exist\n")
			,id);
	}else{
		GROUPS groups;
		if (dodel (u,oper,groups,NULL)==0){
			SSTRING name (u->getname());
			override (NULL,u,user_fct_del);
			remove_del (u);
			ret = writeif(NULL);
			if (ret != -1){
				const char *tbmessage[]={
					name.get(),domain.get()
				};
				module_sendmessage ("deluser",2,tbmessage);
				if (domain.cmp("/")==0){
					QUOTA_EDIT qedit (id,QUOTA_USER);
					qedit.deluser(NULL);
				}
			}
		}else{
			fprintf (stderr,MSG_U(E_PREDELFAIL
				,"Pre-delete command failed for account %s\n")
				,id);
		}
	}
	return ret;
}

/*
	Add one user account
*/
int users_add (
	const char *id,
	const char *group,
	const char *name,
	const char *shell)
{
	USERS users;
	return users.addbatch (id,group,name,shell,true);
}

/*
	Delete one user account
*/
int users_del (const char *id, USER_DELOPER oper)
{
	USERS users;
	return users.delbatch (id,oper);
}

static void users_edituser (const char *v, bool setting)
{
	users_editone (v,NULL,NULL);
}

static void users_list (SSTRINGS &tb)
{
	USERS users;
	for (int i=0; i<users.getnb(); i++){
		USER *u = users.getitem(i);
		tb.add (new SSTRING (u->getname()));
	}
}

#include <modregister.h>
static PUBLISH_VARIABLES_MSG user_var_list1[]={
	{"home",P_MSG_R(F_HOME)},
	{"shell",P_MSG_R(F_SHELL)},
	{"enable",P_MSG_R(I_ENABLED)},
	{"gecos",P_MSG_R(F_FULLNAME)},
	{"uid",P_MSG_R(F_UID)},
	{ NULL, NULL }
};

static REGISTER_VARIABLES user_registry1("user","users",user_var_list1
	,NULL,users_edituser,users_list);


