#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include "../paths.h"
#include <misc.h>
#include "userconf.h"
#include "userconf.m"
#include "internal.h"
#include <daemoni.h>
#include <netconf.h>
#include <dialog.h>

static SSTRINGS tbuser;
static SSTRINGS tbppp;
static SSTRINGS tbslip;

static SSTRING defshell,defslip,defppp;

static const char K_DEFSHELL[]="defshell";
static const char K_SHELLS[]="shells";
static const char K_PPP[]="ppp";
static const char K_SLIP[]="slip";


static USERCONF_HELP_FILE help_shells ("shells");
static CONFIG_FILE f_shells (ETC_SHELLS,help_shells
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL);



/*
	Obtain the default shell for a type of user
*/
static void shells_readdef (const char *key, SSTRING &def, const char *defval)
{
	def.setfrom (linuxconf_getval(K_DEFSHELL,key,defval));
}

static void shells_readuser ()
{
	if (tbuser.getnb() == 0){
		/* #Specification: userconf / /etc/shells & getusershell()
			Shells available to configure user accounts are defined
			by getusershell() (which reads /etc/shells optionnally).

			It is assumed that the first entry in /etc/shells is the default
			shell (When the field is empty in /etc/passwd).
		*/
		setusershell();
		char *pt;
		const char *first = NULL;
		while ((pt=getusershell())!=NULL){
				SSTRING *s = new SSTRING(pt);
				tbuser.add (s);
				if (first == NULL) first = s->get();
		}
		endusershell();
		defshell.setfrom (linuxconf_getval(K_DEFSHELL,K_SHELLS,"/bin/bash"));
	}
}

static void shells_readothers(
	const char *key,
	SSTRINGS &tb,
	SSTRING &def,
	const char *defval)
{
	tb.remove_all();
	linuxconf_getall (K_SHELLS,key,tb,1);
	shells_readdef (key,def,defval);
	if (tb.getnb()==0) tb.add (new SSTRING(defval));
}

static bool shells_were_read = false;
/*
	Read all shells list in memory for all type of users
*/
static void shells_readall()
{
	if (!shells_were_read){
		tbuser.remove_all();
		shells_readuser();
		shells_readothers(K_PPP,tbppp,defppp,"/usr/lib/linuxconf/lib/ppplogin");
		shells_readothers(K_SLIP,tbslip,defslip,SBIN_SLIPLOGIN);
		shells_were_read = true;
	}
}

static void shells_rereadall()
{
	shells_were_read = false;
	shells_readall();
}

/*
	Return != 0 if a path is an accepted shell
*/
bool shells_isok(const char *path)
{
	shells_readuser();
	bool ret = false;
	if (path[0] == '\0'){
		ret = true;
	}else{
		ret = tbuser.lookup(path)!=-1;
	}
	return ret;
}

/*
	Return true is path exist and is executable
	If path is empty, this imply the default shell, which is ok.
*/
bool shells_exist (const char *path)
{
	struct stat sstat;
	bool ret = false;
	if (path[0] == '\0'){
		ret = true;
	}else{
		if (stat(path,&sstat)!=-1
			&& S_ISREG(sstat.st_mode)
			&& sstat.st_mode & 1){
			ret = true;
		}
	}
	return ret;
}


/*
	Edit the list of shells available
	Return 0 if the list has to be saved
*/
static int shell_edit(
	SSTRINGS &tb,
	SSTRING  &def,
	const char *title)
{
	int ret = -1;
	DIALOG dia;
	tb.remove_empty();
	tb.sort();
	tb.add (new SSTRING);
	tb.add (new SSTRING);
	dia.newf_str (MSG_U(F_DEFSHELL,"Default shell"),def);
	dia.last_noempty();
	dia.newf_title ("","");
	int i;
	for (i=0; i<tb.getnb(); i++){
		dia.newf_str ("",*tb.getitem(i));
	}
	dia.addwhat (MSG_U(I_EMPTYLINE,"Select [Add] to add an empty line"));
	int nof = 0;
	while (1){
		MENU_STATUS code = dia.edit (title
			,""
			,help_shells
			,nof
			,MENUBUT_ADD|MENUBUT_ACCEPT|MENUBUT_CANCEL);
		if (code == MENU_CANCEL || code == MENU_ESCAPE){
			break;
		}else if (code == MENU_ADD){
			SSTRING *s = new SSTRING;
			tb.add (s);
			dia.newf_str ("",*s);
		}else if (code == MENU_ACCEPT){
			if (tb.lookup(def.get())==-1){
				xconf_error (MSG_U(E_IVLDDEF
					,"Default shell is not a member\n"
					 "of the list"));
			}else{
				for (i=0; i<tb.getnb(); i++){
					SSTRING *s = tb.getitem(i);
					const char *pts = s->get();
					if (!s->is_empty() && !shells_exist(pts)){
						nof = i;
						if (file_exist(pts)){
							xconf_error (MSG_U(E_SHELLNOTEXEC
								,"Shell %s is not executable")
								,pts);
						}else{
							xconf_error (MSG_U(E_SHELLUNKNOWN
								,"Shell %s does not exist")
								,pts);
						}
						break;
					}
				}
				if (i == tb.getnb()){
					ret = 0;
					break;
				}
			}
		}
	}
	return ret;
}


void shells_edituser()
{
	shells_readall();
	if (shell_edit(tbuser,defshell
		,MSG_U(T_USERSHELL,"Standard user shells"))==0){
		linuxconf_setcursys (subsys_policies);
		tbuser.remove_empty();
		FILE_CFG *fout = f_shells.fopen("w");
		if (fout != NULL){
			for (int i=0; i<tbuser.getnb(); i++){
				SSTRING *s = tbuser.getitem(i);
				fprintf (fout,"%s\n",s->get());
			}
			fclose (fout);
		}
		linuxconf_replace (K_DEFSHELL,K_SHELLS,defshell);
		linuxconf_save();
	}
}

static void shells_editothers(
	const char *key,
	SSTRINGS &tb,
	SSTRING &def,		// Default shell
	const char *title)
{
	shells_readall ();
	if (shell_edit(tb,def,title)==0){
		tb.remove_empty();
		linuxconf_setcursys (subsys_policies);
		linuxconf_removeall (K_SHELLS,key);
		for (int i=0; i<tb.getnb(); i++){
			SSTRING *s = tb.getitem(i);
			linuxconf_add (K_SHELLS,key,s->get());
		}
		linuxconf_replace (K_DEFSHELL,key,def);
		linuxconf_save();
	}else{
		shells_rereadall();
	}
}

void shells_editppp ()
{
	shells_editothers(K_PPP,tbppp,defppp
		,MSG_U(T_PPPSHELL,"Available PPP login shells"));
}
void shells_editslip ()
{
	shells_editothers(K_SLIP,tbslip,defslip
		,MSG_U(T_SLIPSHELL,"Available SLIP login shells"));
}

/*
	Return != 0 if path is an accepted login shell for PPP account
*/
int shells_isppp (const char *path)
{
	shells_readall();
	return tbppp.lookup (path)!=-1;
}

/*
	Return != 0 if path is an accepted login shell for SLIP account
*/
int shells_isslip (const char *path)
{
	shells_readall();
	return tbslip.lookup (path)!=-1;
}

/*
	Get the list of available shells for normal users
*/
const SSTRINGS *shells_getuserlist()
{
	shells_readall ();
	return &tbuser;
}
/*
	Get the list of available shells for PPP accounts
*/
const SSTRINGS *shells_getppplist()
{
	shells_readall ();
	return &tbppp;
}
/*
	Get the list of available shells for SLIP accounts
*/
const SSTRINGS *shells_getsliplist()
{
	shells_readall ();
	return &tbslip;
}
const char *shells_getuucpdefault()
{
	DAEMON_INTERNAL *dae = daemon_find ("uucico");
	const char *path = USR_LIB_UUCP_UUCICO;
	if (dae != NULL) path = dae->getpath();
	return path;
}
/*
	Get the list of available shells for UUCP accounts
*/
const SSTRINGS *shells_getuucplist()
{
	static SSTRINGS tbuucp;
	if (tbuucp.getnb()==0){
		tbuucp.add (new SSTRING(shells_getuucpdefault()));
	}
	return &tbuucp;
}

/*
	Return the path of the default shell for normal user.
*/
const char *shells_getdefault()
{
	shells_readall ();
	return defshell.get();
}
/*
	Return the path of the default shell for ppp users.
*/
const char *shells_getpppdefault()
{
	shells_readall();
	return defppp.get();
}
/*
	Return the path of the default shell for slip users.
*/
const char *shells_getslipdefault()
{
	shells_readall();
	return defslip.get();
}

