#include <string.h>
#include <assert.h>
#include <errno.h>
#include <usercomng.h>
#include "internal.h"
#include "apache.m"
#include "keyword.h"

static APACHE_HELP_FILE help_domain ("domain");

/*
	Make sure the document root directory  for the virtual host does exist
*/
static int vhostedit_createroot(VIEW_SECTION &sec)
{
	int ret = -1;
	VIEWITEM *it = sec.locate (K_DOCUMENTROOT);
	if (it != NULL){
		const char *path = viewsub_getval(it);
		ret = 0;
		if (!context_fexist(path)){
			ret = context_mkdirp (path,"root","root",0755);
		}
	}
	return ret;
}


static int vhostedit_edit(VIEW_SECTION &sec, const char *_name)
{
	int ret = -1;
	DIALOG dia;
	VIEWEDIT vedit (sec,dia);
	SSTRING name(_name);
	dia.newf_str (MSG_U(F_VDOMNAME,"Virtual host name"),name);
	defhost_setdia (vedit,true,false);
	vedit.newf_str (MSG_U(F_SERVERALIAS,"Server aliases"),K_SERVERALIAS);
	HTTPD_FEATURES features (sec);
	features.setdia (dia);
	ssl_setdia(vedit,true);
	int nof = 0;
	while (1){
		MENU_STATUS code = dia.edit (MSG_U(T_VDOMAIN,"Virtual host setup")
			,MSG_U(I_VDOMAIN,"You are allowed to enter configurations\n"
				"for a virtual host/domain\n")
			,help_domain
			,nof,MENUBUT_ACCEPT|MENUBUT_CANCEL|MENUBUT_DEL);
		if (code == MENU_ESCAPE || code == MENU_CANCEL){
			break;
		}else if (code == MENU_DEL){
			if (xconf_delok()){
				ret = 1;
				break;
			}
		}else{
			sec.updatehead (name.get());
			features.update("\t",sec);
			vedit.update("\t");
			vhostedit_createroot(sec);
			ret = 0;
			break;
		}
	}
	if (ret != 0) dia.restore();
	return ret;
}

PUBLIC int HTTPD_CONFIG::editvhosts()
{
	int ret = -1;
	int nof = 0;

	DIALOG_RECORDS dia;
	dia.setkeyformat (HTML_KEY_INDEX);
	dia.newf_head ("",MSG_U(T_DOMAINS,"Virtual host\tServer name"));		

	dia.addwhat (MSG_U(I_NEWDOMAIN,"Select [Add] to add a new virtual domain"));
	while (1){
		SSTRING_IDXS tbs;
		itemss.locatesection (K_VIRTUALHOST,tbs);
		tbs.sort();
		for (int i=0; i<tbs.getnb(); i++){
			SSTRING_IDX *s = tbs.getitem(i);
			const char *v = s->get();
			VIEW_SECTION sec (K_VIRTUALHOST,itemss);
			sec.setview (s->getidx());
			const char *servername = "";
			VIEWITEM *it = sec.locate (K_SERVERNAME);
			if (it != NULL){
				servername = viewsub_getval (it);
			}
			dia.set_menuitem(i,v,servername);
		}
		dia.remove_last (tbs.getnb()+1);

		MENU_STATUS code = dia.editmenu (MSG_R(T_DOMAINS)
			,MSG_U(I_DOMAINS
				,"You can define multiple independant virtual\n"
				 "domains on this server")
			,help_domain
			,nof,0);
		if (code == MENU_ESCAPE || code == MENU_QUIT){
			break;
		}else if (code == MENU_ADD){
			VIEW_SECTION s(K_VIRTUALHOST,itemss);
			ret = vhostedit_edit (s,NULL);
			if (ret == 0) write();
		}else if (nof < tbs.getnb() && nof >= 0){
			SSTRING_IDX *s = tbs.getitem(nof);
			const char *v = s->get();
			VIEW_SECTION sec (K_VIRTUALHOST,itemss);
			sec.setview (s->getidx());
			ret = vhostedit_edit(sec,v);
			if (ret == 0){
				write();
			}else if (ret == 1){
				sec.delall();
				write();
			}
		}
	}
	return ret;
}

class VIRTUAL_COMNG: public USERACCT_COMNG{
	/*~PROTOBEG~ VIRTUAL_COMNG */
public:
	VIRTUAL_COMNG (DICTIONARY&_dict);
	int deluser (PRIVILEGE *);
	int save (PRIVILEGE *priv);
	void setupdia (DIALOG&dia);
	int validate (DIALOG&, int &nof);
	/*~PROTOEND~ VIRTUAL_COMNG */
};

PUBLIC VIRTUAL_COMNG::VIRTUAL_COMNG(
	DICTIONARY &_dict)
	: USERACCT_COMNG (_dict)
{
}


PUBLIC void VIRTUAL_COMNG::setupdia (
	DIALOG &dia)
{
	dia.newf_title (MSG_U(T_MVIRTUAL,"web"),1
			,"",MSG_R(T_MVIRTUAL));
	//defhost_setdia (dia,false);
	//opt.feats.setdia (dia);
}	

PUBLIC int VIRTUAL_COMNG::save(
	PRIVILEGE *priv)
{
	int ret = 0;
	char vhost[PATH_MAX];
	snprintf (vhost,sizeof(vhost)-1,"www.%s",dict.get_str ("vdomain"));
	return ret;
}

/*
	Locate an HTTPD_DOMAIN record (virtual host) from its vhost name
*/
PUBLIC bool HTTPD_CONFIG::locatevhost(const char *vhost)
{
	SSTRING_IDXS tbs;
	itemss.locatesection (K_VIRTUALHOST,tbs);
	return tbs.lookup (vhost) != -1;
}

PUBLIC int VIRTUAL_COMNG::validate(
	DIALOG &,
	int &nof)
{
	int ret = 0;
	char vhost[PATH_MAX];
	snprintf (vhost,sizeof(vhost)-1,"www.%s",dict.get_str ("vdomain"));
	HTTPD_CONFIG http;
	// Check if this vhost is already there
	if (http.locatevhost (vhost)){
		xconf_error (MSG_U(E_VHOSTEXIST,"Virtual host %s\n"
			"is already configured\n"
			"for the service http"),vhost);
		ret = -1;
	}
	return ret;
}

PUBLIC int VIRTUAL_COMNG::deluser (
	PRIVILEGE *)
{
	return 0;
}

static USERACCT_COMNG *httpd_newcomng(
	const char *key,
	DICTIONARY &dict)
{
	USERACCT_COMNG *ret = NULL;
	if (strcmp(key,"virtual")==0){
		ret = new VIRTUAL_COMNG (dict);
	}
	return ret;
}


static REGISTER_USERACCT_COMNG xxx (httpd_newcomng);


int vhostedit_del (const char *host)
{
	int ret = -1;
	HTTPD_CONFIG http;
	if (http.locatevhost (host)){
		VIEW_SECTION sec (K_VIRTUALHOST,http.itemss);
		sec.setview (host);
		sec.delall();
		ret = http.write();
	}else{
		fprintf (stderr,MSG_U(E_MISSING
			,"Apache: Virtual host %s is not configured\n")
			,host);
	}
	return ret;
}

int vhostedit_add (const char *host, int argc, const char *argv[])
{
	int ret = -1;
	HTTPD_CONFIG http;
	if (http.locatevhost (host)){
		fprintf (stderr,MSG_U(E_HOSTEXIST
			,"Apache: Virtual host %s already configured\n")
			,host);
	}else{
		VIEW_SECTION sec (K_VIRTUALHOST,http.itemss);
		sec.updatehead (host);
		bool err = false;
		const char *documentroot = NULL;
		for (int i=0; i<argc; i++){
			const char *opt = argv[i];
			const char *arg = argv[i+1];
			if (strcmp(opt,"--root")==0){
				VIEWITEM *it = new VIEWITEM("");
				it->line.setfromf ("\t%s %s",K_DOCUMENTROOT,arg);
				sec.add (it);
				documentroot = arg;
				i++;
			}else{
				fprintf (stderr,MSG_U(E_IVLDOPT,"Invalid option %s\n"),opt);
				err = true;
			}
		}
		if (documentroot == NULL){
			fprintf (stderr,MSG_U(E_NOROOT,"Apache: no root specified\n"));
		}else if (!err){
			if (vhostedit_createroot(sec) != -1){
				ret = http.write();
			}else{
				fprintf (stderr,MSG_U(E_CREATEROOT,"Can't create document root directory %s(%s)\n")
					,documentroot,strerror(errno));
			}
		}
	}
	return ret;
}

