/****************************************************************************
**
** $Id: directory.cpp,v 1.72 2001/11/04 13:01:54 joseh Exp $
**
** Copyright (C) 2001 The LinCVS development team.
**    Tilo Riemer <riemer@lincvs.org>
**    Falk Brettschneider <gigafalk@yahoo.com>
**    Wim Delvaux <wim.delvaux@chello.be>
**    Jose Hernandez <joseh@tesco.net>
**    Helmut Koll <HelmutKoll@web.de>
**    Sven Trogisch <trogisch@iapp.de>
**
**
**----------------------------------------------------------------------------
**
**----------------------------------------------------------------------------
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
*****************************************************************************/


#include "config.h"
#include "ac_system_defs.h"

#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

//----------------------------------------------------------------------------

#include <qdir.h>
#include <qfileinfo.h>
#include <qfile.h>
#include <qmultilineedit.h>
#include <qpixmap.h>
#include <qapplication.h>
#include <qtextstream.h>
#include <qdatetime.h>
#include <qpalette.h> 
#include <qstringlist.h>
#include <qmessagebox.h>
#include <qvaluelist.h>
#include <qdialog.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qlabel.h>

//----------------------------------------------------------------------------

#include "CmdThread.h"
#include "cvslistview.h"
#include "noncvslistview.h"
#include "directory.h"
#include "globals.h"
#include "PixmapCache.h"

//----------------------------------------------------------------------------

enum CVSmethod {
  local_method,
  pserver_method,
  kserver_method,
  gserver_method,
  server_method,
  ext_method,
  fork_method
};

/*
 * Parse a CVSROOT variable into its constituent parts -- method,
 * username, hostname, directory.  The prototypical CVSROOT variable
 * looks like:
 *
 * :method:user@host:path
 *
 * Some methods may omit fields; local, for example, doesn't need user
 * and host.
 *
 * Returns false on success, true on failure.
 *
 * This version shamelessly stolen from the CVS sources and subsequently
 * modified in order to fit this program.
 */

static bool parse_cvsroot (const QString& CVSroot,
                           QString& m,	              /* Name of the method */
                           QString& CVSroot_username,  /* the username */
                           QString& CVSroot_hostname,  /* the hostname */
                           QString& CVSroot_directory) /* the directory name */
{
  int index = 0;
  bool check_hostname;
  CVSmethod CVSroot_method;	/* one of the enum values defined in cvs.h  */

  if ((CVSroot [index] == ':'))
  {
    int method_index = ++index;

    /* Access method specified, as in
     * "cvs -d :pserver:user@host:/path",
     * "cvs -d :local:e:\path",
     * "cvs -d :kserver:user@host:/path", or
     * "cvs -d :fork:/path".
     * We need to get past that part of CVSroot before parsing the
     * rest of it.
     */
    
    int method_end = CVSroot.find (':', method_index);
    
    if (method_end < 0)
    {
      // bad CVSroot
      return true;
    }

    m = CVSroot.mid (method_index, method_end - method_index);

    /* Now we have an access method -- see if it's valid. */

    if (m == "local")
      CVSroot_method = local_method;
    else if (m == "pserver")
      CVSroot_method = pserver_method;
    else if (m == "kserver")
      CVSroot_method = kserver_method;
    else if (m == "gserver")
      CVSroot_method = gserver_method;
    else if (m == "server")
      CVSroot_method = server_method;
    else if (m == "ext")
      CVSroot_method = ext_method;
    else if (m == "fork")
      CVSroot_method = fork_method;
    else
    {
      // unknown method in CVSroot
      return true;
    }

    index = ++method_end;			// Point after method
  }
  else
  {
    /* If the method isn't specified, assume
       SERVER_METHOD/EXT_METHOD if the string contains a colon or
       LOCAL_METHOD otherwise.  */
    
    if (CVSroot.find(':', index) > 0) {
#ifdef RSH_NOT_TRANSPARENT
        CVSroot_method = server_method;
	m = "server"; 
#else
        CVSroot_method = ext_method;
	m = "ext"; 
#endif
    } else {
        CVSroot_method = local_method;
	m = "local"; 
    }

/*
    CVSroot_method = ((CVSroot.find (':', index) > 0)
#ifdef RSH_NOT_TRANSPARENT
			? server_method
#else
			? ext_method
#endif
			: local_method);
*/			
  }

  /* Check for username/hostname if we're not LOCAL_METHOD. */

  if ((CVSroot_method != local_method)
      && (CVSroot_method != fork_method))
  {
    /* Check to see if there is a username in the string. */
    int at_index = CVSroot.find ('@', index);

    if (at_index >= 0)
    {
      CVSroot_username = CVSroot.mid (index, at_index - index);
      index = ++at_index;

      int colon_index = CVSroot.find (':', index);

      if (colon_index >= 0)
      {
         CVSroot_hostname = CVSroot.mid (index, colon_index - index);
         index = ++colon_index;
      }
    }
  }

  index = CVSroot.findRev (':');
  if (index < 0) index = CVSroot.length ();
  else index = CVSroot.length () - index - 1;
  CVSroot_directory = CVSroot.right (index);
  
  /* Do various sanity checks. */

  if (CVSroot_username.length() && ! CVSroot_hostname.length())
  {
    // missing hostname in CVSROOT
    return true;
  }

  check_hostname = false;
  switch (CVSroot_method)
  {
  case local_method:
  case fork_method:
    if (CVSroot_username.length() || CVSroot_hostname.length())
    {
      // can't specify hostname and username in CVSROOT
      // when using a local access method
      return true;
    }
    
    break;

    case kserver_method:
#ifndef HAVE_KERBEROS
      // Kerberos not supported
      return true;
#else
      check_hostname = true;
      break;
#endif
    case gserver_method:
#ifndef HAVE_GSSAPI
      // GSSAPI not supported
      return true;
#else
      check_hostname = true;
      break;
#endif

    case server_method:
    case ext_method:
    case pserver_method:
      check_hostname = true;
      break;
    }

  if (check_hostname)
  {
    if (CVSroot_hostname.isEmpty())
    {
      // didn't specify hostname in CVSROOT
      return true;
    }
  }

  if (CVSroot_directory.isEmpty())
  {
    // missing directory in CVSROOT
    return true;
  }
    
  /* Hooray!  We finally parsed it! */
  return false;
}

void Entry::setState( EntryStates newState ) {
	// bubble up
	m_state = newState;
	myDirectory->setState( newState );
}

//----------------------------------------------------------------------------

Directory::Directory( QListViewItem * parent, 
	const QString& fileName, 
	CvsListView * CvsFileListView,
	NonCvsListView * NonCvsFileListView )
   : QListViewItem( parent )
{
   init(fileName, CvsFileListView, NonCvsFileListView );
}

//----------------------------------------------------------------------------

Directory::Directory( QListView * parent, 
	const QString& fileName,
	CvsListView * CvsFileListView,
	NonCvsListView * NonCvsFileListView )
   : QListViewItem( parent )
{
   init(fileName, CvsFileListView, NonCvsFileListView );
}


//----------------------------------------------------------------------------

void Directory::activateItem()
{
   activate();
}

//----------------------------------------------------------------------------

void Directory::init(const QString& fileName, 
	CvsListView * CvsFileListView, 
	NonCvsListView * NonCvsFileListView )
{
	m_haveCvsDir = false;
	m_subDirHasCvsDir = false;
	m_evaluateCommand = 0;
	m_activatingIsNecessary = true;
	m_bSearchCaseSensitive = false;

//   setSelectable(false);
   m_pCvsFileListView = CvsFileListView;
   m_pNonCvsFileListView = NonCvsFileListView;
   m_fullName = fileName;
   m_readable = QDir(m_fullName).isReadable();
   m_entries.setAutoDelete(true);
   m_curCvsState = ES_unknown;

   if(m_fullName.length() == 1) {//root
     setText(0, m_fullName);
   } else {
     QString S = QFileInfo( m_fullName ).fileName();
     if( S[ S.length()-1 ] == '/' ) {
       S.truncate( S.length()-1 );
     }
     setText(0, QFileInfo( m_fullName ).fileName() );
   }   

   // clear directory
   setState( ES_probably_up_to_date );

   if ( getDisabled(m_fullName) ) {
     setPixmap( 0, findEmbeddedPixmap ("FolderDisabled16x16"));
     return;   // dont read it
   }

   if (!m_readable )
     setPixmap( 0, findEmbeddedPixmap ("FolderClosedLocked16x16"));
   else
     setPixmap( 0, findEmbeddedPixmap ("FolderClosed16x16"));
   
   qApp->processEvents(1);
   if(!globalStopAction){
      analyzeDirs();
   }
}


QString Directory::shortName( void ) {

        QString R;
        int pos = m_fullName.findRev( '/' );

        R = m_fullName.mid( pos+1 );
        printf( "%s\n", R.ascii() );
        return R;
}
/*---------------------------------------------------------------------------*/
/*!
\fn			void Directory::paintCell( QPainter *p, const QColorGroup &cg,
                                 int column, int width, int alignment )
\brief		Paints a dedicated directory entry with a different color.

\param		*p		
\param		&cg		
\param		column		
\param		width		
\param		alignment		

<BR><HR>*/
/*---------------------------------------------------------------------------*/

void Directory::paintCell( QPainter *p, const QColorGroup &cg,
                           int column, int width, int alignment )
{
    QColorGroup colGrp( cg );
    QColor c = colGrp.text();
 
    if (getDisabled (m_fullName))
    {
      colGrp.setColor( QColorGroup::Text, Qt::gray );
    }
    
    QListViewItem::paintCell( p, colGrp, column, width, alignment );
    colGrp.setColor( QColorGroup::Text, c );
}                                                                           


//----------------------------------------------------------------------------

void Directory::addDir(QString newDir)
{
   if(m_fullName.compare(newDir) == 0){//I am it ;-)
      analyzeDirs();   //das wird wohl nur klappen, wenns dir leer is
      return;
   }

   Directory *myChild = (Directory*)firstChild();
   while(myChild) {
      if(newDir.find(myChild->fullName()) == 0) {
         myChild->addDir(newDir);
         return;      
      }
      myChild = (Directory*)myChild->nextSibling();
   }

   Directory *item = new Directory(this, newDir, m_pCvsFileListView,
   	m_pNonCvsFileListView );
   if(item->haveCvsDir()) {
      m_subDirHasCvsDir = true;
   }
   else {
      delete item;
   }

   return;
}

//----------------------------------------------------------------------------

bool Directory::analyzeDirs()
{
   resetState();

   m_haveCvsDir = false;
   m_subDirHasCvsDir = false;

   QDir myDir(m_fullName);
   QStringList subDirList = myDir.entryList();
   QString path = m_fullName;
   QString curSubDirName;
   QFileInfo fileInfo;
   Directory *item;

   if(!subDirList.isEmpty()){
      if(m_fullName.length() > 1){//is not root
         path += "/";
      }
   
      for(unsigned int i = 0; i < subDirList.count(); i++){
         if(globalStopAction) break;   //cancel this action

         curSubDirName = subDirList[i];
         fileInfo.setFile(path + curSubDirName);

         if((QString::compare(curSubDirName, ".") == 0) || 
            (QString::compare(curSubDirName, "..") == 0) ||
	    fileInfo.isSymLink())
            continue;   //is the current dir or the parent dir or a symlink

         if(fileInfo.isDir() &&
            fileInfo.permission(QFileInfo::ReadUser | QFileInfo::ExeUser))
         {//is a dir and readable
            globalStatusBar->setText(path + curSubDirName);

            if(QString::compare(curSubDirName, "CVS") == 0){//have CVS dir
               m_haveCvsDir = checkCvsDir();
            }
            else {//is a dir, is readable and is not the CVS dir
//noch testen, ob genug speicher (wenn grosse dirs gelesen werden)   
               item = new Directory(this, path + curSubDirName, 
               		m_pCvsFileListView, m_pNonCvsFileListView );
               if(item->haveCvsDir()) {
                  m_subDirHasCvsDir = true;
               }
               else {
		       delete item;
               }
            }
         }
      }
   }
   return m_haveCvsDir || m_subDirHasCvsDir;
}

//----------------------------------------------------------------------------

bool Directory::checkCvsDir( )
{
   QString path = m_fullName;

   if(path.length() > 1){//is not root
         path += "/";
      }
      
   QFile f;
   QString line;
   
   //read Root
   f.setName(path + "CVS/Root");
   if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 

      line = textStream.readLine();
      f.close();
      if (parse_cvsroot (line,
         m_connectMethod,
         m_userName,
         m_host,
         m_rootDir))
         return false;		// invalid CVSROOT

      if ((m_connectMethod == "local")
        || (m_connectMethod == "fork"))
      { //local
         m_userName = getenv("USER");
         m_host = "localhost";
         m_connectMethod = "local";
      }
   }
   else return false;

   //read Repository
   f.setName(path + "CVS/Repository");
   if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 
      m_repository = textStream.readLine();
      f.close();
   }
   else return false;

   //workaround for old cvs version
   if(m_repository.find(m_rootDir) == 0){//old cvs version detected
      m_repository = m_repository.mid(m_rootDir.length() + 1);
   }
   
      
   return true;
}

//----------------------------------------------------------------------------

void Directory::activate()
{
   m_pCvsFileListView->clear();
   m_pNonCvsFileListView->clear();
   
  // das muss fuer die disabled files deaktiviert werden
  
   if (!getDisabled (m_fullName)) {
     checkStatus();
     m_pNonCvsFileListView->signalState();
   }
}

//----------------------------------------------------------------------------

Directory* Directory::searchDirOfRepository(QString repository)
{
   if(repository.compare(m_rootDir + "/" + m_repository) == 0)
      return this;   //its my repository :-)

   bool isSubDir = false;
   if((m_repository.find(".")) == 0 && (m_repository.length() == 1)) {// . detected
      if(repository.find(m_rootDir + "/") == 0) {//is subdir
         isSubDir = true;
      }
   }
   else {
      if(repository.find(m_rootDir + "/" + m_repository) == 0) {//is subdir
         isSubDir = true;
      }
   }

   if(isSubDir) {//is subdir
      Directory *myChild = (Directory*)firstChild();
      Directory *result;
      while(myChild) {
         if( (result = myChild->searchDirOfRepository(repository)) ) {
            return result;
         }
         else {
            myChild = (Directory*)myChild->nextSibling();
         }
      }
   }
   
   return 0;
}

//----------------------------------------------------------------------------

Directory* Directory::searchDirOfPath(QString path)
{
   if(path.compare(m_fullName) == 0)
      return this;   //its my path :-)

   if(path.find(m_fullName) == 0) {//is subdir
      Directory *myChild = (Directory*)firstChild();
      Directory *result;
      while(myChild) {
         if( (result = myChild->searchDirOfPath(path)) ) {
            return result;
         }
         else {
            myChild = (Directory*)myChild->nextSibling();
         }
      }
   }
   
   return 0;
}

//----------------------------------------------------------------------------

bool Directory::startCvsCall(QString cvsCommand, QMultiLineEdit *output, 
                             int evaluateCommand, 
                             QStringList *updateFileList,
                             int Synchronous )
{
   if(!loginOk()) return false;

   if(evaluateCommand) m_activatingIsNecessary = true;    //commit, update etc.
   else m_activatingIsNecessary = false;    //log, diff etc.

   m_evaluateCommand      = evaluateCommand;   //output dieser kommandos auswerten
   m_curCvsFullPath       = "";
   m_curCvsFileName       = "";
   m_curCvsRepository     = "";
   m_pCurCvsDir           = 0;
   m_pCurCvsLastUpdateDir = 0;
   if(updateFileList) m_curCvsUpdateFileList = *updateFileList;
   else m_curCvsUpdateFileList.clear();
   
   output->clear();   //spaeter konfigurierbar, ob alte ausgaben loeschen

   //wenn irgendwann mal farbig, palette hier setzen
   
   bool usingSsh = false;
   QString command = "";

   QString fullName = m_fullName;
   fullName.replace( QRegExp(" "), "\\ " );
   command += "cd ";
   command += fullName + " && ";

   if( (QString::compare(m_connectMethod, "ext") == 0)
        || (QString::compare(m_connectMethod, "") == 0) )
   {//rsh access
//      command = exportCommand + " CVS_RSH=" + cvsRsh + "&& ";
      if (cvsRsh.find("ssh") == 0) {
         if (!sshAgentIsRunning) startSshAgent();
         usingSsh = true;
         
         if (evaluateCommand == COMMIT) {
            int pos;
            QString tmpFileOrComment, tmpStr;
            pos = cvsCommand.find("commit -F");
            if (pos > 0) {//we use a temp file 
               command += "set-ssh-env.sh \"";
               command += "env CVS_RSH=" + cvsRsh;
               command += " cvs commit -F ";
               tmpStr = cvsCommand.mid(pos + 10);
               pos = tmpStr.find(" ", 1);
               tmpFileOrComment = tmpStr.left(pos);
               command += tmpFileOrComment + " ";
            } else {//we use the option -m
               command += "set-ssh-commit-env.sh \"";
               command += "env CVS_RSH=" + cvsRsh;
               pos = cvsCommand.find("commit -m");
               command += " cvs commit -m \"";
               tmpStr = cvsCommand.mid(pos + 10);
               pos = tmpStr.find("\"", 1);
               tmpFileOrComment = tmpStr.left(pos);
               command += " \"" + tmpFileOrComment + "\" \"";
            }
            
            QString fileName = tmpStr.mid(pos + 1);
            command += fileName + "\"";
            output->append(command + "\n");   //hier gleich kommandoausgabe
            output->setCursorPosition(output->numLines(), 0); 
            //command += " 2>&1\"";  doesnt work
         } else {
            command += "set-ssh-env.sh \"env CVS_RSH=" 
                      + cvsRsh + " cvs " + cvsCommand + "\"";
            output->append(command + "\n");   //hier gleich kommandoausgabe
            output->setCursorPosition(output->numLines(), 0); 
            //command += " 2>&1\"";   doesnt work
         }
      } else command += "env CVS_RSH=" + cvsRsh + " ";
   }
   
   if (!usingSsh) {
      command += "cvs ";
      command += cvsCommand;
      output->append(command + "\n");   //hier gleich kommandoausgabe
      output->setCursorPosition(output->numLines(), 0); 
      command += " 2>&1";
   }
   
   m_pThread = new CmdThread(command);
   m_pThread->start();

   if( Synchronous ) {
     // wait for thread to finish
     m_pThread->wait();
   }
   
   return true;
}

//----------------------------------------------------------------------------

bool Directory::checkCvsCallResult(QMultiLineEdit *output)
{
   bool res;
   EntryStates m_locCurCvsState = ES_unknown ;

   //laeuft der thread noch?
   if (m_pThread->finished()) res = false;
   else res = true;

   unsigned int len = m_pThread->m_outputQueue.count();
   unsigned int i;
   int pos;
   QString tmp, fullPath, path, repository;
   fullPath = "";
   
   bool bAddAllLines = true;

   if ((m_evaluateCommand == ANNOTATE_GREP_LINE) || (m_evaluateCommand == ANNOTATE_GREP_LINES)) {
     bAddAllLines = false;
   }

   if (len) {//nicht leer
      for(i = 0; i < len; i++) {
         //hier, wenn noetig, ausgabe analysieren
         switch(m_evaluateCommand)
         {
            case COMMIT:
               if(m_pThread->m_outputQueue[i].find("Checking in ") == 0) {//followed of filename
                  m_curCvsFileName = m_pThread->m_outputQueue[i].mid(12, m_pThread->m_outputQueue[i].length() - 13);   //truncate ";"
                  pos = m_curCvsFileName.findRev("/");
                  if (pos > -1) {
                     fullPath = m_fullName + "/" + m_curCvsFileName.mid(0, pos);
                     removeDoubleSlashes(fullPath);
                     m_curCvsFileName = m_curCvsFileName.mid(pos + 1);
                     if(fullPath.compare(m_curCvsFullPath)) {
                        //zugehoeriges dir finden
                        m_pCurCvsDir = searchDirOfPath(fullPath);
                        m_curCvsFullPath = fullPath;
                     }
                  }
                  else {
                     m_pCurCvsDir = this;   //no subdir
                     fullPath = "";
                  }
               }
               else
               if(m_pThread->m_outputQueue[i].compare("done") == 0){//commit completed successfully
                  if(m_pCurCvsDir) 
                  	m_pCurCvsDir->setEntry(m_curCvsFileName, ES_up_to_date);
               }
               break;
            case STATUS:
               //jetzt ausgabezeile analysieren
               if(m_pThread->m_outputQueue[i].find("File:") == 0){//status output of next file begins
                  m_curCvsFileName = m_pThread->m_outputQueue[i].mid(6, m_pThread->m_outputQueue[i].find("\t", 6) - 6);
                  m_curCvsFileName = m_curCvsFileName.stripWhiteSpace();

                  if(m_curCvsFileName.find("no File") == 0) {
                     m_curCvsFileName = m_curCvsFileName.mid(8);
                  }
      
                  if(m_pThread->m_outputQueue[i].find("Up-to-date") > -1){
                     m_locCurCvsState = ES_up_to_date;
                  } else if(m_pThread->m_outputQueue[i].find("Locally Modified") > -1){
                     m_locCurCvsState = ES_modified;
                  } else if(m_pThread->m_outputQueue[i].find("Needs Patch") > -1){
                     m_locCurCvsState = ES_needs_patch;
                  } else if(m_pThread->m_outputQueue[i].find("Needs Merge") > -1){
                     m_locCurCvsState = ES_needs_merge;
                  } else if(m_pThread->m_outputQueue[i].find("Needs Checkout") > -1){
                     m_locCurCvsState = ES_needs_checkout;
                  } else if(m_pThread->m_outputQueue[i].find("File had conflicts on merge") > -1){
                     m_locCurCvsState = ES_conflict;
                  } else if(m_pThread->m_outputQueue[i].find("Locally Added") > -1){
                     m_locCurCvsState = ES_added;
                  } else if(m_pThread->m_outputQueue[i].find("Locally Removed") > -1){
                     m_locCurCvsState = ES_removed;
                  }
               } else if(m_pThread->m_outputQueue[i].find("Repository revision:") > -1) {//repository of fileName
                  //repository extrahieren
                  if((m_repository.find(".")) == 0 && (m_repository.length() == 1)) {// . detected
                     pos = m_pThread->m_outputQueue[i].find(m_rootDir);
                  }
                  else {
                     pos = m_pThread->m_outputQueue[i].find(m_rootDir + "/" + m_repository);
                  }
         
                  repository = m_pThread->m_outputQueue[i].mid(pos, m_pThread->m_outputQueue[i].findRev("/") - pos);

                  if(repository.compare(m_curCvsRepository)) {
                     //zugehoeriges dir finden
                     m_pCurCvsDir = searchDirOfRepository(repository);
                     m_curCvsRepository = repository;
                  }

                  //status eintragen
                  if(m_pCurCvsDir) m_pCurCvsDir->setAndAppendEntry(m_curCvsFileName, m_locCurCvsState);
               }
               break; 
            case UPDATE_DIR:
               m_locCurCvsState = ES_unknown;

               //jetzt ausgabezeile analysieren
               //updating of dir ... welches verzeichnis?
               if((pos = m_pThread->m_outputQueue[i].find("Updating ")) > -1) {
                  //cvs ist in ein neues verzeichnis gewechselt, altes verzeichnis auf uptodate setzen
                  if (m_pCurCvsLastUpdateDir) m_pCurCvsLastUpdateDir->setFilesToUpToDate(m_curCvsUpdateFileList);
    
                  path = m_pThread->m_outputQueue[i].mid(pos + 9);
                  path = path.stripWhiteSpace();

                  //is a subdir?
                  if (path.compare(".") == 0) fullPath = m_fullName;
                  else fullPath = m_fullName + "/" + path;
                  removeDoubleSlashes(fullPath);

                  //neues dir finden
                  m_pCurCvsLastUpdateDir = searchDirOfPath(fullPath);
               
                  //neue fileliste zusammenstellen
                  m_curCvsUpdateFileList.clear();   
                  if (m_pCurCvsLastUpdateDir) m_pCurCvsLastUpdateDir->fillFileListWithEntries(m_curCvsUpdateFileList);
               }
               
               //dateistatus...
               if((m_pThread->m_outputQueue[i].find("P ") == 0) 
                  || (m_pThread->m_outputQueue[i].find("U ") == 0)){
                  m_locCurCvsState = ES_up_to_date;
               } else if(m_pThread->m_outputQueue[i].find("M ") == 0){
                  m_locCurCvsState = ES_modified;
               } else if(m_pThread->m_outputQueue[i].find("C ") == 0){
                  m_locCurCvsState = ES_conflict;
               } else if(m_pThread->m_outputQueue[i].find("A ") == 0){
                  m_locCurCvsState = ES_added;
               } else if(m_pThread->m_outputQueue[i].find("R ") == 0){
                  m_locCurCvsState = ES_removed;
               }
      
               if(m_locCurCvsState != ES_unknown) {
                  m_curCvsFileName = m_pThread->m_outputQueue[i].mid(2);
                  m_curCvsFileName = m_curCvsFileName.stripWhiteSpace();

                  //is file located in subdir?
                  if((pos = m_curCvsFileName.findRev("/")) > -1) 
                  	m_curCvsFileName = m_curCvsFileName.mid(pos + 1);

                  if(m_pCurCvsLastUpdateDir) {
                     m_pCurCvsLastUpdateDir->setAndAppendEntry(m_curCvsFileName, m_locCurCvsState);
                     m_curCvsUpdateFileList.remove(m_curCvsFileName);
                  }
               }      
               break;
            case UPDATE_FILES:
               m_locCurCvsState = ES_unknown;

               //network unreachable? --> clear update file list
               if(m_pThread->m_outputQueue[i].find("update aborted") > -1) m_curCvsUpdateFileList.clear();
               
               //dateistatus...
               if((m_pThread->m_outputQueue[i].find("P ") == 0) 
                  || (m_pThread->m_outputQueue[i].find("U ") == 0)){
                  m_locCurCvsState = ES_up_to_date;
               } else if(m_pThread->m_outputQueue[i].find("M ") == 0){
                  m_locCurCvsState = ES_modified;
               } else if(m_pThread->m_outputQueue[i].find("C ") == 0){
                  m_locCurCvsState = ES_conflict;
               } else if(m_pThread->m_outputQueue[i].find("A ") == 0){
                  m_locCurCvsState = ES_added;
               } else if(m_pThread->m_outputQueue[i].find("R ") == 0){
                  m_locCurCvsState = ES_removed;
               }
               
               if(m_locCurCvsState != ES_unknown ) {
                  m_curCvsFileName = m_pThread->m_outputQueue[i].mid(2);
                  m_curCvsFileName = m_curCvsFileName.stripWhiteSpace();
                  setEntry(m_curCvsFileName, m_locCurCvsState);
                  m_curCvsUpdateFileList.remove(m_curCvsFileName);
               }
               
               break;
	    case ANNOTATE_GREP_LINE:
               if (m_pThread->m_outputQueue[i].find("Annotations for ") == 0) {
		  tmp += "\n" + m_pThread->m_outputQueue[i] + "\n";
	       } else if (m_pThread->m_outputQueue[i].find("*************") == 0) {
		  tmp += m_pThread->m_outputQueue[i] + "\n";
	       } else if (m_pThread->m_outputQueue[i].find(m_annotateGrepSearchLine, 0, m_bSearchCaseSensitive) > -1) {
		  tmp += m_pThread->m_outputQueue[i] + "\n";
	       }
	       break;
	    case ANNOTATE_GREP_LINES:
               if (m_pThread->m_outputQueue[i].find("Annotations for ") == 0) {
		  tmp += "\n" + m_pThread->m_outputQueue[i] + "\n";
	       } else if (m_pThread->m_outputQueue[i].find("*************") == 0) {
		  tmp += m_pThread->m_outputQueue[i] + "\n";
	       } else {
		 int lineNb;
		 for (lineNb = 0; lineNb < m_annotateGrepSearchLines.count(); lineNb++) {
		   if (m_pThread->m_outputQueue[i].find(m_annotateGrepSearchLines[lineNb], 0, m_bSearchCaseSensitive) > -1) {
		     tmp += m_pThread->m_outputQueue[i] + "\n";
		     break;
		   }
		 }
	       }
	       break;
         }

         if (bAddAllLines) tmp += m_pThread->m_outputQueue[i] + "\n";
      }

      tmp.truncate(tmp.length() - 1);   //cut last newline
      output->append(tmp);   //append adds a "\n"
      output->setCursorPosition(output->numLines(), 0); 

      QStringList::Iterator it = m_pThread->m_outputQueue.begin();

      m_pThread->lockOutputQueue ();
      for (i = 0; i < len; i++) it = m_pThread->m_outputQueue.remove(it);
      m_pThread->unlockOutputQueue ();      
   }

   return res;
}

//----------------------------------------------------------------------------

void Directory::recResetState( ) {

     Directory * D;
     const char * PMName;
     m_curCvsState = ES_probably_up_to_date;

     PMName = (!m_readable ) ? "FolderClosedLocked16x16" : "FolderClosed16x16";
     setPixmap( 0, findEmbeddedPixmap (PMName));

     D = (Directory *)firstChild();
     while( D ) {
         D->recResetState();
         D = (Directory *)D->nextSibling();
     }
}

//----------------------------------------------------------------------------

void Directory::resetState( ) {
     // only reset root directory
     topDir()->recResetState();
}

Directory * Directory::topDir() {
     // bubble up to root directory
     QListViewItem * D = this;
     while( D && D->depth() > 0 ) {
       D = D->parent();
     }
     return (Directory *)D;
}

//----------------------------------------------------------------------------

void Directory::setState( EntryStates newState ) {

   if( newState > m_curCvsState ) {
     // forced or more severe
     const char * PMName;

     m_curCvsState = newState;

     if( m_curCvsState <= ES_up_to_date ) {
       // all is OK
       PMName = (!m_readable ) ? "FolderClosedLocked16x16" : "FolderClosed16x16";
     } else if( m_curCvsState < ES_conflict ) {
       PMName = (!m_readable ) ? 
                     "FolderClosedLockedWarn16x16" :
                     "FolderClosedWarn16x16";
     } else {
       PMName = (!m_readable ) ? 
                     "FolderClosedLockedProblem16x16" : 
                     "FolderClosedProblem16x16";
     }

     setPixmap( 0, findEmbeddedPixmap (PMName));

     if( depth() > 0 ) {
       // non root -> has directory parent
       ((Directory *)parent())->setState( newState );
     }
   }
}

//----------------------------------------------------------------------------

void Directory::killCvsThread()
{
	m_pThread->exit();
}

//----------------------------------------------------------------------------

void Directory::deleteThreadAndActivateIfNecessary()
{
   delete m_pThread;
   m_pThread = 0;
   
   //wenn updatefileliste nicht leer, inhalt auf uptodate setzen
   if (m_evaluateCommand == UPDATE_FILES) setFilesToUpToDate(m_curCvsUpdateFileList);
   else if (m_evaluateCommand == UPDATE_DIR) {
      if (m_pCurCvsLastUpdateDir) m_pCurCvsLastUpdateDir->setFilesToUpToDate(m_curCvsUpdateFileList);
   }

   if(m_activatingIsNecessary) activate();
}

//----------------------------------------------------------------------------

void Directory::addFiles(QMultiLineEdit *output, QStringList *stringList, bool bAsBinary)
{

	if(!loginOk()) return;

	char buf[512];

	output->clear();  

	qApp->mainWidget()->setCursor(waitCursor);
	QPalette pal = output->palette();
	pal.setColor(QColorGroup::Text, QColor(0, 0, 0));   //default: black
	output->setPalette(pal);

	QString BinaryFlag;      
	QString EndSshString;

	if (bAsBinary) BinaryFlag = "-kb ";

	bool usingSsh = false;

	QString command = "";

	command += "cd ";
	{ QString Tmp =m_fullName;
	  Tmp = Tmp.replace( QRegExp(" "), "\\ " );
	  command += Tmp + " && ";
	}

	if( (QString::compare(m_connectMethod, "ext") == 0)
	   || (QString::compare(m_connectMethod, "") == 0) ) {
	  //rsh access
	  //         command = exportCommand + " CVS_RSH=" + cvsRsh + "&& ";
	 if (cvsRsh.find("ssh") == 0) {

	    if (!sshAgentIsRunning) 
	      startSshAgent();

	    usingSsh = true;
	 
	    command += "set-ssh-env.sh \"env CVS_RSH=" 
		     + cvsRsh + " cvs add " + BinaryFlag;

	    EndSshString =  "\"";

	 } else 
	   command += "env CVS_RSH=" + cvsRsh + " ";
	}

        command += "cvs add " + BinaryFlag;

	for(unsigned int i = 0; i < stringList->count(); i++) {
	  QString fileName = (*stringList)[i];
	  fileName = fileName.stripWhiteSpace();
	  QFileInfo info(fileName);

	  if(!info.exists()) 
	    continue;

	  command += info.fileName().replace( QRegExp(" "), "\\ " ) + " ";
	}

        if( ! EndSshString.isEmpty() ) 
          command += EndSshString;

         output->append(command + "\n");
         output->setCursorPosition(output->numLines(), 0);

        if( ! usingSsh ) {
         command += " 2>&1";
        }

	// execute command in 1 step
	FILE* cvs = popen(command, "r");
	if(!cvs){
	   perror("popen");
	   return;
	}

	while (fgets(buf, sizeof buf, cvs)) {
	   QString str = buf;
	   str.truncate(str.find("\n"));

	   for(unsigned int i = 0; i < stringList->count(); i++) {
	      QString fileName = (*stringList)[i];
	      fileName = fileName.stripWhiteSpace();
	      fileName = fileName.replace( QRegExp(" "), "\\ " );
	      QString MsgToSearch = 
		  "scheduling file `" + fileName + "' for addition";

	      if(str.find(MsgToSearch) > 0){
		fileName = (*stringList)[i];
		fileName = fileName.stripWhiteSpace();
		setAndAppendEntry(fileName, ES_added);
	      }
	   }

	   output->append(str);
	   output->setCursorPosition(output->numLines(), 0);
	}

	pclose(cvs);

	output->append("\n");

	activate();
	qApp->mainWidget()->setCursor(arrowCursor);
	output->setCursorPosition(0, 0); 
}

//----------------------------------------------------------------------------

void Directory::setFilesToUpToDate(QStringList& updateFileList)
{
   QString tmp;
   while (!updateFileList.isEmpty()) {
      tmp = updateFileList.first();
      setEntry(tmp, ES_up_to_date);
      updateFileList.remove(tmp);
   }
}

//----------------------------------------------------------------------------

void Directory::fillFileListWithEntries(QStringList& fileList)
{
   QFile f;
   QString line, name;
   int pos;
   
   //read Entries
   f.setName(m_fullName + "/" + "CVS/Entries");
   //QFileInfo fInfo(f);
   
   if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 

      while(!textStream.atEnd()) {
         line = textStream.readLine();
         if(line.find("D") == 0) continue;   //entry is a directory

         if(line.find("/") == 0) {
            if((pos = line.find("/", 1)) > 1) {//entry seems to be correct
               name = line.mid(1, pos - 1);
               fileList.append(name);
               setAndAppendEntry(name, ES_probably_up_to_date);            
            }   
         }
      }
      f.close();
   }
}

//----------------------------------------------------------------------------

void Directory::setAllToUpToDate()
{
   //alle entries im eignen verzeichnis auf uptodate setzen
   QString path = m_fullName;
   if(path.length() > 1){//is not root
      path += "/";
   }

   QFile f;
   QString line, name;
   int posLeft;
   
   //read Entries
   f.setName(path + "CVS/Entries");
   QFileInfo fInfo(f);
   m_modTimeOfEntries = fInfo.lastModified().toString().stripWhiteSpace();
   
   if(f.open(IO_ReadOnly)) {//file is readable
      QTextStream textStream(&f); 

      while(!textStream.atEnd()) {
         line = textStream.readLine();
         if(line.find("D") == 0) continue;   //entry is a directory

         if(line.find("/") == 0) {//entry seems to be correct
            if((posLeft = line.find("/", 1)) > 1) {
               name = line.mid(1, posLeft - 1);
            }   
            
            setAndAppendEntry(name, ES_up_to_date);            
         }
      }
      f.close();
   }

   //jetzt die subdirs...
   Directory *myChild = (Directory*)firstChild();
   while(myChild) {
      myChild->setAllToUpToDate();
      myChild = (Directory*)myChild->nextSibling();
   }
}

//----------------------------------------------------------------------------

bool Directory::setEntry(QString fileName, EntryStates stateId)
{
   bool foundEntry = false;
   Entry *entry = m_entries.first();
   unsigned int i;
   for(i = 0; (i < m_entries.count()) && !foundEntry; i++) {
      if(entry->fileName().compare(fileName) == 0) {
         //status auslesen und umsetzen
         entry->setState(stateId);
         i = m_entries.count();
         foundEntry = true;
      }
      entry = m_entries.next();
   }
   
   return foundEntry;
}

//----------------------------------------------------------------------------

void Directory::setAndAppendEntry(QString fileName, EntryStates stateId)
{
   if(!setEntry(fileName, stateId)) {
      m_entries.append(new Entry(fileName, stateId, this ));
   }
}

//----------------------------------------------------------------------------

EntryStates Directory::alignWithEntries(QString name, EntryStates stateId)
{
   bool isCached = false;
   //is item cached?

   Entry *entry = m_entries.first();
   while(entry && !isCached) {
      if(entry->fileName().compare(name) == 0) {//datei wurde schon erfasst
         //status auslesen und umsetzen
         switch(stateId)
         {
            case ES_modified:
               if((entry->state() == ES_needs_patch)
                  || (entry->state() == ES_needs_merge)) {
                  entry->setState(ES_needs_merge);
                  stateId = ES_needs_merge;
               }
               else{
                  if(entry->state() == ES_added) {
                     stateId = ES_added;
                  }
                  else
                     entry->setState(ES_modified);
               }
               break;
            case ES_missing:
               if(entry->state() == ES_needs_checkout)
                  stateId = ES_needs_checkout;
               break;
            case ES_probably_up_to_date:
               if((entry->state() == ES_added) || (entry->state() == ES_modified)) {
                  entry->setState(ES_probably_up_to_date);
               }
            case ES_probably_up_to_date_and_timezone_incorrect:
               if((entry->state() == ES_added) || (entry->state() == ES_modified)) {
                  entry->setState(ES_probably_up_to_date);
               }
            default:
               stateId = entry->state();
               break;
         }            

         isCached = true;
      }
      entry = m_entries.next();
   }
   
   if(!isCached) {
      m_entries.append(new Entry(name, stateId, this));
   }
   
   return stateId;
}

//----------------------------------------------------------------------------

void Directory::checkStatus(QListViewItem *item) {

	QString path = m_fullName;

	if(path.length() > 1){//is not root
	  path += "/";
	}

	QFile f;
	QString line;

	//read Entries
	f.setName(path + "CVS/Entries");
	if(f.open(IO_ReadOnly)) {//file is readable
	  QTextStream textStream(&f); 

	  if(item) {//only one item
	    bool foundFile = false;
	    QString fileName = item->text(0).stripWhiteSpace();
	    while(!textStream.atEnd() && !foundFile) {
	      line = textStream.readLine();
	      if(line.find("D") == 0) continue;   //entry is a directory

	      if(line.find("/" + fileName + "/") == 0) {
		 foundFile = true;
		 setStatusInFileListView(item, line);
	      }
	    }
	  } else {//all entries
	    QFileInfo fInfo(f);

	    // read entries in directory
	    QStringList AllFilesInDirectory;

	    QDir D( path );

	    AllFilesInDirectory = D.entryList();

	    // remove . and ..
	    AllFilesInDirectory.remove( "." );
	    AllFilesInDirectory.remove( ".." );
	    AllFilesInDirectory.remove( "CVS" );
	    
	    m_modTimeOfEntries = fInfo.lastModified().toString().simplifyWhiteSpace();
	    while(!textStream.atEnd()) {

	      line = textStream.readLine();

	      if(line.find("D") == 0) {
		continue;   //entry is a directory
	      }

	      if(line.find("/") == 0) {//entry seems to be correct
		item = setStatusInFileListView(0, line);
		// remove item from directory
		AllFilesInDirectory.remove(item->text(0));
	      }      
	    }

	    // show non-cvs entries

	    // first all directories

	    int IsDir;
	    for ( QStringList::Iterator it = AllFilesInDirectory.begin(); 
			  it != AllFilesInDirectory.end(); ++it ) {
	      QFileInfo info( path + (*it) );

	      IsDir = 0;

	      if( info.isDir() ) {
		// later: do not show any dir         
		QFile F( path + (*it) + "/CVS/Root" );
		if( F.exists() ) {
		  // CVS directory -> do NOT show
		  continue;
		}
		IsDir = 1;
	      }

	      item = new QListViewItem( m_pNonCvsFileListView );
	      item->setText( 0, (*it) );
              if( IsDir ) {
		// show pixmap that it is a directory
		item->setPixmap (0, findEmbeddedPixmap ("FolderClosed16x16"));
              } else {
		item->setPixmap (0, findEmbeddedPixmap ("FileUnknown16x16"));
	      }

	      QDateTime localDate = info.lastModified();
              item->setText( 1, localDate.toString() );
	      localDate = localDate.addSecs(timeZoneDiffInSecs);
	      item->setText( 2, localDate.toString() );
	    }
	  }
	  f.close();
	}
}

//----------------------------------------------------------------------------

QListViewItem * Directory::setStatusInFileListView(QListViewItem *item, QString line)
{
   EntryStates stateId = ES_unknown;
   QString name, rev, date, state, modDateLocal, modDateUtc;
   int posLeft, posRight;
   bool sticky = false;   //wenn gesetzt, dann true
   char *pix;
   QDateTime localDate;
   QString localDateString;

   QString path = m_fullName;
   if(path.length() > 1){//is not root
      path += "/";
   }

   if((posLeft = line.find("/", 1)) > 1) {
      name = line.mid(1, posLeft - 1);
      if((posRight = line.find("/", posLeft + 1)) > 1) {
         rev  = line.mid(posLeft + 1, posRight - 1 - posLeft);
         posLeft = posRight;
         if((posRight = line.find("/", posLeft + 1)) > 1) {
            date = line.mid(posLeft + 1, posRight - 1 - posLeft);

            if(line.findRev("/") < line.findRev("T")) sticky = true;   //sticky tag
            if(line.findRev("/") < line.findRev("D")) sticky = true;   //sticky date

            //status?
            QFileInfo info(path + name);
            if(info.exists()){
               localDate = info.lastModified();
               localDate = localDate.addSecs(timeZoneDiffInSecs);
               localDateString = localDate.toString().stripWhiteSpace();
               if(localDateString.compare(date.stripWhiteSpace()) == 0){
                  stateId = ES_probably_up_to_date;
               }
               else {
                  if(date.find("dummy timestamp") > -1) {
                     stateId = ES_added;
                  }
                  else {
                     QString localMinSecs = localDateString.mid(localDateString.find(":"), 6);
                     QString reposMinSecs = date.mid(date.find(":"), 6);
               
                     if(localMinSecs.compare(reposMinSecs) == 0){
                        stateId = ES_probably_up_to_date_and_timezone_incorrect;
                        //jetzt warnung ausgeben
                     } else stateId = ES_modified;
                  }
               }
            } else stateId = ES_missing;
               
            stateId = alignWithEntries(name, stateId);
            
            modDateLocal = "";
            modDateUtc = "";
                     
            switch(stateId)
            {
               case ES_probably_up_to_date:
                  state = QObject::tr("seems up to date") + "  ";   //
                  pix = getUnchangedPix(info);
                  break;
               case ES_probably_up_to_date_and_timezone_incorrect:
                  state = QObject::tr("seems up to date") + "  ";   //
                  //pix = &pixUnchangedTimezoneConflict;
                  pix = getUnchangedPix(info);
                  break;
               case ES_up_to_date:
                  state = QObject::tr("up to date") + "  ";   //
                  pix = getUnchangedPix(info);
                  break;
               case ES_modified:
                  state = QObject::tr("modified") + "  ";   //
                  pix = "FileModified16x16";

                  //doppelten aufruf einsparen!
                  localDate = info.lastModified();
                  modDateLocal = localDate.toString();
                  localDate = localDate.addSecs(timeZoneDiffInSecs);
                  modDateUtc = localDate.toString();
                  break;
               case ES_needs_patch:
                  state = QObject::tr("needs patch") + "  ";   //
                  pix = "FileNeedsPatch16x16";
                  break;
               case ES_needs_merge:
                  state = QObject::tr("needs merge") + "  ";   //
                  pix = "FileNeedsMerge16x16";

                  //doppelten aufruf einsparen!
                  localDate = info.lastModified();
                  modDateLocal = localDate.toString();
                  localDate = localDate.addSecs(timeZoneDiffInSecs);
                  modDateUtc = localDate.toString();
                  break;
               case ES_needs_checkout:
                  state = QObject::tr("needs checkout") + "  ";   //
                  pix = "FileNeedsCheckout16x16";
                  break;
               case ES_missing:
                  state = QObject::tr("missing") + "  ";   //
                  pix = "FileRemoved16x16";
                  break;
               case ES_conflict:
                  state = QObject::tr("conflict") + "  ";   //
                  pix = "FileConflict16x16";
                  break;
               case ES_added:
               //welche zeit anzeigen???
                  state = QObject::tr("added") + "  ";   //
                  pix = "FileAdded16x16";
                  break;
               case ES_removed:
               //welche zeit anzeigen???
                  state = QObject::tr("removed") + "  ";   //
                  pix = "FileRemoved16x16";
                  break;
               default:
                  state = QObject::tr("unknown") + "  ";
                  pix = getUnchangedPix(info);
            }
            
/* to strange ;-)
            if (!StatTimerEnable) { // if timer is disabled put ? in front
               pix = "FileRemoved16x16";    // sounds strange but matches well
            }
*/
            if( item == 0 ) {
               item = new QListViewItem(m_pCvsFileListView );
            // double code see hereafter
            //} else {
        //if (sticky) 
        //  item->setPixmap(2, findEmbeddedPixmap ("Tag16x16"));
        //else 
        //  item->setPixmap(2, 0);
            }

	    item->setText(0, name ); //+ "  "); 
	    item->setText(1, rev ); //+ "  "); 
	    item->setText(3, state); 
	    item->setText(4, date ); //+ "  "); 
	    item->setText(5, modDateLocal ); //+ "  "); 
	    item->setText(6, modDateUtc ); //+ "  "); 

            item->setPixmap(0, findEmbeddedPixmap (pix));
            if (sticky)
               item->setPixmap(2, findEmbeddedPixmap ("Tag16x16"));
            else 
               item->setPixmap(2, 0);
         }
      }
   }
   return item;
}

//----------------------------------------------------------------------------

char * Directory::getUnchangedPix(QFileInfo & info)                             
{                                                                               
   if(info.isWritable())                                                        
   {                                                                            
      return "FileWriteable16x16";                                            
   }                                                                            
   else                                                                         
   {                                                                            
      return "FileUnchanged16x16";
   }                                                                            
}                                                                               

//----------------------------------------------------------------------------

bool Directory::entriesFileModified()
{
   QString path = m_fullName;
   if(path.length() > 1){//is not root
      path += "/";
   }

   QFileInfo fInfo(path + "CVS/Entries");
   if (!fInfo.exists()) return false;
      
   if (m_modTimeOfEntries.compare(fInfo.lastModified().toString().stripWhiteSpace()) == 0) {
      return false;
   } else return true;
}

//----------------------------------------------------------------------------

bool Directory::loginOk(bool showMessage)
{
   if(!m_haveCvsDir) {
      QMessageBox::warning(0, QObject::tr("Warning"), QObject::tr("Is not a CVS directory."), 0);
      return false;
   }

   if(m_connectMethod.find("pserver") == -1) {
      if(showMessage) {
         QMessageBox::information(0, QObject::tr("Information"), 
                                  QObject::tr("No login necessary. Its a local CVS repository"), 0);
      }

      return true;
   }

	
	QString cvsRoot = ":" + m_connectMethod + ":" + m_userName + 
                     "@" + m_host + ":" + m_rootDir;

   if(!isInCvsPass(cvsRoot)){
      login(cvsRoot);
      qApp->processEvents(1);
      sleep(1);
      qApp->processEvents(1);
            
      if(!isInCvsPass(cvsRoot)) {
         QMessageBox::warning(qApp->mainWidget(), 
                              QObject::tr("Warning"), QObject::tr("Login failed."), 0);
         return false;
      }

      if(showMessage) {
         QMessageBox::information(qApp->mainWidget(), 
                                  QObject::tr("Information"), QObject::tr("Login successfully."), 0);
      }
      
      return true;
   }
   
   if(showMessage) {
      QMessageBox::information(qApp->mainWidget(), 
                               QObject::tr("Information"), QObject::tr("You are already logged in."), 0);
   }
      
   return true;
}

//----------------------------------------------------------------------------
                      




/*---------------------------------------------------------------------------*/
/*!
\fn			bool Directory::getDisabled(QString filename)
\brief		Searches for disabled directories. (they are currently not scanned)

\param		filename		

\return		TRUE if filename matches any name in m_projectDisabledList <br>
          FALSE otherwise
<BR><HR>*/
/*---------------------------------------------------------------------------*/

bool Directory::getDisabled(QString filename)
{
  for ( QStringList::Iterator it = m_projectDisabledList.begin(); 
        it != m_projectDisabledList.end(); ++it ) {
    
//    if (filename == (*it).latin1()) {
    if (filename == (*it)) {
//      printf( "getDisabled found : %s \n", (*it).latin1() );
      return (TRUE);
    }
  }
  return (FALSE);
}

void Directory::setAllOpen( ) {

        Directory * D = this;
        do {
          D->setOpen( TRUE );
          if( D->depth() == 0 ) 
            // root item 
            break;
          D = (Directory *)D->parent();
        } while ( 1 );
}

int Directory::isTreeEmpty() {

	QString path = m_fullName;

	if(path.length() > 1){//is not root
	  path += "/";
	}

	QFile f;

	//read Entries
	f.setName(path + "CVS/Entries");
	if(f.open(IO_ReadOnly)) {//file is readable
	  QString line;
	  QTextStream textStream(&f); 
	  QFileInfo fInfo(f);
	  QDir D( path );

	  while(!textStream.atEnd()) {

	    line = textStream.readLine();

	    if(line.find("D") == 0) {
	      continue;   //entry is a directory
	    }

	    if(line.find("/") == 0) {//entry seems to be correct
	      // entry found -> not empty
	      return 0;
	    }      
	  }
	  f.close();
	}

        // seems to contain NO regular CVS files

        Directory * C = (Directory *)firstChild();
        while( C ) {
          if( ! C->isTreeEmpty() ) 
            return 0;
          C = (Directory *)C->nextSibling();
        }
        // no entry found
        return 1;
}

void Directory::removeFromRepAndDisk( void ) {
        // check if dir is empty --> security feature
        if( ! isTreeEmpty() ) {
	  QMessageBox::critical(0, QObject::tr("Error"), 
	        QObject::tr("Directory (or tree) is not empty"), 0);
	  return ;
	}

	// seems to be empty -> remove tree items
	doRemoveFromRepOrDisk( 1, 1 );

	// remove this directory
	doRemoveThis( 1, 1 );
}

void Directory::removeFromRep( void ) {

        // check if dir is empty --> security feature
        if( ! isTreeEmpty() ) {
	  QMessageBox::critical(0, QObject::tr("Error"), 
	        QObject::tr("Directory (or tree) is not empty"), 0);
	  return ;
	}

	// seems to be empty -> remove tree items
	doRemoveFromRepOrDisk( 1, 0 );

	// remove this directory
	doRemoveThis( 1, 0 );
}

void Directory::doRemoveThis( int FromRep, int FromDisk ) {

        if( FromRep ) {
	  QString command = "cd ";
	  QString EndSshString;

	  { QString Tmp = ((Directory *)parent())->m_fullName;
	    Tmp = Tmp.replace( QRegExp(" "), "\\ " );
	    command += Tmp + " && ";
	  }

	  if( (QString::compare(m_connectMethod, "ext") == 0)
	     || (QString::compare(m_connectMethod, "") == 0) ) {
	    // rsh access
	    // command = exportCommand + " CVS_RSH=" + cvsRsh + "&& ";
	   if (cvsRsh.find("ssh") == 0) {

	      if (!sshAgentIsRunning) 
		startSshAgent();

	      command += "set-ssh-env.sh \"env CVS_RSH=" 
		       + cvsRsh + " cvs add ";

	      EndSshString =  "\"";

	   } else 
	     command += "env CVS_RSH=" + cvsRsh + " ";
	  }

	  command += "cvs remove";

	  command += " " + shortName();

          if( ! EndSshString.isEmpty() )
	      EndSshString =  "\"";

	  FILE* cvs = popen(command, "r");
	  if( cvs ) fclose( cvs );
        }

        if( FromDisk ) {
	  QString CVSD;
	  CVSD = m_fullName + "/CVS/Entries";
	  unlink( CVSD.ascii() );
	  CVSD = m_fullName + "/CVS/Repository";
	  unlink( CVSD.ascii() );
	  CVSD = m_fullName + "/CVS/Root";
	  unlink( CVSD.ascii() );
	  CVSD = m_fullName + "/CVS";
	  rmdir( CVSD.ascii() );

	  if( rmdir( m_fullName.ascii() ) < 0 ) {
	    perror( "Cannot unlink" );
	  }
        }
}

void Directory::doRemoveFromRepOrDisk( int FromRep, int FromDisk ) {

        // go deeper first

        Directory * C = (Directory *)firstChild();
        while( C ) {
          C->doRemoveFromRepOrDisk( FromRep, FromDisk );
          C = (Directory *)C->nextSibling();
        }

        // Children have cleaned up -> remove children

        if( FromRep ) {
	  QString command = "cd ";
	  QString EndSshString;
	  int argcount = 0;

	  { QString Tmp = m_fullName;
	    Tmp = Tmp.replace( QRegExp(" "), "\\ " );
	    command += Tmp + " && ";
	  }

	  if( (QString::compare(m_connectMethod, "ext") == 0)
	     || (QString::compare(m_connectMethod, "") == 0) ) {
	    // rsh access
	    // command = exportCommand + " CVS_RSH=" + cvsRsh + "&& ";
	   if (cvsRsh.find("ssh") == 0) {

	      if (!sshAgentIsRunning) 
		startSshAgent();

	      command += "set-ssh-env.sh \"env CVS_RSH=" 
		       + cvsRsh + " cvs add ";

	      EndSshString =  "\"";

	   } else 
	     command += "env CVS_RSH=" + cvsRsh + " ";
	  }

	  command += "cvs remove";

	  Directory * C = (Directory *)firstChild();
	  while( C ) {
	    command += " " + C->shortName();
	    argcount ++;
	    C = (Directory *)C->nextSibling();
	  }

	  if( argcount ) {
	    if( ! EndSshString.isEmpty() )
		EndSshString =  "\"";
	    FILE* cvs = popen(command, "r");
	    if( cvs ) fclose( cvs );
	  }
        }

        if( FromDisk ) {
	  Directory *C = (Directory *)firstChild();
	  while( C ) {
	    QString CVSD;
	    CVSD = C->m_fullName + "/CVS/Entries";
	    unlink( CVSD.ascii() );
	    CVSD = C->m_fullName + "/CVS/Repository";
	    unlink( CVSD.ascii() );
	    CVSD = C->m_fullName + "/CVS/Root";
	    unlink( CVSD.ascii() );
	    CVSD = C->m_fullName + "/CVS";
	    rmdir( CVSD.ascii() );

	    rmdir( C->m_fullName.ascii() );
	    C = (Directory *)C->nextSibling();
	  }
        }
}
