/*!!***************************************************************************
 *!! FILE NAME
 *!!	$Source: /var/cvs/pub/repository/LinCVS/src/diffview.cpp,v $
 *!!
 *!! AUTHOR
 *!!	$Author: riemer $
 *!!
 *!! DATE, REVISION AND STATE
 *!!	$Date: 2001/07/06 01:19:59 $
 *!!	$Revision: 1.14 $
 *!!	$State: Exp $
 *!!
 *!! DESCRIPTION
 *!!    Implementation of LinCVS diff view.
 *!!
 *!!    Based on code created by Bernd Gehrmann
 *!!    Copyright (C) 1999 Bernd Gehrmann
 *!!    bernd@@physik.hu-berlin.de
 *!!
 *!!	Copyright (C) 2001 The LinCVS development team.
 *!!    Tilo Riemer <riemer@lincvs.org>
 *!!    Falk Brettschneider <gigafalk@yahoo.com>
 *!!    Wim Delvaux <wim.delvaux@chello.be>
 *!!    Jose Hernandez <joseh@tesco.net>
 *!!    Helmut Koll <HelmutKoll@web.de>
 *!!    Sven Trogisch <trogisch@iapp.de>
 *!!
 *!!	This program is free software; you can redistribute it and/or modify
 *!!	it under the terms of the GNU General Public License as published by
 *!!	the Free Software Foundation; either version 2, or (at your option)
 *!!	any later version.
 *!!
 *!!	This program is distributed in the hope that it will be useful,
 *!!	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *!!	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *!!	GNU General Public License for more details.
 *!!
 *!!**************************************************************************/


#include "config.h"
#include "ac_system_defs.h"

#include <iostream>
#include <stdlib.h>
#include <ctype.h> 
#include <qlist.h>
#include <qpainter.h>
#include <qscrollbar.h>

#include "diffview.h"

class DiffViewItem
{
public:
    QString line;
    DiffView::DiffType type;
    bool inverted;
    int no;
    int pixelwidth;
};
    
class DiffViewItemList : public QList<DiffViewItem>
{
protected:
    virtual int compareItems(QCollection::Item item1, QCollection::Item item2);
};


int DiffViewItemList::compareItems(QCollection::Item item1, QCollection::Item item2)
{
    return (static_cast<DiffViewItem*>(item1)->no
	    == static_cast<DiffViewItem*>(item2)->no)? 0 : 1;
}


static const int BORDER = 7;

static bool static_initialized = false;
static int  marker_typewidth; // always fixed
static int  linenos_typewidth; // always fixed


DiffView::DiffView( bool withlinenos, bool withmarker,
		    QWidget *parent, const char *name )
    : QTableView(parent, name)
{
    QFontMetrics fm( font() );

    if (!static_initialized) {
	static_initialized = true;
	marker_typewidth = QMAX(QMAX( fm.width("-"),
				      fm.width("+")),
				fm.width("|"))+2*BORDER;
	linenos_typewidth = fm.width("10000");
    }
    
    setFocusPolicy( QWidget::WheelFocus );
    textwidth = 0;
    setNumRows(0);
    setNumCols( 1 + (withlinenos?1:0) + (withmarker?1:0) );
    setTableFlags( Tbl_autoVScrollBar|Tbl_autoHScrollBar|
		   Tbl_smoothVScrolling
		   );
    setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
    setBackgroundMode( PaletteBase );
    setWFlags( WResizeNoErase );

    setCellWidth(0);
    setCellHeight(fm.lineSpacing());

    items = new DiffViewItemList;
    TableSize = 0;

    items->setAutoDelete(true);
    linenos = withlinenos;
    marker = withmarker;
}

DiffView::~DiffView( ) {
	delete items;
}

void DiffView::setPartner(DiffView *other)
{
    partner = other;
    if (partner)
	{
	    connect( verticalScrollBar(), SIGNAL(valueChanged(int)),
		     SLOT(positionChanged(int)) );
	    connect( verticalScrollBar(), SIGNAL(sliderMoved(int)),
		     SLOT(positionChanged(int)) );
	}
}


void DiffView::positionChanged(int val)
{
    if (partner)
	partner->setYOffset(val);
}


// *offset methods are only for views withlineno
void DiffView::removeAtOffset(int offset)
{
	items->remove(offset);
	setNumRows(numRows()-1);
	recalcTableSize();
}

void DiffView::recalcTableSize( DiffViewItem * NewItem ) {
        int maxline = 0;

	if( NewItem ) {
	  // just update
	  maxline = NewItem->pixelwidth;
	} else {
	  DiffViewItem * it;
	  // find largest line
	  for( unsigned int i = 0 ; i < items->count(); i ++ ) {
	    it = items->at(i);
	    if( it->pixelwidth > maxline )
	      maxline = it->pixelwidth;
	  }
	}

	if( maxline >= TableSize ) {
	  TableSize = maxline;
	  updateTableSize();
	}
}

void DiffView::insertAtOffset(QString line, DiffType type, int offset)
{
    DiffViewItem *item = new DiffViewItem;
    QFontMetrics fm(font());
    item->line = line;
    // remove trailing spaces
    int index = item->line.length()-1;

//    while( isspace( item->line[index].latin1() ) ) {
    while( item->line[index].isSpace() ) {
      index --;
    } 

    //contains trailing spaces
    item->line.truncate( index+1 );

    item->type = type;
    item->no = -1;
    item->pixelwidth = fm.width( item->line );
    item->inverted = false;

    items->insert(offset, item);

    recalcTableSize( item );

    //    DEBUGOUT( "added line " );
    setNumRows(numRows()+1);
    textwidth = QMAX(textwidth, fm.width(line));
}    


void DiffView::setCenterOffset(int offset)
{
    if (!rowIsVisible(offset))
	{
	    int visiblerows = viewHeight()/cellHeight(0);
	    //	    DEBUGOUT( "pos:" << pos );
	    //	    DEBUGOUT( "visiblerows:" << visiblerows );
	    setTopCell( QMAX(0, offset - visiblerows/2) );
	}
}


void DiffView::addLine(QString line, DiffType type, int no)
{
    DiffViewItem *item = new DiffViewItem;
    QFontMetrics fm(font());
    item->line = line;

    int index = item->line.length()-1;

//    while( isspace( item->line[index].latin1() ) ) {
    while( item->line[index].isSpace() ) {
      index --;
    } 

    //contains trailing spaces
    item->line.truncate( index+1 );

    item->type = type;
    item->no = no;
    item->inverted = false;
    item->pixelwidth = fm.width( item->line );
    items->append(item);

    recalcTableSize( item );

    //    DEBUGOUT( "added line " );
    setNumRows(numRows()+1);
    textwidth = QMAX(textwidth, fm.width(line));
}


QString DiffView::stringAtOffset(unsigned offset)
{
    if (offset >= items->count())
	{
	    exit( 234 );// ( "Internal error: lineAtOffset" );
	}
    return items->at(offset)->line;
}


int DiffView::count()
{
    return items->count();
}


int DiffView::findLine(int lineno)
{
    int offset;
    DiffViewItem tmp;
    tmp.no = lineno;
    if ( (offset = items->find(&tmp)) == -1)
	{
	    exit( 234);
	    return -1;
	}
    return offset;
}


void DiffView::findNextDiff (void)
{
    if (numRows() < 1) return;
    
    QListIterator<DiffViewItem> it(*items);
    DiffType type;

    /* Look for the next change and move the view to it */
    for (it+=topCell(), type=it.current()->type; it.current(); ++it)
	{
	    if (!it.current()) return;

	    if (type != it.current()->type)
		{
		    type = it.current()->type;

		    if ((type != Neutral) && (type != Unchanged))
			{
			    setTopCell (items->findRef (it.current()));
			    return;
			}
		}
	}

    /* Scroll to the bottom of the view if no further changes were found */
    setTopCell (numRows ());
}

void DiffView::findPrevDiff (void)
{
    if (numRows() < 1) return;
    
    QListIterator<DiffViewItem> it(*items);
    DiffType type;
	/* Look for the previous change and move the view to it */
	for (it+=topCell(), type=it.current()->type; it.current(); --it)
	{
		if (type != it.current()->type)
		{
			type = it.current()->type;

			if ((type != Neutral) && (type != Unchanged))
			{
				/* Adjust so we show the begining of the
				 * change rather than the end.
				 */
				while (type == it.current()->type) {
					--it;
					if (it == 0) {
						setTopCell (0);
						return;
					}
				}
				setTopCell (items->findRef (++it));
				return;
			}
		}
	}

	/* Scroll to the top of the view if no further changes were found */
	setTopCell (0);
}


void DiffView::setInverted(int lineno, bool inverted)
{
    int offset;
    if ( (offset = findLine(lineno)) != -1)
	items->at(offset)->inverted = inverted;
}


void DiffView::setCenterLine(int lineno)
{
    int offset;
    if ( (offset = findLine(lineno)) != -1)
	setCenterOffset(offset);
}


QString DiffView::stringAtLine(int lineno)
{
    int pos;
    if ( (pos = findLine(lineno)) != -1 )
	{
            //	    DEBUGOUT( "Returning " << pos << "," << items->at(pos)->line );
	    return items->at(pos)->line;
	}
    return NULL;
}

int DiffView::cellWidth(int col) {

    if (linenos && col == 0 ) {
      // col 0 contains linenos in this case
      return linenos_typewidth;
    } else if (marker && col <= 1 )
      return marker_typewidth;
    else
      return TableSize;
}

QSize DiffView::sizeHint() const
{
    QFontMetrics fm(font());
    return QSize( 4*fm.width("0123456789"), fm.lineSpacing()*8 );
}


void DiffView::paintCell(QPainter *p, int row, int col)
{
    DiffViewItem *item = items->at(row);

    int width = cellWidth(col);
    int height = cellHeight();

    QColor backgroundColor;
    bool inverted;
    int align;
    int innerborder;
    QString str;
    
    if (col == 0 && linenos) {
	backgroundColor = QColor(222, 222, 222);
	inverted = false;
	align = AlignLeft;
	innerborder = 0;
	if (item->no == -1)
	    str = "+++++";
	else
	    str.setNum(item->no);
    } else if (marker && (col == 0 || col == 1)) {

   //backgroundColor = lightGray;   --> lightgray is wrongly mapped under IRIX
   backgroundColor = QColor(222, 222, 222);
       
	inverted = false;
	align = AlignRight;
	innerborder = BORDER;
	str = (item->type==Change)? "|"
	    : (item->type==Insert)? "+" 
	    : (item->type==Delete)? "-" : " ";
    } else {
	backgroundColor =
	    (item->type==Change)? QColor(237, 190, 190)
	    : (item->type==Insert)? QColor(190, 190, 237)
	    : (item->type==Delete)? QColor(190, 237, 190)
	    //: (item->type==Neutral)? gray : white;   --> gray is wrongly mapped under IRIX
	    : (item->type==Neutral)? QColor(170, 170, 170) : white;
	inverted = item->inverted;
	align = AlignLeft;
	innerborder = 0;
	str = item->line;
    }

    if (inverted) {
	p->setPen(backgroundColor);
	//backgroundColor = black; --> black also
	backgroundColor = QColor(90, 90, 90);
    } else
	p->setPen(black);

    p->fillRect(0, 0, width, height, backgroundColor);
    p->drawText(innerborder, 0, width-2*innerborder, height, align, str);
    //    if (inverted)
    //	p->setPen(backgroundColor);
}    



// Local Variables:
// c-basic-offset: 4
// End:
