/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_return_poolindex.c
    \brief  This interface returns the poolindex, i.e. leaseid, using LCMAPS
    \author Martijn Steenbakkers for the EU DataGrid.

    If LCMAPS is invoked via this interface it will return the poolindex,
    alternatively named leaseid, to the calling application.
    -# lcmaps_return_poolindex_va
        Returns the poolindex based on a variable argument list
    -# lcmaps_return_poolindex
        Returns the poolindex based on fixed arguments:
                security context, buffer and buffer length
    -# lcmaps_return_poolindex_from_gss_cred
        Returns the poolindex based on fixed arguments:
                gss credential, buffer and buffer length
*/

#include "lcmaps_config.h"

/* Needed for NULL */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

/* globus headers */
#include <gssapi.h>
#include <globus_gss_assist.h>

/* LCMAPS includes */
#include "lcmaps.h"
#include "lcmaps_log.h"
#include "lcmaps_return_poolindex.h"

#include "pluginmanager/_lcmaps_log.h"
#include "pluginmanager/_lcmaps_utils.h"

/* Next header defines functions needed to dig into the internal globus structs
 * Actual structs are defined in globus_internal.h */
#include "grid_credential_handling/gsi_handling/_lcmaps_gsi_utils.h"

/******************************************************************************
                             Module specific defines
******************************************************************************/
/* Default settings */
#define LCMAPS_MAX_POLICIES 10

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static gss_cred_id_t dig_up_gss_cred_from_context(gss_ctx_id_t);
static int           lcmaps_return_poolindex_va(int , ...);

/******************************************************************************
Function:       Dig up the user gss credential from the security context
Description:    Dig up the user gss credential from the security context. Uses
                gss_assist libraries and, in case the delegate credential is empty,
                uses internal gsi libraries as well (hmmmm).
Parameters:
                context_handle:  the security context handle
Returns:        the user credential or GSS_C_NO_CREDENTIAL in case of failure
******************************************************************************/
static gss_cred_id_t dig_up_gss_cred_from_context(
    gss_ctx_id_t context_handle
)
{
    gss_name_t                          peer = GSS_C_NO_NAME;
    gss_buffer_desc                     peer_name_buffer = GSS_C_EMPTY_BUFFER;
    OM_uint32                           major_status = 0;
    OM_uint32                           minor_status = 0;
    int                                 rc = -1;
    int                                 initiator = -1;
    gss_cred_id_t                       user_cred_handle = GSS_C_NO_CREDENTIAL;

    rc = globus_module_activate(GLOBUS_GSI_GSS_ASSIST_MODULE);
    if (rc != GLOBUS_SUCCESS)
    {
        lcmaps_log (LOG_ERR, "Error activating GLOBUS_GSI_GSS_ASSIST_MODULE\n");
        return GSS_C_NO_CREDENTIAL;
    }

    rc = globus_module_activate(GLOBUS_GSI_GSSAPI_MODULE);
    if (rc != GLOBUS_SUCCESS)
    {
        lcmaps_log (LOG_ERR, "Error activating GLOBUS_GSI_GSSAPI_MODULE\n");
        return GSS_C_NO_CREDENTIAL;
    }

    major_status = gss_inquire_context(&minor_status,
                                       context_handle,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       &initiator,
                                       GLOBUS_NULL);

    if(GSS_ERROR(major_status))
    {
        lcmaps_log (LOG_ERR, "GSS failed: Cannot inquire context, Major:%8.8x Minor:%8.8x\n",
                   major_status, minor_status);
        goto dig_up_gss_cred_from_context_finalize;
    }

    major_status = gss_inquire_context(&minor_status,
                                       context_handle,
                                       initiator ? GLOBUS_NULL : &peer,
                                       initiator ? &peer : GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL);

    if(GSS_ERROR(major_status))
    {
        lcmaps_log (LOG_ERR, "GSS failed: Cannot inquire context, Major:%8.8x Minor:%8.8x\n",
                   major_status, minor_status);
        goto dig_up_gss_cred_from_context_finalize;
    }
    
    major_status = gss_display_name(&minor_status,
                                    peer,
                                    &peer_name_buffer,
                                    GLOBUS_NULL);
    
    if(GSS_ERROR(major_status))
    {
        lcmaps_log (LOG_ERR, "GSS failed: Cannot display name of peer, Major:%8.8x Minor:%8.8x\n",
                   major_status, minor_status);
        gss_release_name(&minor_status, &peer);
        goto dig_up_gss_cred_from_context_finalize;
    }

    gss_release_name(&minor_status, &peer);

    /* peer_name_buffer.value is a void* */
    lcmaps_log (LOG_INFO, "Found name of peer: %s\n",
		(char*)(peer_name_buffer.value));
    /*
     * Sometimes the delegated credential returned by
     * globus_gss_assist_accept_sec_context()/gss_accept_sec_context() is empty.
     * In particular this is true for gatekeeper pings
     * (globusrun -a: "authenticate only", should be "authenticate+authorize only").
     * In this case the gatekeeper should not act on behalf of the user so it does 
     * not receive the delegated proxy.
     * However, LCAS and LCMAPS need a user credential for the authorization chain.
     * This user credential is contained in the context, we have to dig it out.
     * This requires two additional header files: gssapi_openssl.h, globus_gsi_gss_constants.h.
     */

    user_cred_handle = lcmaps_ctx_to_cred(context_handle);

dig_up_gss_cred_from_context_finalize:
    globus_module_deactivate(GLOBUS_GSI_GSSAPI_MODULE);
    globus_module_deactivate(GLOBUS_GSI_GSS_ASSIST_MODULE);

    return user_cred_handle; /* might be equal to GSS_C_NO_CREDENTIAL */
}

/******************************************************************************
Function:   lcmaps_return_poolindex_va
Description:
    If LCMAPS is invoked via this interface it will return the poolindex,
    alternatively named leaseid, to the calling application.
    The arguments of this function are passed as a variable argument list.

Parameters:
    narg:  the number of arguments that follow
    ap:    vararg list, that consists of
        - The GSS Security context established during invocation of the calling
          service. This parameter is of type gss_ctx_id_t.
        - A pointer to a buffer. This buffer will contain the poolindex (NUL
          terminated string) upon successful return.
          This parameter is of type char *.
        - The length of the above mentioned buffer.
          This parameter is of type unsigned int.

Returns:
    0: success
    1: failure
******************************************************************************/
/*!
    \fn lcmaps_return_poolindex_va(
        int        narg,
        va_list    ap
    )
    \brief LCMAPS returns the poolindex
    If LCMAPS is invoked via this interface it will return the poolindex,
    alternatively named leaseid, to the calling application.
    The arguments of this function are passed as a variable argument list.

    \param narg  the number of arguments that follow
        This list currently contains:
        -# The GSS Security context established during invocation of the calling
           service. This parameter is of type gss_ctx_id_t.
        -# A pointer to a buffer. This buffer will contain the poolindex (NUL
           terminated string) upon successful return.
           This parameter is of type char *.
        -# The length of the above mentioned buffer.
           This parameter is of type unsigned int.

    \retval 0 success.
    \retval 1 failure.
*/
static int lcmaps_return_poolindex_va(
    int        narg,
    ...
)
{
    va_list                             ap;
    gss_ctx_id_t                        context_handle = GSS_C_NO_CONTEXT;
    char *                              poolindex_buffer = NULL;
    size_t                              buffer_length = 0;
    char *                              lcmaps_policy_string = NULL;
    char *                              lcmaps_policies[LCMAPS_MAX_POLICIES];
    int                                 lcmaps_npols = LCMAPS_MAX_POLICIES;
    int                                 rc = -1;
    gss_cred_id_t                       user_cred_handle = GSS_C_NO_CREDENTIAL;
    char *                              poolindex = NULL;
    int                                 jj = 0;

    /* First initialize LCMAPS, initialize without file pointer or name which
     * will result in getting env var LCMAPS_LOG_FILE. Specify DO_USRLOG to try
     * that first, if that fails we will go automatically to syslog */
    if (lcmaps_init_and_logfile(NULL, NULL, (unsigned short)(DO_USRLOG)))
    {
        lcmaps_log(LOG_ERR, "%s: LCMAPS initialization failure\n", __func__);
        return 1;
    }


    /* Parse arguments from va_list */
    va_start(ap, narg);
    if (narg == 3)
    {
        context_handle = va_arg(ap, gss_ctx_id_t);
        poolindex_buffer = va_arg(ap, char *);
        buffer_length = (size_t)va_arg(ap, unsigned int);
    }
    else if (narg == 4)
    {
        context_handle = va_arg(ap, gss_ctx_id_t);
        poolindex_buffer = va_arg(ap, char *);
        buffer_length = (size_t)va_arg(ap, unsigned int);
        user_cred_handle = va_arg(ap, gss_cred_id_t);
    }
    else
    {
        lcmaps_log(LOG_ERR, "%s: The number of arguments (%d) should be in the range %d-%d\n", __func__, narg, 3, 4);
        return 1;
    }
    va_end(ap);

    /* Parse policy environment variable */
    for (jj=0; jj < LCMAPS_MAX_POLICIES; jj++) lcmaps_policies[jj] = 0;
    lcmaps_policy_string = getenv("LCMAPS_POLICY_NAME");
    if ((rc = lcmaps_tokenize(lcmaps_policy_string, lcmaps_policies, &lcmaps_npols, ":")) != 0)
    {
        lcmaps_log(LOG_ERR, "%s: Cannot parse LCMAPS_POLICY_NAME environment variable, because\n", __func__);
        switch (rc)
        {
            case -1:
                lcmaps_log(LOG_ERR, "%s: of a malloc error\n", __func__);
                break;
            case -2:
                lcmaps_log(LOG_ERR, "%s: the policy list is too large (max = %d)\n",
                    __func__, LCMAPS_MAX_POLICIES);
                break;
            case -3:
                lcmaps_log(LOG_ERR, "%s: of a non-matching quote\n", __func__);
                break;
            case -4:
                lcmaps_log(LOG_ERR, "%s: of invalid input\n", __func__);
                break;
            default:
                lcmaps_log(LOG_ERR, "%s: of an unknown error\n", __func__);
                break;
        }
        goto return_poolindex_error;
    }

    /* 
     * Retrieve clients (delegated) credential
     * Check if by any chance the gss credential was directly given to us and
     * then use it. If not then plan B: use the security context.
     * Plan B:
     *   First check who initiated the context (should not be us, I think) and
     *   retrieve the name of the peer (=client)
     *   Then dig out the peer's credential from the context structure
     *   (not nice, will globus provide a mechanism for this in the future ?)
     */
    if (user_cred_handle == GSS_C_NO_CREDENTIAL)
    {
        /* plan B */
        user_cred_handle = dig_up_gss_cred_from_context(context_handle);
        if (user_cred_handle == GSS_C_NO_CREDENTIAL)
        {
            lcmaps_log (LOG_ERR, "LCMAPS could not find a valid user gss credential\n");
            goto return_poolindex_error;
        }
    }

    /*
     * Now that we have the credential let's run (and terminate) LCMAPS !
     */
#if ALLOW_EMPTY_CREDENTIALS
    rc = lcmaps_run_and_return_poolindex(
            NULL,
            user_cred_handle,
            NULL,
            &poolindex,
            lcmaps_npols,
            lcmaps_policies);
#else
    rc = lcmaps_run_and_return_poolindex(
            user_cred_handle,
            NULL,
            &poolindex,
            lcmaps_npols,
            lcmaps_policies);
#endif
    if (rc != 0)
    {
        lcmaps_log (LOG_ERR, "LCMAPS failed to do mapping and return poolindex\n");
        if (lcmaps_term())
        {
            lcmaps_log (LOG_ERR, "LCMAPS termination failure\n");
            goto return_poolindex_error;
        }
        goto return_poolindex_error;
    }
    /* Now copy poolindex into buffer */
    if (poolindex)
    {
        if(strlen(poolindex) + 1 > buffer_length)
        {
            lcmaps_log (LOG_ERR, "Buffer (length=%lu) too small for poolindex (length=%lu)\n",
                       (long unsigned)(buffer_length-1), (long unsigned)strlen(poolindex));
            free(poolindex);
            goto return_poolindex_error;
        }
        else
        {
            strncpy(poolindex_buffer, poolindex, buffer_length);
            lcmaps_log(LOG_DEBUG, "lcmaps_return_poolindex(): Returning poolindex %s\n", poolindex);
        }
        free(poolindex);
    }
    else
    {
        lcmaps_log (LOG_ERR, "LCMAPS could not find the poolindex\n");
        goto return_poolindex_error;
    }

    rc = lcmaps_term();
    if (rc)
    {
        lcmaps_log (LOG_ERR, "LCMAPS termination failure\n");
        goto return_poolindex_error;
    }

    return 0;

 return_poolindex_error:

    return 1;
}

/******************************************************************************
Function:   lcmaps_return_poolindex
Description:
    If LCMAPS is invoked via this interface it will return the poolindex,
    alternatively named leaseid, to the calling application.

Parameters:
    context_handle:
        The GSS Security context established during invocation of the calling
        service. This parameter is of type gss_ctx_id_t.
    poolindex_buffer:
        A pointer to a buffer. This buffer will contain the poolindex (NUL
        terminated string) upon successful return.
        This parameter is of type char *.
    buffer_length:
        The length of the above mentioned buffer.
        This parameter is of type unsigned int.

Returns:
    0: success
    1: failure
******************************************************************************/
int lcmaps_return_poolindex(
    gss_ctx_id_t context_handle,
    char *       poolindex_buffer,
    unsigned int buffer_length
)
{
    return (lcmaps_return_poolindex_va(3, context_handle, poolindex_buffer, buffer_length));
}

/******************************************************************************
Function:   lcmaps_return_poolindex_from_gss_cred
Description:
    If LCMAPS is invoked via this interface it will return the poolindex,
    alternatively named leaseid, to the calling application.
    Instead of using the security context as an input it uses a gss credential
    (type gss_cred_id_t).

Parameters:
    credential_handle:
        The GSS credential (of type gss_cred_id_t) of the user.
    poolindex_buffer:
        A pointer to a buffer. This buffer will contain the poolindex (NUL
        terminated string) upon successful return.
        This parameter is of type char *.
    buffer_length:
        The length of the above mentioned buffer.
        This parameter is of type unsigned int.

Returns:
    0: success
    1: failure
******************************************************************************/
int lcmaps_return_poolindex_from_gss_cred(
    gss_cred_id_t credential_handle,
    char *        poolindex_buffer,
    unsigned int  buffer_length
)
{
    return (lcmaps_return_poolindex_va(
                4,
                GSS_C_NO_CONTEXT,
                poolindex_buffer,
                buffer_length,
                credential_handle)
           );
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/lcmaps_return_poolindex.c,v $
    $Date: 2015-02-02 11:46:42 +0100 (Mon, 02 Feb 2015) $
    $Revision: 18216 $
    $Author: msalle $
******************************************************************************/
