package latexDraw.figures;

import java.awt.*;
import java.awt.geom.Line2D;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.ObjectInputStream;

import javax.swing.JComboBox;

import latexDraw.figures.properties.Arrowable;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.components.LaTeXDrawComboBox;
import latexDraw.ui.components.MagneticGrid;
import latexDraw.util.LaTeXDrawNumber;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * This class defines axes.<br>
 *<br>
 * TODO: it seems to have an redefinition of psaxes in the pstricks-add package, but this class
 * define only the psaxes of the pst-plot package. It possible to add the possibility to choose
 * which definition we want to use.<br><br>
 * Ox and Oy parameters can be float, but it occurs problems; for this reason these parameters will
 * be only integer in LaTeXDraw. cf. http://tug.org/mailman/htdig/pstricks/1998/000227.html<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE.  See the GNU General Public License for more details.<br>
 *<br>
 * 31/09/06<br>
 * @author Arnaud BLOUIN<br>
 * @since 1.8<br>
 * @version 2.0.0<br>
 */
public class Axe extends GridShape implements Arrowable
{
	private static final long serialVersionUID = 1L;
	
	/** The increment of each axe; Dx and Dy in PSTricks. */
	protected LaTeXDrawPoint2D increment;
	
	/** The distance between each label; if 0, the default value will be used; dx and oy in PSTricks (in cm). */
	protected LaTeXDrawPoint2D distLabels;
	
	/** Define which labels must be displayed. */
	protected int labelsDisplayed;
	
	/** Define the origin must be shown. */
	protected boolean showOrigin;
	
	/** Define how the ticks must be shown. (use the LABELS_DISPLAYED_XXX constants to define its value).*/
	protected int ticksDisplayed;
	
	/** Define the style of the ticks. */
	protected int ticksStyle;
	
	/** The size of the ticks. */
	protected double ticksSize;
	
	/** The style of the axes. */
	protected int axesStyle;
	
	/** The first arrow. */
	protected ArrowHead arrowHead1;
	
	/** The second arrow. */
	protected ArrowHead arrowHead2;
	
	
	public static final String ACTION_CMD_AXE_STYLE_CHOICE = "axeStyleCmd"; //$NON-NLS-1$
	
	public static final String ACTION_CMD_TICKS_STYLE_CHOICE = "ticksStyleCmd";//$NON-NLS-1$
	
	/** The value of axesStyle when the axes have the axes shape. */
	public static final short AXES_STYLE_AXES_VALUE 	= 0;
	
	/** The value of axesStyle when the axes have the frame shape. */
	public static final short AXES_STYLE_FRAME_VALUE 	= 1;
	
	/** The value of axesStyle when the axes must not be shown. */
	public static final short AXES_STYLE_NONE_VALUE 	= 2;
	
	/** The value of ticksStyle when the ticks must be fully shown. */
	public static final short TICKS_STYLE_FULL_VALUE 	= 0;
	
	/** The value of ticksStyle when only the top of the ticks must be shown. */
	public static final short TICKS_STYLE_TOP_VALUE 	= 1;
	
	/** The value of ticksStyle when only the bottom of the ticks must be shown. */
	public static final short TICKS_STYLE_BOTTOM_VALUE = 2;
	
	/** The value of labelsDisplayed when all the labels must be displayed. */
	public static final short LABELS_DISPLAYED_ALL_VALUE 	= 0;
	
	/** The value of labelsDisplayed when only the X labels must be displayed. */
	public static final short LABELS_DISPLAYED_X_VALUE 	= 1;
	
	/** The value of labelsDisplayed when only the Y labels must be displayed. */
	public static final short LABELS_DISPLAYED_Y_VALUE 	= 2;
	
	/** The value of labelsDisplayed when no labels must be displayed. */
	public static final short LABELS_DISPLAYED_NON_VALUE 	= 3;

	public static final int DEFAULT_LABELSSIZE = 20;
	
	public static final String LABEL_STYLE_NONE = LaTeXDrawLang.getString1_8("Axe.0"); //$NON-NLS-1$
	
	public static final String LABEL_STYLE_AXE = LaTeXDrawLang.getString1_8("Axe.1"); //$NON-NLS-1$
	
	public static final String LABEL_STYLE_FRAME = LaTeXDrawLang.getString1_8("Axe.2"); //$NON-NLS-1$
	
	public static final String LABEL_STYLE_TICKS_FULL 	= LaTeXDrawLang.getString1_8("Axe.3"); //$NON-NLS-1$
	
	public static final String LABEL_STYLE_TICKS_TOP 	= LaTeXDrawLang.getString1_8("Axe.4"); //$NON-NLS-1$
	
	public static final String LABEL_STYLE_TICKS_BOTTOM = LaTeXDrawLang.getString1_8("Axe.5"); //$NON-NLS-1$
	
	/** The interval between the labels and the axes. */
	public static final double GAP_LABEL = 5.;

	
	
	/**
	 * The constructor using two arguments.
	 * @param pos The current position of the origin of the axes.
	 */
	public Axe(LaTeXDrawPoint2D pos, boolean increaseMeter)
	{
		super(pos, increaseMeter);
		
		canHaveArrow		= true;
		isCustomizable 		= true;
		isDashableOrDotable = true;
		isThicknessable		= true;
		showOrigin 	= PSTricksConstants.DEFAULT_SHOW_ORIGIN;
		ticksSize 	= PSTricksConstants.DEFAULT_TICKS_SIZE*Figure.PPC;
		distLabels	= new LaTeXDrawPoint2D(PSTricksConstants.DEFAULT_DIST_X_LABEL, 
											PSTricksConstants.DEFAULT_DIST_Y_LABEL);
		increment 	= new LaTeXDrawPoint2D(PSTricksConstants.DEFAULT_DX, PSTricksConstants.DEFAULT_DY);
		arrowHead1 	= new ArrowHead(new LaTeXDrawPoint2D(pos), 
									new Line(new LaTeXDrawPoint2D(pos), new LaTeXDrawPoint2D(pos), false), this);
		arrowHead2 	= new ArrowHead(new LaTeXDrawPoint2D(pos), 
									new Line(new LaTeXDrawPoint2D(pos), new LaTeXDrawPoint2D(pos), false), this);

		BufferedImage bufferImage = new BufferedImage(2, 2, BufferedImage.TYPE_INT_RGB);
		Graphics2D g = bufferImage.createGraphics();
		font = new Font(g.getFont().getName(), Font.PLAIN, DEFAULT_LABELSSIZE);
		
		setAxesStyleToken(PSTricksConstants.DEFAULT_AXES_STYLE);
		setTicksDisplayedToken(PSTricksConstants.DEFAULT_TICKS_DISPLAYED);
		setTicksStyleToken(PSTricksConstants.DEFAULT_TICKS_STYLE);
		setLabelsDisplayedToken(PSTricksConstants.DEFAULT_LABELS_DISPLAYED);

		updateFonts();
		updateShape();
		
		g.dispose();
		bufferImage.flush();
		g = null;
		bufferImage = null;
	}

	

	@Override
	public void updateBorders()
	{
		if(borders==null)
			borders = new LaTeXDrawRectangle(false);
		
		double minX, maxX, minY, maxY, add1X=0, add2X=0, add1Y=0, add2Y=0;
		double ticks = ticksSize;
		int ticksS = getTicksStyle();
		boolean ticksTop = ticksS==TICKS_STYLE_TOP_VALUE || ticksS==TICKS_STYLE_FULL_VALUE;
		boolean ticksBot = ticksS==TICKS_STYLE_BOTTOM_VALUE || ticksS==TICKS_STYLE_FULL_VALUE;
		
		if(gridEnd.x<gridStart.x)
		{
			minX = gridEnd.x;
			maxX = gridStart.x;
		}
		else
		{
			minX = gridStart.x;
			maxX = gridEnd.x;
		}
		
		if(gridEnd.y<gridStart.y)
		{
			minY = gridEnd.y;
			maxY = gridStart.y;
		}
		else
		{
			minY = gridStart.y;
			maxY = gridEnd.y;
		}
		
		if(minX>=0 && isYLabelWest)
			add1X = Math.max(fontMetrics.stringWidth(String.valueOf(minY)), 
							 fontMetrics.stringWidth(String.valueOf(maxY)))+(ticksBot?ticks:0);
		
		if(maxX<=0)
			add2X = Math.max(fontMetrics.stringWidth(String.valueOf(minY)), 
							 fontMetrics.stringWidth(String.valueOf(maxY)))+(ticksBot?ticks:0);
		
		if(minY>=0 && isXLabelSouth)
			add2Y = fontMetrics.getHeight()+GAP_LABEL+(ticksBot?ticks:0);
		
		if(maxY<=0)
			add1Y = fontMetrics.getHeight()+GAP_LABEL+(ticksTop?ticks:0);
		
		borders.setFirstPoint(position.x+Math.min(0, minX)*PPC-add1X,
							  position.y-Math.max(0, maxY)*PPC-add1Y);
		borders.setLastPoint(position.x+Math.max(0, maxX)*PPC+add2X,
							 position.y-Math.min(0, minY)*PPC+add2Y);
		
		gravityCenter = borders.getGravityCenter();
		updateShape();
	}



	@Override
	public void draw(Graphics2D g, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering)
	{
		Color formerColor = g.getColor();
		Font formerFont = g.getFont();
		double minX, maxX, minY, maxY, maxX3, minX3, maxY3, minY3;
		double gapX = distLabels.x==0. ? PPC : (distLabels.x/increment.x)*PPC, i; 
		double gapY = distLabels.y==0. ? PPC : (distLabels.y/increment.y)*PPC;
		LaTeXDrawPoint2D NW = borders.getTheNWPoint();
		LaTeXDrawPoint2D SE = borders.getTheSEPoint();	
		double cx = (NW.x+SE.x)/2., cy = (NW.y+SE.y)/2.;
		double c2x = Math.cos(rotationAngle)*cx - Math.sin(rotationAngle)*cy;
		double c2y = Math.sin(rotationAngle)*cx + Math.cos(rotationAngle)*cy;
		double c3x = Math.cos(-rotationAngle)*(cx-c2x) - Math.sin(-rotationAngle)*(cy-c2y);
		double c3y = Math.sin(-rotationAngle)*(cx-c2x) + Math.cos(-rotationAngle)*(cy-c2y);
		double lgth;
		boolean ticksBot = ticksStyle==TICKS_STYLE_BOTTOM_VALUE || ticksStyle==TICKS_STYLE_FULL_VALUE;
		int j;

		if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
		else 
		if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_ROUND, BasicStroke.JOIN_MITER,
					1.f, new float[]{0,thickness+dotSep}, 0));
		else
		if(lineStyle.equals(PSTricksConstants.LINE_DASHED_STYLE))
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER,
					1.f, new float[]{blackDashLength, whiteDashLength}, 0));

		g.setColor(linesColor);
		g.setFont(font);
			
		if(rotationAngle%(Math.PI*2)!=0)
		{		
			g.rotate(rotationAngle);
			g.translate(c3x,c3y);
		}
		
		if(gridEnd.x<gridStart.x)
		{
			minX = gridEnd.x;
			maxX = gridStart.x;
		}
		else
		{
			minX = gridStart.x;
			maxX = gridEnd.x;
		}
		
		if(gridEnd.y<gridStart.y)
		{
			minY = gridEnd.y;
			maxY = gridStart.y;
		}
		else
		{
			minY = gridStart.y;
			maxY = gridEnd.y;
		}
		
		if(distLabels.x!=0.)
		{
			if(maxX!=0.)
			{
				maxX = maxX/distLabels.x;
				if(maxX==0.)
					maxX=0.1;
			}
			
			if(minX!=0)
			{
				minX = minX/distLabels.x;
				if(minX==0.)
					minX=0.1;
			}
		}
		
		if(distLabels.y!=0.)
		{
			if(maxY!=0.)
			{
				maxY = maxY/distLabels.y;
				if(maxY==0.)
					maxY=0.1;
			}
			
			if(minY!=0.)
			{
				minY = minY/distLabels.y;
				if(minY==0.)
					minY=0.1;
			}
		}
		
		boolean arrow1Ok = !arrowHead1.getArrowStyle().equals(PSTricksConstants.NONEARROW_STYLE);
		boolean arrow2Ok = !arrowHead2.getArrowStyle().equals(PSTricksConstants.NONEARROW_STYLE);
		
		minX3 = axesStyle==AXES_STYLE_AXES_VALUE && ((arrow1Ok && isYLabelWest) || 
				(arrow2Ok && !isYLabelWest)) && gridStart.x!=0 && 
				(distLabels.x==0. || distLabels.x==1.)? minX+1 : minX; 		
		
		minY3 = axesStyle==AXES_STYLE_AXES_VALUE && ((arrow1Ok && isXLabelSouth) || 
				(arrow2Ok && !isXLabelSouth)) && gridStart.y!=0 && 
				(distLabels.y==0. || distLabels.y==1.)? minY+1 : minY; 			 
					 
		maxX3 = axesStyle==AXES_STYLE_AXES_VALUE &&
		 		((arrow2Ok && isYLabelWest) || (arrow1Ok && !isYLabelWest)) && 
		 		(distLabels.x==0. || distLabels.x==1.)? maxX-1 : maxX; 			 
		maxY3 = axesStyle==AXES_STYLE_AXES_VALUE &&
		 		((arrow2Ok && isXLabelSouth) || (arrow1Ok && !isXLabelSouth)) && 
		 		(distLabels.y==0. || distLabels.y==1.)? maxY-1 : maxY; 	
		
		minX3 = minX>0 ? 0 : minX3;
		maxX3 = maxX<0 ? 0 : maxX3;
		minY3 = minY>0 ? 0 : minY3;
		maxY3 = maxY<0 ? 0 : maxY3;
		
		switch(axesStyle)
		{
			case AXES_STYLE_AXES_VALUE:
				
				boolean arrowOk = (float)minX!=(float)maxX || (float)maxX!=(float)minY || (float)minY!=(float)maxY;
				Line xLine = new Line(new LaTeXDrawPoint2D(position.x+gridStart.x*PPC, position.y), 
										new LaTeXDrawPoint2D(position.x+gridEnd.x*PPC, position.y), true);
				Line yLine = new Line(new LaTeXDrawPoint2D(position.x, position.y-gridEnd.y*PPC), 
										new LaTeXDrawPoint2D(position.x, position.y-gridStart.y*PPC), true);

				xLine.setLinesColor(linesColor);
				xLine.setLineStyle(lineStyle);
				xLine.setThickness(thickness);
				yLine.setLinesColor(linesColor);
				yLine.setLineStyle(lineStyle);
				yLine.setThickness(thickness);
				
				if(!arrowHead1.getArrowStyle().equals(PSTricksConstants.NONEARROW_STYLE) && arrowOk)
				{
					String arrowStyle = arrowHead1.getArrowStyle();
					lgth = arrowHead1.getArrowHeadLength();
					boolean isArrow = arrowHead1.needReduceLine();
					
					if((minX!=0 && isYLabelWest) || (maxX!=0 && !isYLabelWest) || maxY==minY)
					{
						if(isArrow && !isYLabelWest)
						{
							if(arrowStyle.equals(PSTricksConstants.RARROW_STYLE))
								arrowStyle = PSTricksConstants.LARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LARROW_STYLE))
								arrowStyle = PSTricksConstants.RARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DLARROW_STYLE))
								arrowStyle = PSTricksConstants.DRARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DRARROW_STYLE))
								arrowStyle = PSTricksConstants.DLARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LSBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RSBRACKET_STYLE;
						}
						
						if(isArrow)
							if(isYLabelWest)
								 xLine.getPt1().x-=lgth;
							else xLine.getPt2().x+=lgth;
						
						if(isYLabelWest)
							xLine.setArrow1Style(arrowStyle);
						else
							xLine.setArrow2Style(arrowStyle);
					}
					
					if((minY!=0 && isXLabelSouth) || (maxY!=0 && !isXLabelSouth) || maxX==minX)
					{
						arrowStyle = arrowHead1.getArrowStyle();
						
						if(isArrow && isXLabelSouth)
						{
							if(arrowStyle.equals(PSTricksConstants.RARROW_STYLE))
								arrowStyle = PSTricksConstants.LARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LARROW_STYLE))
								arrowStyle = PSTricksConstants.RARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DLARROW_STYLE))
								arrowStyle = PSTricksConstants.DRARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DRARROW_STYLE))
								arrowStyle = PSTricksConstants.DLARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LSBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RSBRACKET_STYLE;
						}
						
						if(isArrow)
							if(isXLabelSouth)
								 yLine.getPt2().y+=lgth;
							else yLine.getPt1().y-=lgth;
						
						if(isXLabelSouth)
							yLine.setArrow2Style(arrowStyle);
						else
							yLine.setArrow1Style(arrowStyle);
					}
				}
				
				if(!arrowHead2.getArrowStyle().equals(PSTricksConstants.NONEARROW_STYLE) && arrowOk)
				{
					String arrowStyle = arrowHead2.getArrowStyle();
					lgth = arrowHead2.getArrowHeadLength();
					boolean isArrow = arrowHead2.needReduceLine();
					
					if((maxY!=0 && isXLabelSouth) || (minY!=0 && !isXLabelSouth) || maxY==minY)
					{
						if(isArrow && !isXLabelSouth)
						{
							if(arrowStyle.equals(PSTricksConstants.RARROW_STYLE))
								arrowStyle = PSTricksConstants.LARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LARROW_STYLE))
								arrowStyle = PSTricksConstants.RARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DLARROW_STYLE))
								arrowStyle = PSTricksConstants.DRARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DRARROW_STYLE))
								arrowStyle = PSTricksConstants.DLARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LSBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RSBRACKET_STYLE;
						}
						
						if(maxY==minY && minY==0)
						{
							if(isArrow)
								if(maxX>0)
									 xLine.getPt1().y-=lgth;
								else xLine.getPt2().y+=lgth;
							
							if(maxX>0)
								xLine.setArrow1Style(arrowStyle);
							else
								xLine.setArrow2Style(arrowStyle);
						}
						else
						{
							if(isArrow)
								if(isXLabelSouth)
									 yLine.getPt1().y-=lgth;
								else yLine.getPt2().y+=lgth;
							
							if(isXLabelSouth)
								yLine.setArrow1Style(arrowStyle);
							else
								yLine.setArrow2Style(arrowStyle);
						}
					}
					
					if((maxX!=0 && isYLabelWest) || (minX!=0 && !isYLabelWest) || maxX==minX)
					{
						arrowStyle = arrowHead2.getArrowStyle();
						
						if(isArrow && isYLabelWest)
						{
							if(arrowStyle.equals(PSTricksConstants.RARROW_STYLE))
								arrowStyle = PSTricksConstants.LARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LARROW_STYLE))
								arrowStyle = PSTricksConstants.RARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DLARROW_STYLE))
								arrowStyle = PSTricksConstants.DRARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DRARROW_STYLE))
								arrowStyle = PSTricksConstants.DLARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LSBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RSBRACKET_STYLE;
						}
						
						if(maxX==minX && minX==0)
						{
							if(isArrow)
								if(maxY>0)
									 yLine.getPt2().x+=lgth;
								else xLine.getPt1().x-=lgth;
							
							if(maxY>0)
								yLine.setArrow2Style(arrowStyle);
							else
								yLine.setArrow1Style(arrowStyle);
						}
						else
						{
							if(isArrow)
								if(isYLabelWest)
									 xLine.getPt2().x+=lgth;
								else xLine.getPt1().x-=lgth;
							
							if(isYLabelWest)
								xLine.setArrow2Style(arrowStyle);
							else
								xLine.setArrow1Style(arrowStyle);
						}
					}
				}

				xLine.draw(g, antiAlias, rendering, alphaInter, colorRendering); // We draw the axes
				yLine.draw(g, antiAlias, rendering, alphaInter, colorRendering);
				
				break;
				
			case AXES_STYLE_FRAME_VALUE:
				
				if(gridEnd.x>0 || gridEnd.y>0)
				{
					double xMax = position.x+gridEnd.x*PPC;
					double yMax = position.y-gridEnd.y*PPC;
					
					LaTeXDrawRectangle r = new LaTeXDrawRectangle(
							new LaTeXDrawPoint2D(position.x, gridEnd.y>0?yMax:position.y), 
							new LaTeXDrawPoint2D(gridEnd.x>0?xMax:position.x, position.y), false);
					
					r.setBordersPosition(PSTricksConstants.BORDERS_MIDDLE);
					r.setLinesColor(linesColor);
					r.setLineStyle(lineStyle);
					r.setThickness(thickness);
					r.draw(g, antiAlias, rendering, alphaInter, colorRendering);
				}
				break;
		}
		
		double endX = distLabels.x!=0. ? maxX3 : maxX3/increment.x;
		double endY = distLabels.y!=0. ? maxY3 : maxY3/increment.y;
		double startX = distLabels.x!=0. ? minX3 : minX3/increment.x;
		double startY = distLabels.y!=0. ? minY3 : minY3/increment.y;
		
		if(ticksDisplayed!=LABELS_DISPLAYED_NON_VALUE)// We draw the ticks
		{
			if(ticksStyle==TICKS_STYLE_FULL_VALUE || (ticksStyle==TICKS_STYLE_TOP_VALUE && maxY>0) ||
			  (ticksStyle==TICKS_STYLE_BOTTOM_VALUE && maxY<=0))
				if(ticksDisplayed==LABELS_DISPLAYED_X_VALUE || ticksDisplayed==LABELS_DISPLAYED_ALL_VALUE)
				{
					for(j=1; j<=endX; j++)
						g.draw(new Line2D.Double(position.x+j*gapX*increment.x, position.y, 
								 position.x+j*gapX*increment.x, position.y-ticksSize-thickness/2.));
					
					for(j=-1; j>=startX; j--)
						g.draw(new Line2D.Double(position.x+j*gapX*increment.x, position.y, 
								position.x+j*gapX*increment.x, position.y-ticksSize-thickness/2.));
				}
			
			if(ticksStyle==TICKS_STYLE_FULL_VALUE || (ticksStyle==TICKS_STYLE_TOP_VALUE && maxX>0) ||
			  (ticksStyle==TICKS_STYLE_BOTTOM_VALUE && maxX<=0))
				if(ticksDisplayed==LABELS_DISPLAYED_Y_VALUE || ticksDisplayed==LABELS_DISPLAYED_ALL_VALUE)
				{
					for(j=1; j<=endY; j++)
						g.draw(new Line2D.Double(position.x, position.y-j*gapY*increment.y, 
								 position.x+ticksSize+thickness/2., position.y-j*gapY*increment.y));
					
					for(j=-1; j>=startY; j--)
						g.draw(new Line2D.Double(position.x, position.y-j*gapY*increment.y, 
								position.x+ticksSize+thickness/2., position.y-j*gapY*increment.y));
				}
			
			if(ticksStyle==TICKS_STYLE_FULL_VALUE || (ticksStyle==TICKS_STYLE_BOTTOM_VALUE && maxY>0) ||
			  (ticksStyle==TICKS_STYLE_TOP_VALUE && maxY<=0))
				if(ticksDisplayed==LABELS_DISPLAYED_X_VALUE || ticksDisplayed==LABELS_DISPLAYED_ALL_VALUE)
				{
					for(j=1; j<=endX; j++)
						g.draw(new Line2D.Double(position.x+j*gapX*increment.x, position.y, 
								 position.x+j*gapX*increment.x, position.y+ticksSize+thickness/2.));
				
					for(j=-1; j>=startX; j--)
						g.draw(new Line2D.Double(position.x+j*gapX*increment.x, position.y, 
								position.x+j*gapX*increment.x, position.y+ticksSize+thickness/2.));
				}
			
			if(ticksStyle==TICKS_STYLE_FULL_VALUE || (ticksStyle==TICKS_STYLE_BOTTOM_VALUE && maxX>0) ||
				(ticksStyle==TICKS_STYLE_TOP_VALUE && maxX<=0))
				if(ticksDisplayed==LABELS_DISPLAYED_Y_VALUE || ticksDisplayed==LABELS_DISPLAYED_ALL_VALUE)
				{
					for(j=1; j<=endY; j++)
						g.draw(new Line2D.Double(position.x, position.y-j*gapY*increment.y, 
								 position.x-ticksSize-thickness/2., position.y-j*gapY*increment.y));
					
					for(j=-1; j>=startY; j--)
						g.draw(new Line2D.Double(position.x, position.y-j*gapY*increment.y, 
								position.x-ticksSize-thickness/2., position.y-j*gapY*increment.y));
				}
		}

		g.setColor(Color.BLACK);
		
		if(labelsDisplayed==LABELS_DISPLAYED_ALL_VALUE || labelsDisplayed==LABELS_DISPLAYED_X_VALUE)
		{// We show the labels on the X-axe.
			float height 	= fontMetrics.getAscent();
			double gap 		= ((ticksDisplayed==LABELS_DISPLAYED_ALL_VALUE||
							  ticksDisplayed==LABELS_DISPLAYED_X_VALUE)&&
							((isXLabelSouth && (ticksStyle==TICKS_STYLE_BOTTOM_VALUE || ticksStyle==TICKS_STYLE_FULL_VALUE)) ||
							 (!isXLabelSouth && (ticksStyle==TICKS_STYLE_TOP_VALUE || ticksStyle==TICKS_STYLE_FULL_VALUE)))?
							  ticksSize:0)+thickness/2.+GAP_LABEL;
			double sep		= maxY<=0 || !isXLabelSouth? -gap-GAP_LABEL : gap+height;
			String str;

			if(((isXLabelSouth && gridStart.y>=0) || (!isXLabelSouth && 
				gridEnd.y<=0)) && (gridStart.x!=gridEnd.x || gridStart.x!=0) && gridStart.y<=0 && showOrigin)
					g.drawString(String.valueOf((int)origin.x), 
							(int)(position.x-fontMetrics.stringWidth(String.valueOf((int)origin.x))/2.), (int)(position.y+sep));
			
			for(i=increment.x, j=1; j<=endX; i+=increment.x, j++)
			{
				str = (((int)increment.x)==increment.x?String.valueOf((int)(i+origin.x)) : String.valueOf(i+origin.x));
				g.drawString(str, (int)(position.x+j*gapX*increment.x-fontMetrics.stringWidth(str)/2.), (int)(position.y+sep));
			}
			
			for(i=-increment.x, j=-1; j>=startX; i-=increment.x, j--)
			{
				str = (((int)increment.x)==increment.x?String.valueOf((int)(i+origin.x)): String.valueOf(i+origin.x));
				g.drawString(str, (int)(position.x+j*gapX*increment.x-fontMetrics.stringWidth(str)/2.), (int)(position.y+sep));
			}
		}
		
		if(labelsDisplayed==LABELS_DISPLAYED_ALL_VALUE || labelsDisplayed==LABELS_DISPLAYED_Y_VALUE)
		{// We show the labels on the Y-axe.
			float height = fontMetrics.getAscent();
			String str;

			if(maxX>0 && isYLabelWest)
			{
				double gap	= (ticksBot && (ticksDisplayed==LABELS_DISPLAYED_ALL_VALUE ||
								ticksDisplayed==LABELS_DISPLAYED_Y_VALUE)?ticksSize:0)+thickness/2.;
			
				if(gridStart.x==0 && (gridStart.y!=gridEnd.y || gridStart.y!=0) && showOrigin)
					g.drawString(String.valueOf((int)origin.y), 
							(int)(position.x-gap-g.getFontMetrics().stringWidth(String.valueOf((int)origin.y))-GAP_LABEL), 
							(int)(position.y+height/2.));
				
				for(i=increment.y, j=1; j<=endY; i+=increment.y, j++)
				{
					str = ((int)increment.y)==increment.y?String.valueOf((int)(i+origin.y)):String.valueOf(i+origin.y);
					g.drawString(str, (int)(position.x-gap-fontMetrics.stringWidth(str)-GAP_LABEL), 
								(int)(position.y+height/2.-j*gapY*increment.y));
				}
				
				for(i=-increment.y, j=-1; j>=startY; i-=increment.y, j--)
				{
					str = ((int)increment.y)==increment.y?String.valueOf((int)(i+origin.y)):String.valueOf(i+origin.y);
					g.drawString(str, (int)(position.x-gap-fontMetrics.stringWidth(str)-GAP_LABEL), 
								(int)(position.y+height/2.-j*gapY*increment.y));
				}
			}
			else
			{
				double gap	= ((!ticksBot || ticksStyle==TICKS_STYLE_FULL_VALUE) && (ticksDisplayed==LABELS_DISPLAYED_ALL_VALUE||
								ticksDisplayed==LABELS_DISPLAYED_Y_VALUE)?ticksSize:0)+thickness/2.;
				
				if((!isYLabelWest && gridEnd.x<=0) || (isYLabelWest && gridStart.x>=0) && (gridStart.y!=gridEnd.y  || gridStart.y!=0) && showOrigin)
					g.drawString(String.valueOf((int)origin.y), (int)(position.x+gap+GAP_LABEL), (int)(position.y+height/2.));
				
				for(i=increment.y, j=1; j<=endY; i+=increment.y, j++)
					g.drawString(((int)increment.y)==increment.y?String.valueOf((int)(i+origin.y)):String.valueOf(i+origin.y), 
								(int)(position.x+gap+GAP_LABEL), (int)(position.y+height/2.-j*gapY*increment.y));	
				
				for(i=-increment.y, j=-1; j>=startY; i-=increment.y, j--)
					g.drawString(((int)increment.y)==increment.y?String.valueOf((int)(i+origin.y)):String.valueOf(i+origin.y), 
								(int)(position.x+gap+GAP_LABEL), (int)(position.y+height/2.-j*gapY*increment.y));	
			}
		}
		
		if(rotationAngle%(Math.PI*2)!=0)
		{
			g.translate(-c3x, -c3y);
			g.rotate(-rotationAngle);
		}
		
		g.setColor(formerColor);
		g.setFont(formerFont);
		
		if(isSelected)
			borders.draw(g, false, antiAlias, rendering, alphaInter, colorRendering);
	}





	@Override
	public String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		double startX, startY, endX, endY, x, y;
		LaTeXDrawPoint2D d = drawBorders.getOriginPoint();
		String start = "", end = "", arrowsCode = "";//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
		String arrowHead1Style = arrowHead1.getArrowStyle(), add = "";//$NON-NLS-1$
		String arrowHead2Style = arrowHead2.getArrowStyle();
		double threshold = 0.001;
		
		if(isXLabelSouth)
		{
			startY = gridStart.y;
			endY   = gridEnd.y;
		}else
		{
			startY = gridEnd.y;
			endY   = gridStart.y;

		}
		
		if(isYLabelWest)
		{
			startX = gridStart.x;
			endX   = gridEnd.x;
		}else
		{
			startX = gridEnd.x;
			endX   = gridStart.x;
		}
		
		String coord = "("+(int)startX+","+(int)startY+")("+ //$NON-NLS-1$ //$NON-NLS-3$ //$NON-NLS-2$
						(int)endX+","+(int)endY+")"; //$NON-NLS-1$ //$NON-NLS-2$

		coord = "(" + 0 + "," + 0 + ")" + coord; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

		if(position.x!=0 || position.y!=0)
		{
			double posX = position.x, posY = position.y;
			if(Math.abs(position.x) < 0.001) posX = 0;
			if(Math.abs(position.x) < 0.001) posY = 0;
			
			end +="}"; //$NON-NLS-1$
			start = "\\rput(" + LaTeXDrawNumber.getCutNumber((float)((posX-d.x)/ppc), threshold) + "," + //$NON-NLS-1$ //$NON-NLS-2$
					LaTeXDrawNumber.getCutNumber((float)((d.y-posY)/ppc), threshold) + "){" + start; //$NON-NLS-1$
		}
		
		if(rotationAngle % (Math.PI*2) != 0.)
		{
			updateBorders();
			updateGravityCenter();
			double angle = -Math.toDegrees(rotationAngle);
			double cx = (gravityCenter.x - position.x) / ppc;
			double cy = (position.y - gravityCenter.y) / ppc;
			x = -Math.cos(-rotationAngle) * cx + Math.sin(-rotationAngle) * cy + cx;
			y = -Math.sin(-rotationAngle) * cx - Math.cos(-rotationAngle) * cy + cy;

			start = start+"\\rput{" + (float)angle + "}(" + LaTeXDrawNumber.getCutNumber((float)x, threshold) + ',' + LaTeXDrawNumber.getCutNumber((float)y, threshold) + "){"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			end = end+"}"; //$NON-NLS-1$
		}
		
		if(!arrowHead1Style.equals(PSTricksConstants.NONEARROW_STYLE) || 
			!arrowHead2Style.equals(PSTricksConstants.NONEARROW_STYLE))
		{
			if(arrowHead2Style.equals(PSTricksConstants.DLARROW_STYLE))
				arrowHead2Style = PSTricksConstants.DRARROW_STYLE;
			else if(arrowHead2Style.equals(PSTricksConstants.DRARROW_STYLE))
				arrowHead2Style = PSTricksConstants.DLARROW_STYLE;
			else if(arrowHead2Style.equals(PSTricksConstants.RARROW_STYLE))
				arrowHead2Style = PSTricksConstants.LARROW_STYLE;
			else if(arrowHead2Style.equals(PSTricksConstants.LARROW_STYLE))
				arrowHead2Style = PSTricksConstants.RARROW_STYLE;
			else if(arrowHead2Style.equals(PSTricksConstants.DLARROW_STYLE))
				arrowHead2Style = PSTricksConstants.DRARROW_STYLE;
			else if(arrowHead2Style.equals(PSTricksConstants.LRBRACKET_STYLE))
				arrowHead2Style = PSTricksConstants.RRBRACKET_STYLE;
			else if(arrowHead2Style.equals(PSTricksConstants.RRBRACKET_STYLE))
				arrowHead2Style = PSTricksConstants.LRBRACKET_STYLE;
			else if(arrowHead2Style.equals(PSTricksConstants.RSBRACKET_STYLE))
				arrowHead2Style = PSTricksConstants.LSBRACKET_STYLE;
			else if(arrowHead2Style.equals(PSTricksConstants.LSBRACKET_STYLE))
				arrowHead2Style = PSTricksConstants.RSBRACKET_STYLE;
			
			String paramsR = ","+ arrowHead1.getParametersCode(); //$NON-NLS-1$
			String paramsL = ","+ arrowHead2.getParametersCode(); //$NON-NLS-1$
			if(paramsR.equals(",")) paramsR = ""; //$NON-NLS-1$ //$NON-NLS-2$
			if(paramsL.equals(",")) paramsL = ""; //$NON-NLS-1$ //$NON-NLS-2$
			
			if(!arrowHead1Style.equals(PSTricksConstants.NONEARROW_STYLE))
			{
				arrowsCode="{"+arrowHead1Style+'-'; //$NON-NLS-1$
				
				if(!arrowHead2Style.equals(PSTricksConstants.NONEARROW_STYLE))
					arrowsCode+=arrowHead2Style;
				arrowsCode+='}';
				
				if(!arrowHead1.isOfTheSameTypeAs(arrowHead2))
				{
					if((arrowHead2Style.equals(PSTricksConstants.LRBRACKET_STYLE) ||
						arrowHead2Style.equals(PSTricksConstants.RRBRACKET_STYLE)  ||
						arrowHead2Style.equals(PSTricksConstants.RSBRACKET_STYLE)  ||
						arrowHead2Style.equals(PSTricksConstants.RSBRACKET_STYLE))	&&
						(arrowHead1Style.equals(PSTricksConstants.BAREND_STYLE) ||
						arrowHead1Style.equals(PSTricksConstants.BARIN_STYLE) ||
						arrowHead1Style.equals(PSTricksConstants.LRBRACKET_STYLE) ||
						arrowHead1Style.equals(PSTricksConstants.RRBRACKET_STYLE)  ||
						arrowHead1Style.equals(PSTricksConstants.RSBRACKET_STYLE)  ||
						arrowHead1Style.equals(PSTricksConstants.RSBRACKET_STYLE)))
						add = paramsR;
					else
						add+= paramsL + paramsR;
				}else add+=paramsR;
			}
			else 
				if(!arrowHead2Style.equals(PSTricksConstants.NONEARROW_STYLE))
				{
					add+=paramsR;
					arrowsCode="{-"+arrowHead2Style+'}'; //$NON-NLS-1$
				}
		}

		String str = getPSTricksCodeLine(ppc);
		if(str.length()>0) add=add+','+str;
		
		if(!getAxesStyleToken().equals(PSTricksConstants.DEFAULT_AXES_STYLE))
			add+=",axesstyle="+getAxesStyleToken();//$NON-NLS-1$
		
		if(!getTicksStyleToken().equals(PSTricksConstants.DEFAULT_TICKS_STYLE))
			add+=",tickstyle="+getTicksStyleToken();//$NON-NLS-1$
		
		if(!getLabelsDisplayedToken().equals(PSTricksConstants.DEFAULT_LABELS_DISPLAYED))
			add+=",labels="+getLabelsDisplayedToken();//$NON-NLS-1$
		
		if(!getTicksDisplayedToken().equals(PSTricksConstants.DEFAULT_LABELS_DISPLAYED))
			add+=",ticks="+getTicksDisplayedToken();//$NON-NLS-1$
		
		if(((float)ticksSize/ppc)!=((float)PSTricksConstants.DEFAULT_TICKS_SIZE))
			add+=",ticksize="+LaTeXDrawNumber.getCutNumber((float)(ticksSize/ppc), threshold)+"cm";//$NON-NLS-1$//$NON-NLS-2$
			
		if(distLabels.x!=0.)
			add+=",dx="+(float)distLabels.x+"cm";//$NON-NLS-1$//$NON-NLS-2$
		
		if(distLabels.y!=0.)
			add+=",dy="+(float)distLabels.y+"cm";//$NON-NLS-1$//$NON-NLS-2$

		if(increment.x!=PSTricksConstants.DEFAULT_DX)
			add+=",Dx="+(((int)increment.x)==increment.x?String.valueOf((int)increment.x): //$NON-NLS-1$
						String.valueOf((float)increment.x));
		
		if(increment.y!=PSTricksConstants.DEFAULT_DY)
			add+=",Dy="+(((int)increment.y)==increment.y?String.valueOf((int)increment.y): //$NON-NLS-1$
						String.valueOf((float)increment.y));

		if(origin.x!=PSTricksConstants.DEFAULT_OX)
			add+=",Ox="+(int)origin.x;//$NON-NLS-1$
		
		if(origin.y!=PSTricksConstants.DEFAULT_OY)
			add+=",Oy="+(int)origin.y;//$NON-NLS-1$

		if(showOrigin!=PSTricksConstants.DEFAULT_SHOW_ORIGIN)
			add+=",showorigin="+showOrigin;//$NON-NLS-1$
		
		return start +"\\psaxes[linewidth="+ LaTeXDrawNumber.getCutNumber((thickness/ppc), threshold) + //$NON-NLS-1$ 
				add + "]" + arrowsCode + coord + end;//$NON-NLS-1$
	}





	@Override
	public Object clone() throws CloneNotSupportedException
	{
		Axe g = (Axe)super.clone();

		g.axesStyle 		= axesStyle;
		g.ticksDisplayed 	= ticksDisplayed;
		g.ticksSize 		= ticksSize;
		g.ticksStyle 		= ticksStyle;
		g.increment 		= (LaTeXDrawPoint2D)increment.clone();
		g.showOrigin 		= showOrigin;
		g.distLabels 		= (LaTeXDrawPoint2D)distLabels.clone();
		g.labelsDisplayed 	= labelsDisplayed;
		g.arrowHead1		= (ArrowHead)arrowHead1.clone();
		g.arrowHead2		= (ArrowHead)arrowHead2.clone();
		
		return g;
	}

	
	
	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException
	{
		gridEnd 		= (LaTeXDrawPoint2D)ois.readObject();
		gridStart 		= (LaTeXDrawPoint2D)ois.readObject();
		position 		= (LaTeXDrawPoint2D)ois.readObject();
		origin 			= (LaTeXDrawPoint2D)ois.readObject();
		isXLabelSouth 	= ois.readBoolean();
		isYLabelWest 	= ois.readBoolean();
		borders 		= (LaTeXDrawRectangle)ois.readObject();
		font 			= (Font)ois.readObject();
		increment		= (LaTeXDrawPoint2D)ois.readObject();
		ticksSize		= ois.readDouble();
		showOrigin		= ois.readBoolean();
		ticksDisplayed	= ois.readInt();
		ticksStyle		= ois.readInt();
		labelsDisplayed	= ois.readInt();
		distLabels		= (LaTeXDrawPoint2D)ois.readObject();
		axesStyle		= ois.readInt();
		arrowHead1		= (ArrowHead)ois.readObject();
		arrowHead2		= (ArrowHead)ois.readObject();
		
		updateFonts();
		updateShape();
	}
	
	


	/**
	 * @return the increment.
	 */
	public synchronized LaTeXDrawPoint2D getIncrement()
	{
		return increment;
	}





	/**
	 * @param increment the X increment to set.
	 */
	public synchronized void setIncrementX(double increment)
	{
		this.increment.x = increment;
	}
	
	
	
	/**
	 * @param increment the Y increment to set.
	 */
	public synchronized void setIncrementY(double increment)
	{
		this.increment.y = increment;
	}




	/**
	 * @return the distLabels.x.
	 */
	public synchronized double getDistLabelsX()
	{
		return distLabels.x;
	}

	
	
	/**
	 * @return the distLabels.y.
	 */
	public synchronized double getDistLabelsY()
	{
		return distLabels.y;
	}


	
	
	/**
	 * @param distLabelsX the distLabels.x to set.
	 */

	public synchronized void setDistLabelsX(double distLabelsX)
	{
		distLabels.x = distLabelsX;
	}
	
	
	
	/**
	 * @param distLabelsY the distLabels.y to set.
	 */

	public void setDistLabelsY(double distLabelsY)
	{
		distLabels.y = distLabelsY;
	}




	/**
	 * @return the labelsDisplayed.
	 */
	public int getLabelsDisplayed()
	{
		return labelsDisplayed;
	}

	
	
	
	/**
	 * Allows to get the token of 'labelsDisplayed' (null if there is a problem).
	 * @return The token.
	 */
	public String getLabelsDisplayedToken()
	{
		if(labelsDisplayed==LABELS_DISPLAYED_ALL_VALUE)
			return PSTricksConstants.TOKEN_LABELS_DISPLAYED_ALL;
		
		if(labelsDisplayed==LABELS_DISPLAYED_NON_VALUE)
			return PSTricksConstants.TOKEN_LABELS_DISPLAYED_NON;
		
		if(labelsDisplayed==LABELS_DISPLAYED_X_VALUE)
			return PSTricksConstants.TOKEN_LABELS_DISPLAYED_X;
		
		if(labelsDisplayed==LABELS_DISPLAYED_Y_VALUE)
			return PSTricksConstants.TOKEN_LABELS_DISPLAYED_Y;
		
		return null;
	}

	




	/**
	 * @param labelsDisplayed the labelsDisplayed to set.
	 */
	public synchronized void setLabelsDisplayed(short labelsDisplayed)
	{
		this.labelsDisplayed = labelsDisplayed;
	}

	


	/**
	 * @param labelsDisplayed the labelsDisplayed to set.
	 */
	public synchronized void setLabelsDisplayedToken(String labelsDisplayed)
	{
		if(labelsDisplayed.equals(PSTricksConstants.TOKEN_LABELS_DISPLAYED_ALL))
			this.labelsDisplayed = LABELS_DISPLAYED_ALL_VALUE;
		else
			if(labelsDisplayed.equals(PSTricksConstants.TOKEN_LABELS_DISPLAYED_NON))
				this.labelsDisplayed = LABELS_DISPLAYED_NON_VALUE;
			else
				if(labelsDisplayed.equals(PSTricksConstants.TOKEN_LABELS_DISPLAYED_X))
					this.labelsDisplayed = LABELS_DISPLAYED_X_VALUE;
				else
					if(labelsDisplayed.equals(PSTricksConstants.TOKEN_LABELS_DISPLAYED_Y))
						this.labelsDisplayed = LABELS_DISPLAYED_Y_VALUE;
	}
	


	/**
	 * @return the showOrigin.
	 */
	public synchronized boolean isShowOrigin()
	{
		return showOrigin;
	}





	/**
	 * @param showOrigin the showOrigin to set.
	 */
	public synchronized void setShowOrigin(boolean showOrigin)
	{
		this.showOrigin = showOrigin;
	}





	/**
	 * @return the ticksDisplayed.
	 */
	public synchronized int getTicksDisplayed()
	{
		return ticksDisplayed;
	}

	
	
	
	/**
	 * Allows to get the token of 'ticksDisplayed' (null if there is a problem).
	 * @return The token.
	 */
	public synchronized String getTicksDisplayedToken()
	{
		if(ticksDisplayed==LABELS_DISPLAYED_ALL_VALUE)
			return PSTricksConstants.TOKEN_LABELS_DISPLAYED_ALL;
		
		if(ticksDisplayed==LABELS_DISPLAYED_NON_VALUE)
			return PSTricksConstants.TOKEN_LABELS_DISPLAYED_NON;
		
		if(ticksDisplayed==LABELS_DISPLAYED_X_VALUE)
			return PSTricksConstants.TOKEN_LABELS_DISPLAYED_X;
		
		if(ticksDisplayed==LABELS_DISPLAYED_Y_VALUE)
			return PSTricksConstants.TOKEN_LABELS_DISPLAYED_Y;
		
		return null;
	}
	
	


	/**
	 * @param ticksDisplayed the ticksDisplayed to set.
	 */
	public synchronized void setTicksDisplayed(short ticksDisplayed)
	{
		this.ticksDisplayed = ticksDisplayed;
	}



	
	/**
	 * @param ticksDisplayed the ticksDisplayed to set.
	 */
	public synchronized void setTicksDisplayedToken(String ticksDisplayed)
	{
		if(ticksDisplayed.equals(PSTricksConstants.TOKEN_LABELS_DISPLAYED_ALL))
			this.ticksDisplayed = LABELS_DISPLAYED_ALL_VALUE;
		else
			if(ticksDisplayed.equals(PSTricksConstants.TOKEN_LABELS_DISPLAYED_NON))
				this.ticksDisplayed = LABELS_DISPLAYED_NON_VALUE;
			else
				if(ticksDisplayed.equals(PSTricksConstants.TOKEN_LABELS_DISPLAYED_X))
					this.ticksDisplayed = LABELS_DISPLAYED_X_VALUE;
				else
					if(ticksDisplayed.equals(PSTricksConstants.TOKEN_LABELS_DISPLAYED_Y))
						this.ticksDisplayed = LABELS_DISPLAYED_Y_VALUE;
	}
	
	


	/**
	 * @return the ticksStyle.
	 */
	public synchronized int getTicksStyle()
	{
		return ticksStyle;
	}





	/**
	 * @param ticksStyle the ticksStyle to set.
	 */
	public synchronized void setTicksStyle(short ticksStyle)
	{
		this.ticksStyle = ticksStyle;
	}

	
	
	
	/**
	 * @param ticksStyle the ticksStyle to set.
	 */
	public synchronized void setTicksStyleToken(String ticksStyle)
	{
		if(ticksStyle.equals(PSTricksConstants.TOKEN_TICKS_STYLE_BOTTOM))
			this.ticksStyle = TICKS_STYLE_BOTTOM_VALUE;
		else
			if(ticksStyle.equals(PSTricksConstants.TOKEN_TICKS_STYLE_FULL))
				this.ticksStyle = TICKS_STYLE_FULL_VALUE;
			else
				if(ticksStyle.equals(PSTricksConstants.TOKEN_TICKS_STYLE_TOP))
					this.ticksStyle = TICKS_STYLE_TOP_VALUE;
	}

	
	
	
	/**
	 * Allows to get the token of the ticks style (null if there is a problem).
	 * @return The token.
	 */
	public synchronized String getTicksStyleToken()
	{
		if(ticksStyle==TICKS_STYLE_BOTTOM_VALUE)
			return PSTricksConstants.TOKEN_TICKS_STYLE_BOTTOM;
		
		if(ticksStyle==TICKS_STYLE_FULL_VALUE)
			return PSTricksConstants.TOKEN_TICKS_STYLE_FULL;
		
		if(ticksStyle==TICKS_STYLE_TOP_VALUE)
			return PSTricksConstants.TOKEN_TICKS_STYLE_TOP;
		
		return null;
	}

	
	
	
	/**
	 * @return the ticksSize.
	 */
	public synchronized double getTicksSize()
	{
		return ticksSize;
	}





	/**
	 * @param ticksSize the ticksSize to set.
	 */
	public synchronized void setTicksSize(double ticksSize)
	{
		this.ticksSize = ticksSize;
	}





	/**
	 * @return the axesStyle.
	 */
	public synchronized int getAxesStyle()
	{
		return axesStyle;
	}


	
	
	/**
	 * Allows to get the token of the axes style (null if there is a problem).
	 * @return The token.
	 */
	public synchronized String getAxesStyleToken()
	{
		if(axesStyle==AXES_STYLE_AXES_VALUE)
			return PSTricksConstants.TOKEN_AXES_STYLE_AXES;
		
		if(axesStyle==AXES_STYLE_FRAME_VALUE)
			return PSTricksConstants.TOKEN_AXES_STYLE_FRAME;
		
		if(axesStyle==AXES_STYLE_NONE_VALUE)
			return PSTricksConstants.TOKEN_AXES_STYLE_NONE;
		
		return null;
	}



	/**
	 * @param axesStyle the axesStyle to set.
	 */
	public synchronized void setAxesStyle(short axesStyle)
	{
		this.axesStyle = axesStyle;
	}


	
	
	/**
	 * @param axesStyle the axesStyle to set.
	 */
	public synchronized void setAxesStyleToken(String axesStyle)
	{
		if(axesStyle.equals(PSTricksConstants.TOKEN_AXES_STYLE_AXES))
			this.axesStyle = AXES_STYLE_AXES_VALUE;
		else
			if(axesStyle.equals(PSTricksConstants.TOKEN_AXES_STYLE_FRAME))
				this.axesStyle = AXES_STYLE_FRAME_VALUE;
			else
				if(axesStyle.equals(PSTricksConstants.TOKEN_AXES_STYLE_NONE))
					this.axesStyle = AXES_STYLE_NONE_VALUE;
	}
	
	


	@Override
	public void updateBorders(FontMetrics fm)
	{
		updateBorders();
	}




	public String getArrow1Style()
	{
		return arrowHead1.getArrowStyle();
	}




	public String getArrow2Style()
	{
		return arrowHead2.getArrowStyle();
	}




	public ArrowHead getArrowHead1()
	{
		return arrowHead1;
	}




	public ArrowHead getArrowHead2()
	{
		return arrowHead2;
	}




	public void setArrow1Style(String style)
	{
		arrowHead1.setArrowStyle(style);
	}




	public void setArrow2Style(String style)
	{
		arrowHead2.setArrowStyle(style);
	}




	/**
	 * @return A combobox allowing to choice the style of the axes.
	 */
	public static JComboBox createAxeStyleChoice()
	{
		JComboBox choice = new LaTeXDrawComboBox();

		choice.addItem(LABEL_STYLE_AXE);
		choice.addItem(LABEL_STYLE_FRAME);
		choice.addItem(LABEL_STYLE_NONE);
		
		choice.setActionCommand(ACTION_CMD_AXE_STYLE_CHOICE);
		choice.setName(ACTION_CMD_AXE_STYLE_CHOICE);
		choice.setMinimumSize(new Dimension(70,30));
		choice.setPreferredSize(new Dimension(70,30));
		choice.setMaximumSize(new Dimension(70,30));
		
		return choice;
	}
	
	
	
	
	/**
	 * @return A combobox allowing to choice the style of the ticks.
	 */
	public static JComboBox createTicksStyleChoice()
	{
		JComboBox choice = new LaTeXDrawComboBox();
		
		choice.addItem(LABEL_STYLE_TICKS_FULL);
		choice.addItem(LABEL_STYLE_TICKS_TOP);
		choice.addItem(LABEL_STYLE_TICKS_BOTTOM);
		
		choice.setActionCommand(ACTION_CMD_TICKS_STYLE_CHOICE);
		choice.setName(ACTION_CMD_TICKS_STYLE_CHOICE);
		choice.setMinimumSize(new Dimension(70,30));
		choice.setPreferredSize(new Dimension(70,30));
		choice.setMaximumSize(new Dimension(70,30));
		
		return choice;
	}




	@Override
	public void updateToGrid(MagneticGrid grid)
	{
		position.setLocation(grid.getTransformedPointToGrid(position, false));
		updateBorders();
		updateShape();
	}
	
	
	@Override
	public int hashCode()
	{
		return super.hashCode()^4;
	}
	
	
	
	public boolean hasTwoLeftArrows()
	{
		return true;
	}
}




