/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: wdelattr.c,v 1.1 2001/02/03 20:26:17 jsquyres Exp $
 *
 *	Function:	- delete attribute
 *			- detach attribute from window
 *	Accepts:	- window
 *			- attribute key
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <errno.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>


/*@
    MPI_Win_delete_attr - Delete an attribute

Input Parameters:
+ win - Window from which the attribute is deleted
- key - Key value

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_WIN
.N MPI_ERR_KEYVAL
.N MPI_ERR_ARG

.seealso MPI_Win_create_keyval(3), MPI_Win_free_keyval(3), MPI_Win_set_attr(3), MPI_Win_get_attr(3)

.N WEB
@*/
int
MPI_Win_delete_attr(MPI_Win win, int key)
{
	struct _attrkey	*p;
	struct _attr	*pk;
	int		err;

	lam_initerr_m();
	lam_setfunc_m(BLKMPIWINDELETEATTR);
/*
 * Check the arguments.
 */
	if (win == MPI_WIN_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIWINDELETEATTR, lam_mkerr(MPI_ERR_WIN, 0)));
	}

	p = lam_getattr(key);
	if (p == 0) {
		return(lam_errfunc(win->w_comm,
			BLKMPIWINDELETEATTR, lam_mkerr(MPI_ERR_KEYVAL, 0)));
	}
/*
 * Check if the key is attached to the window.
 */
	pk = lam_getkey(win->w_keys, key);
	if (pk == 0) {
		return(lam_errfunc(win->w_comm,
			BLKMPIWINDELETEATTR, lam_mkerr(MPI_ERR_KEYVAL, 0)));
	}
/*
 * Delete the attribute via the callback function.
 */
	if (((MPI_Win_delete_attr_function *) (p->ak_del))
			!= MPI_WIN_NULL_DELETE_FN) {

		if (p->ak_flags & LAM_LANGF77) {
			(*((MPI_F_delete_function *)(p->ak_del)))
				(&win->w_f77handle, &key,
				(int *) &(pk->a_value),
				(int *) &(p->ak_extra), &err);
		} else {
			err = (*((MPI_Win_delete_attr_function *) (p->ak_del)))
				(win, key, pk->a_value, p->ak_extra);
		}

		if (err != MPI_SUCCESS) {
			return(lam_err_win(win, err, 0, "deleting attribute"));
		}
	}
/*
 * Detach the key from the window.
 */
	if (lam_delkey(win->w_keys, key)) {
		return(lam_err_win(win, MPI_ERR_INTERN, errno, "deleting key"));
	}

	lam_resetfunc_m(BLKMPIWINDELETEATTR);
	return(MPI_SUCCESS);
}
