/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/GDB/NJN
 *
 *	$Id: testall.c,v 6.4 1999/12/31 21:27:39 jsquyres Exp $
 *
 *	Function:	- check if all requests completed
 *	Accepts:	- # of requests
 *			- array of requests
 *			- flag (out)
 *			- array of status structures (modified)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>


/*@
    MPI_Testall - Tests for the completion of all previously initiated
    communications

Input Parameters:
+ count - lists length (integer) 
- reqs - array of requests (array of handles) 

Output Parameters:
+ flag - (logical) 
- stats - array of status objects (array of Status), which can be the
MPI constant 'MPI_STATUSES_IGNORE'

Notes:

'flag' is true (1) only if all requests have completed.  Otherwise,
'flag' is false (0) and neither 'reqs' nor 'stats' is modified.

.N waitstatus

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COUNT
.N MPI_ERR_ARG
.N MPI_ERR_IN_STATUS

.N ACK
@*/
int MPI_Testall(int count, MPI_Request *reqs, 
		int *flag, MPI_Status *stats)
{
	int		i;			/* favourite index */
	int		fl_done;		/* request done? */
	int		err;			/* error code */
	int		fl_trace;		/* do tracing? */
	MPI_Request	*req;			/* ptr request */
	MPI_Status	*stat;			/* ptr status */

	lam_initerr_m();
	lam_setfunc_m(BLKMPITESTALL);

	if (count < 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPITESTALL,
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if ((count > 0) && (reqs == 0)) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPITESTALL,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if (flag == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPITESTALL,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Generate a start wrapper trace.
 */
	fl_trace = ((_kio.ki_rtf & RTF_TRON) == RTF_TRON);

	if (fl_trace) {
		lam_tr_wrapstart(BLKMPITESTALL);
	}
/*
 * Initialize all status members.
 */
	if (stats != MPI_STATUSES_IGNORE) {
		for (i = 0, stat = stats; i < count; ++i, ++stat) {
			lam_emptystat(stat);
		}
	}
/*
 * Loop testing all requests non-destructively.
 * Stop at the first request not done.
 */
	req = reqs;
	stat = stats;

	for (i = 0; i < count; ++i, ++req) {

		err = lam_test(req, 0, &fl_done, stat);

		if (err != MPI_SUCCESS) {
			stat->MPI_ERROR = err;

			if (fl_trace) {
				lam_tr_wrapend(BLKMPITESTALL);
			}

			return(lam_errfunc(MPI_COMM_WORLD, BLKMPITESTALL,
					lam_mkerr(MPI_ERR_IN_STATUS, 0)));
		}

		if (fl_done == 0) {
			*flag = 0;

			if (fl_trace) {
				lam_tr_wrapend(BLKMPITESTALL);
			}

			lam_resetfunc_m(BLKMPITESTALL);
			return(MPI_SUCCESS);
		}

		if (stats != MPI_STATUSES_IGNORE) {
			stat++;
		}
	}
/*
 * All active requests are done, loop doing the job.
 */
	req = reqs;
	stat = stats;

	for (i = 0; i < count; ++i, ++req) {
		err = lam_test(req, 1, &fl_done, stat);

		if (err != MPI_SUCCESS) {
			stat->MPI_ERROR = err;

			if (fl_trace) {
				lam_tr_wrapend(BLKMPITESTALL);
			}

			return(lam_errfunc(MPI_COMM_WORLD, BLKMPITESTALL,
					lam_mkerr(MPI_ERR_IN_STATUS, 0)));
		}

		if (stats != MPI_STATUSES_IGNORE) {
			stat++;
		}
	}

	*flag = 1;
/*
 * Generate an end wrapper trace.
 */
	if (fl_trace) {
		lam_tr_wrapend(BLKMPITESTALL);
	}

	lam_resetfunc_m(BLKMPITESTALL);
	return(MPI_SUCCESS);
}
