/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN/JRV/RBD
 *
 *	$Id: scan.c,v 6.5 1999/12/31 21:27:36 jsquyres Exp $
 *
 *	Function:	- partial reduce from 0 to rank i, at each rank i
 *	Accepts:	- send buffer
 *			- receive buffer
 *			- count of elements
 *			- type of elements
 *			- operation to perform
 *			- communicator
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <stdlib.h>

#include <lam_config.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Scan - Computes the scan (partial reductions) of data on a collection of
           processes

Input Parameters:
+ sbuf - starting address of send buffer (choice) 
. count - number of elements in input buffer (integer) 
. dtype - data type of elements of input buffer (handle) 
. op - operation (handle) 
- comm - communicator (handle) 

Output Parameter:
. rbuf - starting address of receive buffer (choice) 

.N IMPI

.N fortran

.N collops

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_INTERCOMM
.N MPI_ERR_IMPI
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_OP
.N MPI_ERR_BUFFER
.N MPI_ERR_BUFFER_ALIAS

.N ACK
@*/
int MPI_Scan(void *sbuf, void *rbuf, int count, 
	     MPI_Datatype dtype, MPI_Op op, MPI_Comm comm)
{
	int		size;			/* group size */
	int		rank;			/* my rank */
	int		err;			/* error code */
	char		*tmpbuf = 0;		/* temporary buffer */
	char		*origin;		/* data origin in temp buffer */
	MPI_Status	stat;			/* recv status */

	lam_initerr_m();
	lam_setfunc_m(BLKMPISCAN);
/*
 * Check for invalid arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return (lam_errfunc(comm, BLKMPISCAN,
				lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (dtype == MPI_DATATYPE_NULL) {
		return(lam_errfunc(comm, BLKMPISCAN,
				lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if (count < 0) {
		return(lam_errfunc(comm, BLKMPISCAN,
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if (op == MPI_OP_NULL) {
		return(lam_errfunc(comm, BLKMPISCAN,
				lam_mkerr(MPI_ERR_OP, 0)));
	}


#if LAM_WANT_IMPI

	/* Remove this when IMPI collectives are implemented */

        if (LAM_IS_IMPI(comm)) {
	  return lam_err_comm(comm, MPI_ERR_COMM, 0, 
			      "Collectives not yet implemented on IMPI communicators");
	}
#endif

	LAM_TRACE(lam_tr_cffstart(BLKMPISCAN));
/*
 * Check for zero count case.
 */
	if (count == 0) {
		LAM_TRACE(lam_tr_cffend(BLKMPISCAN, -1, comm, dtype, count));

		lam_resetfunc_m(BLKMPISCAN);
		return(MPI_SUCCESS);
	}
/*
 * Initialize.
 */
	MPI_Comm_rank(comm, &rank);
	MPI_Comm_size(comm, &size);
/*
 * Switch to collective communicator.
 */
	lam_mkcoll(comm);
/*
 * If I'm rank 0, initialize the recv buffer.
 */
	if (rank == 0) {
		err = lam_dtsndrcv(sbuf, count, dtype,
				rbuf, count, dtype, BLKMPISCAN, comm);
		if (err != MPI_SUCCESS) {
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPISCAN, err));
		}
	}
/*
 * Otherwise receive previous buffer and reduce.
 */
	else {
		if (!op->op_commute) {
/*
 * Allocate a temporary buffer.
 */
			err = lam_dtbuffer(dtype, count, &tmpbuf, &origin);
			if (err != MPI_SUCCESS) {
				lam_mkpt(comm);
				return(lam_errfunc(comm, BLKMPISCAN, err));
			}
/*
 * Copy the send buffer into the receive buffer.
 */
			err = lam_dtsndrcv(sbuf, count, dtype, rbuf,
					count, dtype, BLKMPISCAN, comm);
			
			if (err != MPI_SUCCESS) {
				if (tmpbuf) free(tmpbuf);
				lam_mkpt(comm);
				return(lam_errfunc(comm, BLKMPISCAN, err));
			}
			
			err = MPI_Recv(origin, count, dtype,
					rank - 1, BLKMPISCAN, comm, &stat);
		}
		else {
			origin = sbuf;
			
			err = MPI_Recv(rbuf, count, dtype,
					rank - 1, BLKMPISCAN, comm, &stat);
		}

		if (err != MPI_SUCCESS) {
			if (tmpbuf) free(tmpbuf);
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPISCAN, err));
		}

		if (op->op_flags & LAM_LANGF77) {
			(op->op_func)
				(origin, rbuf, &count, &dtype->dt_f77handle);
		} else {
			(op->op_func)(origin, rbuf, &count, &dtype);
		}

		if (tmpbuf) free(tmpbuf);
	}
/*
 * Send result to next process.
 */
	if (rank < (size - 1)) {
		err = MPI_Send(rbuf, count, dtype, rank + 1, BLKMPISCAN, comm);
		if (err != MPI_SUCCESS) {
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPISCAN, err));
		}

	}

	lam_mkpt(comm);
	LAM_TRACE(lam_tr_cffend(BLKMPISCAN, -1, comm, dtype, count));
	lam_resetfunc_m(BLKMPISCAN);
	return(MPI_SUCCESS);
}
