/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD
 *
 *	$Id: ibsend.c,v 6.5 1999/12/31 21:27:29 jsquyres Exp $
 *
 *	Function:	- immediate buffered send
 *	Accepts:	- buffer
 *			- count
 *			- datatype
 *			- destination
 *			- tag
 *			- communicator
 *			- request (out)
 *	Returns: 	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>


/*
 * external functions
 */
extern void		lam_initerr();
extern void		lam_resetfunc();
extern void		lam_setfunc();
extern int		lam_errfunc();
extern int		_mpi_req_build();

/*@
    MPI_Ibsend - Starts a nonblocking buffered send

Input Parameters:
+ buf - initial address of send buffer (choice) 
. count - number of elements in send buffer (integer) 
. dtype - datatype of each send buffer element (handle) 
. dest - rank of destination (integer) 
. tag - message tag (integer) 
- comm - communicator (handle) 

Output Parameter:
. preq - communication request (handle) 

Notes:

It is generally a bad idea to use the 'MPI_Ibsend' function, as it
guarantees that the entire message will suffer the overhead of an
additional memory copy.  For large messages, or when shared memory
message transports are being used, this overhead can be quite
expensive.

The communication started with this function is not guaranteed to
progress or complete until 'req' has been given to one of the test or
wait functions ('MPI_Test', 'MPI_Testall', 'MPI_Testany',
'MPI_Testsome', 'MPI_Wait', 'MPI_Waitall', 'MPI_Waitany',
'MPI_Waitsome').

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_TAG
.N MPI_ERR_RANK
.N MPI_ERR_BUFFER

.seealso MPI_Bsend, MPI_Test, MPI_Testall, MPI_Testany, MPI_Testsome, MPI_Wait, MPI_Waitall, MPI_Waitany, MPI_Waitsome

.N ACK
@*/
int MPI_Ibsend(void *buf, int count, MPI_Datatype dtype, 
	       int dest, int tag, MPI_Comm comm, 
	       MPI_Request *preq)
{
	int		err;

	lam_initerr_m();
	lam_setfunc_m(BLKMPIIBSEND);
/*
 * Check the tag.
 */
	if (tag < 0 || tag > LAM_MAXTAG) {
		return(lam_errfunc(comm, BLKMPIIBSEND,
				lam_mkerr(MPI_ERR_TAG, 0)));
	}
/*
 * Create the fake user request.  We do not add it into the
 * request list.  This is an exception to the design that says
 * that all active (inc. done) requests are to be found in the
 * request list but fake requests, really ibsend requests are
 * very exceptional MPI creatures.
 */
	*preq = MPI_REQUEST_NULL;
	err = _mpi_req_build(buf, count, dtype, dest, tag, comm,
			LAM_RQIFAKE, preq);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIBSEND, err));
	}

	(*preq)->rq_status.MPI_ERROR = MPI_SUCCESS;
	(*preq)->rq_status.MPI_TAG = MPI_UNDEFINED;
	(*preq)->rq_status.MPI_SOURCE = MPI_PROC_NULL;
	(*preq)->rq_state = LAM_RQSDONE;
/*
 * Bsend() the message.
 */
	err = MPI_Bsend(buf, count, dtype, dest, tag, comm);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIBSEND, err));
	}

	lam_resetfunc_m(BLKMPIIBSEND);
	return(MPI_SUCCESS);
}
