/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1995 The Ohio State University
 *	GDB/RBD
 *
 *	$Id: couter.c,v 1.1 2000/02/08 09:32:38 jsquyres Exp $
 *
 *	Function:	- client protocols
 *			- OTB version
 */

#include <lam_config.h>

#include <errno.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>

#include <kio.h>
#include <kreq.h>
#include <terror.h>
#include <typical.h>

/*
 * local functions
 */
static void		cipc_preempt();

/*
 * external functions
 */
extern int		_cio_kreq();		/* kernel request/reply */
extern int		_cio_kreqfront();	/* kernel request */
extern int		_cio_kreqback();	/* kernel reply */
extern int		_cio_init();		/* init kernel interface */
extern int		_cio_send();		/* transfer kernel msg */
extern int		_cio_recv();		/* transfer kernel msg */
extern void		_cio_cleanup();		/* frees resources */
extern void		_ksig_follow();		/* check signals */
extern void		(*_lam_signal())();	/* portable signal() */

/*
 * external variables
 */
extern struct kio_t	_kio;			/* Kernel I/O */

/*
 *	_cipc_init
 *
 *	Function:	- initializes OTB client interface
 *	Returns:	- 0 or ERROR
 */
int
_cipc_init()

{
	if (_cio_init()) return(LAMERROR);
/*
 * Vector LAM_SIGUSR to preemption routine.
 */
	if (_lam_signal(LAM_SIGUSR, cipc_preempt) == SIG_ERR) return(LAMERROR);

	return(0);
}

/*
 *	_cipc_ksend
 *
 *	Function:	- OTB kernel message send
 *			- makes KQSEND request to kernel
 *	Accepts:	- kernel request ptr
 *			- kernel reply ptr
 *	Returns:	- 0 or ERROR
 */
int
_cipc_ksend(pkq, pkr)

struct kreq		*pkq;
struct kreply		*pkr;

{
	int		r;
	sigset_t	sigs_preempt;

	if (_kio.ki_pid != getpid()) {
		errno = ENOTATTACHED;
		return(LAMERROR);
	}

	memcpy((char *) pkq->kq_fyi, (char *) _kio.ki_fyi,
			sizeof(_kio.ki_fyi));

	sigemptyset(&sigs_preempt);
	sigaddset(&sigs_preempt, LAM_SIGUSR);
	sigprocmask(SIG_BLOCK, &sigs_preempt, (sigset_t *) 0);

	if (_cio_kreq(pkq, pkr)) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	if (pkr->kr_reply) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(0);
	}

	pkq->kq_msg.k_length = pkr->kr_length;
	r = _cio_send(&pkq->kq_msg);
	sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);

	return(r);
}

/*
 *	_cipc_krecvfront
 *
 *	Function:	- OTB kernel message receive
 *			- makes KQRECV request to kernel
 *	Accepts:	- kernel request ptr
 *	Returns:	- client/kernel comm. handle or ERROR
 */
int
_cipc_krecvfront(pkq)

struct kreq		*pkq;

{
	sigset_t	sigs_preempt;
	int		r;

	if (_kio.ki_pid != getpid()) {
		errno = ENOTATTACHED;
		return(LAMERROR);
	}

	memcpy((char *) pkq->kq_fyi, (char *) _kio.ki_fyi,
			sizeof(_kio.ki_fyi));

	sigemptyset(&sigs_preempt);
	sigaddset(&sigs_preempt, LAM_SIGUSR);
	sigprocmask(SIG_BLOCK, &sigs_preempt, (sigset_t *) 0);

	if ((r = _cio_kreqfront(pkq)) < 0) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	return(r);
}

/*
 *	_cipc_krecvback
 *
 *	Function:	- OTB kernel message receive
 *			- receives reply for KQRECV from kernel
 *			- transfers message
 *	Accepts:	- kernel request ptr
 *			- kernel reply ptr
 *	Returns:	- 0 or ERROR
 */
int
_cipc_krecvback(pkq, pkr)

struct kreq		*pkq;
struct kreply		*pkr;

{
	int		r;
	sigset_t	sigs_preempt;

	sigemptyset(&sigs_preempt);
	sigaddset(&sigs_preempt, LAM_SIGUSR);

	if (_cio_kreqback(pkr)) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	if (pkr->kr_reply) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(0);
	}

	pkq->kq_msg.k_length = pkr->kr_length;
	r = _cio_recv(&pkq->kq_msg);
	sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);

	return(r);
}

/*
 *	_cipc_ksrfront
 *
 *	Function:	- OTB kernel message send/receive front end
 *			- makes KQSEND followed by KQRECV request to kernel
 *	Accepts:	- kernel request ptr
 *			- kernel reply ptr
 *	Returns:	- 0 or client/kernel socket
 */
int
_cipc_ksrfront(pkq, pkr)

struct kreq		*pkq;
struct kreply		*pkr;

{
	sigset_t	sigs_preempt;
	int		r;

	if (_kio.ki_pid != getpid()) {
		errno = ENOTATTACHED;
		return(LAMERROR);
	}

	memcpy((char *) pkq->kq_fyi, (char *) _kio.ki_fyi,
			sizeof(_kio.ki_fyi));

	sigemptyset(&sigs_preempt);
	sigaddset(&sigs_preempt, LAM_SIGUSR);
	sigprocmask(SIG_BLOCK, &sigs_preempt, (sigset_t *) 0);

	if ((r = _cio_kreqfront(pkq)) < 0) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	if (_cio_kreqback(pkr) < 0) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	if (pkr->kr_reply) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(0);
	}

	pkq->kq_msg.k_length = pkr->kr_length;

	if (_cio_send(&pkq->kq_msg)) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	return(r);
}

/*
 *	_cipc_ksrback
 *
 *	Function:	- OTB kernel message send/receive back end
 */
int
_cipc_ksrback(pkq, pkr)

struct kreq		*pkq;
struct kreply		*pkr;

{
	int		r;
	sigset_t	sigs_preempt;

	sigemptyset(&sigs_preempt);
	sigaddset(&sigs_preempt, LAM_SIGUSR);

	if (_cio_kreqback(pkr)) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(LAMERROR);
	}

	if (pkr->kr_reply) {
		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
		return(0);
	}

	pkq->kq_msg2.k_length = pkr->kr_length;

	r = _cio_recv(&pkq->kq_msg2);
	sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);

	return(r);
}

/*
 *	_cipc_kreq
 *
 *	Function:	- OTB kernel request
 *			- makes request to kernel
 *	Accepts:	- kernel request ptr
 *			- kernel reply ptr
 *	Returns:	- 0 or ERROR
 */
int
_cipc_kreq(pkq, pkr)

struct kreq		*pkq;
struct kreply		*pkr;

{
	sigset_t	sigs_preempt;

	if (_kio.ki_pid != getpid()) {
		errno = ENOTATTACHED;
		return(LAMERROR);
	}

	memcpy((char *) pkq->kq_fyi, (char *) _kio.ki_fyi,
			sizeof(_kio.ki_fyi));

	do {
		sigemptyset(&sigs_preempt);
		sigaddset(&sigs_preempt, LAM_SIGUSR);
		sigprocmask(SIG_BLOCK, &sigs_preempt, (sigset_t *) 0);

		if (_cio_kreq(pkq, pkr)) {
			sigprocmask(SIG_UNBLOCK, &sigs_preempt, 
					(sigset_t *) 0);
			return(LAMERROR);
		}

		sigprocmask(SIG_UNBLOCK, &sigs_preempt, (sigset_t *) 0);
/*
 * Do we need to follow a signal?
 */
		if (pkr->kr_signal) {
			_kio.ki_signal |= pkr->kr_signal;
			_ksig_follow();
		}
	} while (pkr->kr_reply == EINTR);

	return(0);
}

/*
 *	_cipc_cleanup
 *
 *	Function:	- frees up client resources
 */
void
_cipc_cleanup()

{
	_cio_cleanup();
}

/*
 *	cipc_preempt
 *
 *	Function:	- preempts myself
 *			- signal handler for LAM_SIGUSR
 */
static void
cipc_preempt()

{
	struct kreq	req;			/* kernel request */
	struct kreply	reply;			/* kernel reply */

	if (_kio.ki_pid != getpid()) return;

	req.kq_req = KQSURRENDER;
	req.kq_index = _kio.ki_index;

	if (_cio_kreq(&req, &reply)) return;

	if (reply.kr_signal) {
		_kio.ki_signal |= reply.kr_signal;
		_ksig_follow();
	}
/*
 * Vector LAM_SIGUSR to preemption routine.
 */
	if (_lam_signal(LAM_SIGUSR, cipc_preempt) == SIG_ERR) return;
}
