/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: wcomplete.c,v 1.0 1999/07/21 19:00:19 jsquyres Exp $
 *
 *	Function:	- complete a window
 *	Accepts:	- window
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <errno.h>
#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>


int
MPI_Win_complete(win)

MPI_Win			win;

{
	MPI_Comm	comm;			/* window's communicator */
	MPI_Request	*reqs;			/* completing message reqs */
	MPI_Request	*r;
	int		complete_msg[1];	/* complete message */
	int		err;			/* error code */
	int		i;

	lam_initerr_m();
	lam_setfunc_m(BLKMPIWINCOMPLETE);
/*
 * Check arguments.
 */
	if (win == MPI_WIN_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIWINCOMPLETE, lam_mkerr(MPI_ERR_WIN, 0)));
	}

	comm = win->w_comm;

	if (!(win->w_flags & LAM_WFSTARTED)) {
		return(lam_errfunc(comm,
			BLKMPIWINCOMPLETE, lam_mkerr(MPI_ERR_EPOCH, 0)));
	}
/*
 * Send "i-am-completing" mesages to all target processes in the 2 party
 * synchronization.  
 */
	reqs = (MPI_Request *) malloc(win->w_ngroup * sizeof(MPI_Request));
	if (reqs == 0) {
		return(lam_errfunc(comm,
			BLKMPIWINCOMPLETE, lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	complete_msg[0] = LAM_OSD_COMPLETE;

	for (i = 0, r = reqs; i < comm->c_group->g_nprocs; i++) {

		if (win->w_pstate[i] & LAM_WFSTARTED) {
			err = MPI_Send_init(complete_msg, 1, MPI_INT, i,
						LAM_OSD_HEADER, comm, r++);

			if (err != MPI_SUCCESS) {
				return(lam_errfunc(comm,
						BLKMPIWINCOMPLETE, err));
			}

			win->w_pstate[i] &= ~LAM_WFSTARTED;
		}
	}

	err = MPI_Startall(win->w_ngroup, reqs);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(win->w_comm, BLKMPIWINCOMPLETE, err));
	}
/*
 * Complete all locally originated one-sided requests in the window.
 */
	err = lam_osd_complete(win, LAM_RQFOSORIG);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWINCOMPLETE, err));
	}
/*
 * Complete the "i-am-completing" messages.
 */
	err = MPI_Waitall(win->w_ngroup, reqs, MPI_STATUSES_IGNORE);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWINCOMPLETE, err));
	}
	free((char *) reqs);

	win->w_ngroup = 0;
	win->w_flags &= ~LAM_WFSTARTED;

        lam_resetfunc_m(BLKMPIWINCOMPLETE);
	return(MPI_SUCCESS);
}
