/***************************************************************************
                          xmltreeview.cpp  -  description
                             -------------------
    begin                : Tue Jul 10 2001
    copyright            : (C) 2001 by The KXMLEditor Team
    email                : lvanek@eanet.cz, olaf@punkbands.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "xmltreeview.h"

// include files for Qt
#include <qheader.h>
#include <qdragobject.h>
#include <qtimer.h>

// include files for KDE
#include <kglobalsettings.h>
#include <kmessagebox.h>
#include <klocale.h>

// application specific includes
#include "xmlstructureparser.h"
#include "xml_base.h"
#include "optionsdialog.h"
#include "kxmleditordoc.h"

static const int autoOpenTimeout = 750;

XmlTreeView::XmlTreeView(QWidget *parent, const char *name, KXMLEditorDoc *pDoc)
	: KListView(parent, name)
{
	m_pDoc = pDoc;
	
	setSorting(-1); // no sorting
	addColumn(i18n("Qualified name"));
	addColumn(i18n("Namespace"));
	//xmlTreeView->setColumnWidthMode(1, QListView::Maximum);
	
	if(OptionsDialog::getTreeViewElemDisplayMode() > 0)
		addColumn(i18n("Attributes"));
	
	if(OptionsDialog::getTreeViewEnableDragDrop())
		{ setAcceptDrops(true);
  		viewport()->setAcceptDrops(true);
  	}
  	
	m_bDrag = false;
	m_pCurrentBeforeDropItem = 0;
  m_pDropItem = 0;
  setSelectionMode(QListView::Single);
	
  if(OptionsDialog::getTreeViewRootDecorated())
  	setRootIsDecorated(true);
  	
  // --- added by Olaf Hartig (29.Jun.2001)	
  setItemsRenameable(true);
	setRenameable(0, true);
	
  m_autoOpenTimer = new QTimer(this);
  connect(m_autoOpenTimer, SIGNAL(timeout()), this, SLOT(slotAutoOpenFolder()));
}

XmlTreeView::~XmlTreeView()
{
}

/** Overrides KListView::contentsMousePressEvent */
void XmlTreeView::contentsMousePressEvent(QMouseEvent *e)
{
	KListView::contentsMousePressEvent(e);

  QPoint p(contentsToViewport(e->pos()));
  QListViewItem *i = itemAt(p);

  if(e->button() == LeftButton && i)
  	{ // if the user clicked into the root decoration of the item, don't try to start a drag!
      if(p.x() > header()->cellPos(header()->mapToActual(0)) +
             treeStepSize() * ( i->depth() + (rootIsDecorated() ? 1 : 0)) + itemMargin() ||
             p.x() < header()->cellPos(header()->mapToActual(0)))
        {
            m_dragPos = e->pos();
            m_bDrag = true;
        }
    }
}

/** Overrides KListView::contentsMouseMoveEvent */
void XmlTreeView::contentsMouseMoveEvent(QMouseEvent *e)
{
	KListView::contentsMouseMoveEvent(e);

  if(!m_bDrag || (e->pos() - m_dragPos).manhattanLength() <= KGlobalSettings::dndEventDelay())
        return;

  m_bDrag = false;

  QListViewItem *item = itemAt(contentsToViewport(m_dragPos));

  if(!item || !item->isSelectable())
  	return;

	// copy item into clipboard
	XmlTreeItem *pXmlTreeItem = static_cast <XmlTreeItem *> (item);
	QTextDrag *pDrag = copyItem(pXmlTreeItem);
			
	// Start a drag
  const QPixmap *pix = item->pixmap(0);
  if(pix && pDrag->pixmap().isNull())
  	{ QPoint hotspot(pix->width() / 2, pix->height() / 2);
      pDrag->setPixmap(*pix, hotspot);
    }

  pDrag->drag();
}

/** Copy tree item into clipboard */
QTextDrag * XmlTreeView::copyItem(XmlTreeItem *pXmlTreeItem)
{
	QTextDrag *pDrag = 0;
	
	switch(pXmlTreeItem->itemType())
		{ case XmlTreeItem::itemElement:
		  case XmlTreeItem::itemProcInstr:
							{ QString strXML;
				
								pXmlTreeItem->save(strXML, 0);
								pDrag = new QTextDrag(strXML, this);
								pDrag->setSubtype(pXmlTreeItem->contentsTypeAsText());
							}
								break;
								
			case XmlTreeItem::itemText:
			case XmlTreeItem::itemCDATA:
			case XmlTreeItem::itemComment:			
							{	XmlContentsItem* pXmlContentsItem = static_cast<XmlContentsItem *> (pXmlTreeItem);
			
								// Create drag object with contents
  							pDrag = new QTextDrag(pXmlContentsItem->contents(), this);
  							pDrag->setSubtype(pXmlContentsItem->contentsTypeAsText());
							}
								break;

			default:	ASSERT(false);
		}
		
	return pDrag;
}

/** Overrides KListView::contentsMouseReleaseEvent */
void XmlTreeView::contentsMouseReleaseEvent(QMouseEvent *e)
{
	KListView::contentsMouseReleaseEvent(e);
  m_bDrag = false;
}

/** Overrides QScrollView::contentsDragEnterEvent */
void XmlTreeView::contentsDragEnterEvent(QDragEnterEvent *e)
{
	m_pDropItem = 0;
  m_pCurrentBeforeDropItem = selectedItem();

	// Save the available formats
  m_lstDropFormats.clear();
  for(int i = 0; e->format(i); i++)
  	{ if(*(e->format(i)))
    		{ m_lstDropFormats.append(e->format(i));
    		}
		}
}

/** Overrides QScrollView::contentsDragMoveEvent */
void XmlTreeView::contentsDragMoveEvent(QDragMoveEvent *e)
{
	QListViewItem *item = itemAt(contentsToViewport(e->pos()));

  // Accept drops on the background, if Texts
  if(!item && (m_lstDropFormats.contains("text/" + XmlTreeItem::m_strSubtypeXML) ||
						 		m_lstDropFormats.contains("text/" + XmlTreeItem::m_strSubtypeXML_comment) ||
						 		m_lstDropFormats.contains("text/" + XmlTreeItem::m_strSubtypeXML_procins)))
    {	m_pDropItem = 0;
     	e->acceptAction();
     	if(selectedItem())
     		setSelected(selectedItem(), false); // no item selected
     	return;
    }

  if(!item || !item->isSelectable() || ! static_cast <XmlTreeItem*> (item)->acceptsDrops(m_lstDropFormats))
    { m_pDropItem = 0;
      m_autoOpenTimer->stop();
      e->ignore();
      return;
    }

	e->acceptAction();


 	setSelected(item, true);

  if(item != m_pDropItem )
    { m_autoOpenTimer->stop();
      m_pDropItem = item;
      m_autoOpenTimer->start(autoOpenTimeout);
    }
}

/** Overrides QScrollView::contentsDragLeaveEvent */
void XmlTreeView::contentsDragLeaveEvent(QDragLeaveEvent *e)
{
	// Restore the current item to what it was before the dragging (#17070)
  if(m_pCurrentBeforeDropItem)
  	setSelected(m_pCurrentBeforeDropItem, true);
  else
    setSelected(m_pDropItem, false); // no item selected

  m_pCurrentBeforeDropItem = 0;
  m_pDropItem = 0;
  m_lstDropFormats.clear();
}

/** Overrides QScrollView::contentsDropEvent */
void XmlTreeView::contentsDropEvent(QDropEvent *pDropEvent)
{
	m_autoOpenTimer->stop();

	drop(selectedItem(), pDropEvent);
}

/** Called, when m_autoOpenTimer timeout occured */
void XmlTreeView::slotAutoOpenFolder()
{
	m_autoOpenTimer->stop();

  if(!m_pDropItem || m_pDropItem->isOpen())
  	return;

  m_pDropItem->setOpen( true );
  m_pDropItem->repaint();
}

/** Drop or paste text into item */
bool XmlTreeView::drop(QListViewItem *pItem, QDropEvent *pDropEvent)
{
	XmlTreeItem* pTreeItem = 0;
  if(pItem)
  	pTreeItem = static_cast <XmlTreeItem *> (pItem);
  	
  // First, make check, if moved item is not moved to their children
  if((pDropEvent->source() == this) && (pDropEvent->action() == QDropEvent::Move))
		{ // make check, if moved item is not moved to itself
		  if(m_pCurrentBeforeDropItem && pTreeItem && (m_pCurrentBeforeDropItem == pTreeItem))
				{	return false;
				}	
				
			if(m_pCurrentBeforeDropItem && pTreeItem &&
				  static_cast <XmlTreeItem*> (m_pCurrentBeforeDropItem)->isMyChildren(pTreeItem))
				{	KMessageBox::sorry(0, i18n("Item cannot be moved into their child subtree !"));
					return false;
				}	
		}
	
	if(pasteItem(pTreeItem, pDropEvent))
		{ if((pDropEvent->source() == this) && (pDropEvent->action() == QDropEvent::Move))
				{ if(m_pCurrentBeforeDropItem)
						delete m_pCurrentBeforeDropItem; // remove source item
						
				  pDropEvent->acceptAction();
				}
			return true;
		}
	return false;
}

/** Paste clipboard contents into tree item */
bool XmlTreeView::pasteItem(XmlTreeItem* pTreeItem, QMimeSource* data)			
{	
	QString strText;
	
	// Drop XML Element			
  if(QTextDrag::decode(data, strText, XmlTreeItem::m_strSubtypeXML) &&
  										 ((pTreeItem == 0) || (pTreeItem->itemType() == XmlTreeItem::itemElement)                                                                                                                                                                                                                                                                                                                                           )
  										)
		{	// check if root item already exists
		   if(!pTreeItem && rootXmlElement())
				{ KMessageBox::sorry(this, i18n("Root element already exists !"));
					return false;
				}
						
		  // Paste clipboard contents as XML element
		  if(OptionsDialog::getXmlParser() == OptionsDialog::parserSAX2)
		  	{ QXmlInputSource xmlInputSource;
  			  xmlInputSource.setData(strText);
        	
  				QXmlSimpleReader xmlReader;
					XmlStructureParser handlerSAX2(this, m_pDoc);
			
					// if any element is selected, push it into XmlStructureParser stack
					if(pTreeItem) // otherwise pasted element becomes root
						{ // Pasted element becomes child of pSelectedItem
							handlerSAX2.pushXmlElement(static_cast <XmlElement *> (pTreeItem));
						}
						
					xmlReader.setContentHandler(&handlerSAX2);
					xmlReader.setLexicalHandler(&handlerSAX2);
					xmlReader.setErrorHandler(&handlerSAX2);
								
					xmlReader.setFeature( "http://xml.org/sax/features/namespaces", TRUE );
  				xmlReader.setFeature( "http://xml.org/sax/features/namespace-prefixes", TRUE );

  				if(xmlReader.parse(xmlInputSource) == false)
						{ KMessageBox::error(0,
								 								 i18n("%1 in line %2, column %3").arg(handlerSAX2.GetErrorMsg()).arg(handlerSAX2.GetLineNumber()).arg(handlerSAX2.GetColumnNumber()),
								 								 i18n("Parsing error !"));
		 					emit contentsChanged();
		 					return false;
						}
				}
			else
				{ m_pDoc->parseDOM(strText, static_cast <XmlElement *> (pTreeItem));
				}
			
			emit contentsChanged();													
			return true;
		}
		
	// Drop XML Processing Instruction			
  if(QTextDrag::decode(data, strText, XmlTreeItem::m_strSubtypeXML_procins))
		{	XmlProcessingInstruction *pXmlProcInstr = 0;
		
			if(pTreeItem && (pTreeItem->itemType() == XmlTreeItem::itemProcInstr))
					{ pXmlProcInstr = static_cast <XmlProcessingInstruction *> (pTreeItem);
					}
					
			if(pTreeItem && (pTreeItem->itemType() == XmlTreeItem::itemElement))
				{ pXmlProcInstr = new XmlProcessingInstruction(pTreeItem, "", "");
				}
					
			if(pTreeItem && (pTreeItem->itemType() >= XmlTreeItem::itemText) && (pTreeItem->itemType() <= XmlTreeItem::itemComment))
				{ // Paste clipboard contents as XML text
					XmlContentsItem* pXmlContentsItem = static_cast <XmlContentsItem *> (pTreeItem);
					pXmlContentsItem->setContents(strText);
				}
				
			if(!pTreeItem)
				{ pXmlProcInstr = new XmlProcessingInstruction(this, "", ""); // root proc. instruction
				}
						
			bool bRetVal = false;	
			if(pXmlProcInstr)
				bRetVal = pXmlProcInstr->setContents(strText);
			
			emit contentsChanged();		
			return bRetVal;
		}
		
	// Drop XML contents - Text
	if(pTreeItem && QTextDrag::decode(data, strText, XmlTreeItem::m_strSubtypeXML_text))
		{ if(pTreeItem->itemType() == XmlTreeItem::itemElement)
				{ // create new text item
					new XmlContentsItem(pTreeItem, strText, XmlTreeItem::itemText);
				}
			else
				{ // Paste clipboard contents as XML text
					XmlContentsItem* pXmlContentsItem = static_cast <XmlContentsItem *> (pTreeItem);
					pXmlContentsItem->setContents(strText);
					pXmlContentsItem->setItemType(XmlTreeItem::itemText);
				}	
 			
			emit contentsChanged();									
			return true;
		}

	// Drop XML contents - CDATA
	if(pTreeItem && QTextDrag::decode(data, strText, XmlTreeItem::m_strSubtypeXML_cdata))
		{ if(pTreeItem->itemType() == XmlTreeItem::itemElement)
				{ // create new text item
					new XmlContentsItem(pTreeItem, strText, XmlTreeItem::itemCDATA);
				}
			else
				{ // Paste clipboard contents as XML text
					XmlContentsItem* pXmlContentsItem = static_cast <XmlContentsItem *> (pTreeItem);
					pXmlContentsItem->setContents(strText);
					pXmlContentsItem->setItemType(XmlTreeItem::itemCDATA);
				}	
 			
			emit contentsChanged();										
			return true;
		}
		
	// Drop XML contents - comment
	if(QTextDrag::decode(data, strText, XmlTreeItem::m_strSubtypeXML_comment))
		{ if(pTreeItem)
				{ if(pTreeItem->itemType() == XmlTreeItem::itemElement)
						{ // create new text item
							new XmlContentsItem(pTreeItem, strText, XmlTreeItem::itemComment);
						}
					else
						{ // Paste clipboard contents as XML text
							XmlContentsItem* pXmlContentsItem = static_cast <XmlContentsItem *> (pTreeItem);
							pXmlContentsItem->setContents(strText);
							pXmlContentsItem->setItemType(XmlTreeItem::itemComment);
						}
				}
			else
				{ // create new root text item
					new XmlContentsItem(this, strText, XmlTreeItem::itemComment);
				}	
 			
			emit contentsChanged();										
			return true;
		}		 	
 			
 	return false;
}

/** Return pointer to root XML element */
XmlElement* XmlTreeView::rootXmlElement()
{
	XmlTreeItem* pXmlTreeItem = static_cast <XmlTreeItem *> (firstChild());
	while(pXmlTreeItem)
  		{ if(pXmlTreeItem->itemType() == XmlTreeItem::itemElement)
  				{ return static_cast <XmlElement *> (pXmlTreeItem);
  				}
  			pXmlTreeItem = static_cast <XmlTreeItem *> (pXmlTreeItem->nextSibling());
  		}
	return 0L;
}

// --- added by Olaf Hartig (29.Jun.2001)
/** Rename column @p c of @p item, if it's a XmlTreeItem::itemElement. */
void XmlTreeView::rename(QListViewItem *item, int c)
{
	XmlTreeItem* pItem = static_cast<XmlTreeItem *> (item);
	                                                     // if we have an
	if(pItem->itemType() == XmlTreeItem::itemElement)    // itemElement,
		{ KListView::rename(item, c);                      // rename it
			emit contentsChanged();
		}
}

