/***************************************************************************
 *   Copyright (C) 2004 Nadeem Hasan <nhasan@kde.org>                      *
 *   Copyright (C) 2004 Stefan Kombrink <katakombi@web.de>                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <sys/shm.h>

#include <kdebug.h>
#include <kstaticdeleter.h>

#include "synconfig.h"
#include "touchpad.h"

#define SYNSHM self()->m_synShm
//#define PI 3.1415926535897931

TouchPad *TouchPad::m_self = 0;
static KStaticDeleter<TouchPad> staticTouchPadDeleter;

static int finger_low[] = { 53, 38, 25, 18, 10 };

TouchPad *TouchPad::self()
{
  if ( !m_self )
  {
    staticTouchPadDeleter.setObject( m_self, new TouchPad() );
  }

  return m_self;
}

TouchPad::TouchPad()
  : m_synShm( 0 )
{
    int shmid;

    // try to access the shared memory segment
    if ((shmid = shmget(SHM_SYNAPTICS, sizeof(SynapticsSHM), 0)) == -1) {
        if ((shmid = shmget(SHM_SYNAPTICS, 0, 0)) == -1) 
            kdError() << "Access denied to driver shared memory" << endl;
        else
            kdError() << "Shared memory segment size mismatch" << endl;
	m_synShm = NULL;
    } else
    {
      // get the pointer
      if ((m_synShm = (SynapticsSHM*) shmat(shmid, NULL, 0)) == NULL) 
          kdError() << "Error attaching to shared memory segment" << endl;
    }
}

TouchPad::~TouchPad()
{
  if (  m_self == this )
    staticTouchPadDeleter.setObject( m_self, 0, false );
}

bool TouchPad::isValid()
{
  return SYNSHM != 0;
}

int TouchPad::isEnabled()
{
  return isValid()? SYNSHM->touchpad_off : false;
}

void TouchPad::setEnabled( int enable )
{
  if ( !isValid() )
    return;

  SYNSHM->touchpad_off = enable;
}

QRect TouchPad::edges()
{
  if ( !isValid() )
    return QRect();

  return QRect( QPoint( SYNSHM->top_edge, SYNSHM->left_edge ),
                QPoint( SYNSHM->bottom_edge, SYNSHM->right_edge ) );
}

void TouchPad::setEdges( const QRect &edge )
{
  if ( !isValid() )
    return;

  SYNSHM->top_edge = edge.top();
  SYNSHM->left_edge = edge.left();
  SYNSHM->bottom_edge = edge.bottom();
  SYNSHM->right_edge = edge.right();
}

bool TouchPad::isTappingEnabled()
{
  return isValid()? SYNSHM->tap_time>0 : false;
}

int TouchPad::tapTime()
{
  return isValid()? SYNSHM->tap_time : 0;
}

void TouchPad::setTapTime( int time )
{
  if ( !isValid() )
    return;

  SYNSHM->tap_time = time;
}

int TouchPad::sensitivity()
{
  if ( !isValid() )
    return 0;

  int i;

  for ( i=0; i<5; ++i )
  {
    if ( SYNSHM->finger_low >= finger_low[ i ] )
      return i;
  }

  return i-1;
}

void TouchPad::setSensitivity( int i )
{
  if ( !isValid() )
    return;

  if ( i< 0 && i>4 )
    return;
  
  if ( !SynConfig::treatAsALPS() )
  {
    SYNSHM->finger_low = finger_low[ i ];
    SYNSHM->finger_high = finger_low[ i ]+5;
  }else{
    SYNSHM->finger_low = finger_low[ i ]-11;
    SYNSHM->finger_high = finger_low[ i ]-10;
  }
}

bool TouchPad::isHorizontalScrollEnabled()
{
  return isValid()? SYNSHM->scroll_dist_horiz>0 : false;
}

int TouchPad::horizontalScrollDelta()
{
  return isValid()? SYNSHM->scroll_dist_horiz : 0;
}

void TouchPad::setHorizontalScrollDelta( int delta )
{
  if ( !isValid() )
    return;

  SYNSHM->scroll_dist_horiz = delta;
}

bool TouchPad::isVerticalScrollEnabled()
{
  return isValid()? SYNSHM->scroll_dist_vert>0 : false;
}

int TouchPad::verticalScrollDelta()
{
  return isValid()? SYNSHM->scroll_dist_vert : 0;
}

void TouchPad::setVerticalScrollDelta( int delta )
{
  if ( !isValid() )
    return;

  SYNSHM->scroll_dist_vert = delta;
}

bool TouchPad::isEdgeMotionEnabled()
{
  if ( isValid() ) 
    return SYNSHM->edge_motion_use_always;
  else 
    return false;
}

bool TouchPad::isCoastingEnabled()
{
  if ( isValid() )
    return ( SYNSHM->coasting_speed < 0.1 );
  else
    return false;
}

void TouchPad::setEdgeMotionEnabled( bool enable )
{
  if ( !isValid() )
    return;

  SYNSHM->edge_motion_use_always = enable;
}

void TouchPad::setCoastingEnabled( bool enable )
{
  if ( !isValid() )
    return;
  
  if ( enable ) 
    SYNSHM->coasting_speed = coastingSpeedThreshold; // set to reasonable speed
  else
    SYNSHM->coasting_speed = 0.0; // disable it completely
}

bool TouchPad::isCircularScrollEnabled()
{
  return isValid()? SYNSHM->circular_scrolling : false;
}

void TouchPad::setCircularScrollEnabled( bool enable )
{
  if ( !isValid() )
    return;

  SYNSHM->circular_scrolling = enable;
}

int TouchPad::circularScrollDelta()
{
  return isValid()? ( int )( SYNSHM->scroll_dist_circ*1000 ) : 0;
}

void TouchPad::setCircularScrollDelta( int delta )
{
  if ( !isValid() )
    return;

  SYNSHM->scroll_dist_circ = ( ( double )delta )/1000;
}

ScrollTrigger TouchPad::circularScrollTrigger()
{
  return isValid()? ( ScrollTrigger )SYNSHM->circular_trigger : NoTrigger;
}

void TouchPad::setCircularScrollTrigger( ScrollTrigger t )
{
  if ( !isValid() )
    return;

  SYNSHM->circular_trigger = t;
}

Button TouchPad::buttonForTap( TapType tap )
{
  return isValid()? ( Button )SYNSHM->tap_action[ tap ] : None;
}

bool TouchPad::areFastTapsEnabled()
{
  return isValid()? SYNSHM->fast_taps : false;
}

void TouchPad::setFastTaps( bool enable )
{
  if ( !isValid() )
    return;

  SYNSHM->fast_taps = enable;
}

void TouchPad::setButtonForTap( TapType tap, Button button )
{
  if ( !isValid() )
    return;

  SYNSHM->tap_action[ tap ] = button;
}

int TouchPad::absCoordX()
{
  return isValid()? SYNSHM->x : 0;
}

int TouchPad::absCoordY()
{
  return isValid()? SYNSHM->y : 0;
}

void TouchPad::applyConfig()
{
  setEnabled( SynConfig::enableTouchPad() );
  setSensitivity( SynConfig::sensitivity() );
  setTapTime( SynConfig::enableTapping()? SynConfig::tapTime() : 0 );
  setButtonForTap( OneFinger, ( Button )SynConfig::tapOneFinger() );
  setButtonForTap( TwoFingers, ( Button )SynConfig::tapTwoFingers() );
  setButtonForTap( ThreeFingers, ( Button )SynConfig::tapThreeFingers() );
  setButtonForTap( RightTop, ( Button )SynConfig::tapRightTop() );
  setButtonForTap( RightBottom, ( Button )SynConfig::tapRightBottom() );
  setButtonForTap( LeftTop, ( Button )SynConfig::tapLeftTop() );
  setButtonForTap( LeftBottom, ( Button )SynConfig::tapLeftBottom() );  
  setHorizontalScrollDelta( SynConfig::enableHorizontalScrolling()?
      SynConfig::horizontalScrollDelta() : 0 );
  setVerticalScrollDelta( SynConfig::enableVerticalScrolling()?
      SynConfig::verticalScrollDelta() : 0 );
  setEdgeMotionEnabled( SynConfig::enableEdgeMotion() );
  setCoastingEnabled( SynConfig::enableCoasting() );
  setCircularScrollEnabled( SynConfig::enableCircularScrolling() );
  setCircularScrollDelta( SynConfig::circularScrollDelta() );
  setCircularScrollTrigger( ( ScrollTrigger )SynConfig::circularScrollTrigger() );

  setEdges( SynConfig::edges() );
  setFastTaps( SynConfig::fastTaps() );
}

