/***************************************************************************
 *   Copyright (C) by                                                      *
 *     - 2005: Christian Leh <moodwrod@web.de>                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qfontmetrics.h>

#include "scaler.h"

Scaler::Scaler(QSize baseResolution, QSize targetResolution)
{
  mBaseResolution = baseResolution;
  mTargetResolution = targetResolution;
}


void Scaler::autoCoords(QPoint* pt, const QFont& f, const QString& s)
{
  QFontMetrics fm(f);
  QSize fmSize(fm.size(0L, s));

  autoCoords(pt, fmSize);
}


void Scaler::autoCoords(QPoint* pt, const QSize s)
{
  scaleCoords(pt);

  if ((pt->x() == -1) && (pt->y() != -1))
    pt->setX(center(mTargetResolution.width(), s.width()));
  else if ((pt->y() == -1) && (pt->x() != -1))
    pt->setY(center(mTargetResolution.height(), s.height()));
  else if (*pt == QPoint(-1, -1))
    *pt = QPoint(center(mTargetResolution.width(), s.width()), center(mTargetResolution.height(), s.height()));
}


void Scaler::scaleCoords(QPoint* pt)
{
  if (mBaseResolution == mTargetResolution)
    return;

  int ox = pt->x();
  int oy = pt->y();

  float tx = float(mBaseResolution.width()) / float(ox);
  float ty = float(mBaseResolution.height()) / float(oy);

  int nx = intIt(float(mTargetResolution.width()) / tx);
  int ny = intIt(float(mTargetResolution.height()) / ty);

  pt->setX((ox == -1) ? -1 : nx);
  pt->setY((oy == -1) ? -1 : ny);
}


bool Scaler::scaleSize(QImage* i)
{
  if ((!i) || !resolutionDiff())
    return false;

  float tx = float(mTargetResolution.width()) / float(mBaseResolution.width());
  float ty = float(mTargetResolution.height()) / float(mBaseResolution.height());
  int nx = intIt(float(i->width()) * tx);
  int ny = intIt(float(i->height()) * ty);

  *i = i->smoothScale(nx, ny);

  return true;
}


bool Scaler::scaleSize(QFont* f)
{
  if ((!f) || !resolutionDiff())
    return false;

  const float d = 96 / 72;
  const float tx = float(mTargetResolution.height()) / float(mBaseResolution.height());
  float pt = f->pointSizeFloat();
  int hPx = intIt(pt * d);
  
  f->setPixelSize(intIt(float(hPx) * tx));

  return true;
}


bool Scaler::resolutionDiff()
{
  return (mBaseResolution != mTargetResolution);
}


int Scaler::intIt(const float v)
{
  float t = v - float(int(v));
  float tt = (t < 0.5) ? 0 : 1;
  
  return int(v + tt);
}


int Scaler::center(const int width, const int size, int offset)
{
  return int(width / 2) - int(size / 2) + offset;
}
