/*
 *  KSeg
 *  Copyright (C) 1999-2001 Ilya Baran
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send comments and/or bug reports to:
 *                 ibaran@mit.edu
 */


#include <qapplication.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qcolordialog.h>
#include <qfontdialog.h>
#include <qsplitter.h>
#include "KSegConstructionList.H"
#include "KSegWindow.H"
#include "KSegView.H"
#include "KSegDocument.H"
#include "KSegConstruction.H"
#include "G_ref.H"
#include "defs.H"
#include "KSegProperties.H"


QString KSegWindow::quickPlayDirectory;

KSegWindow::KSegWindow(KSegDocument *doc)
  : QMainWindow(0, 0, WDestructiveClose | WType_TopLevel)
{
  quickPlayDirectory = KSegProperties::getProperty("QuickPlayDirectory");

  if(doc == 0) {
    doc = new KSegDocument();
  }
  else {
    if(doc->getFilename().isEmpty()) setCaption(tr("kseg: Untitled"));
    else setCaption(QString("kseg: ") + doc->getFilename());
  }

  if(doc->isConstruction() == false) {
    view = new KSegView(doc, this);

    setCentralWidget(view);

    resize(640, 480);
  }
  else {
    QSplitter *sp = new QSplitter(this);

    view = new KSegView(doc, sp);
    KSegConstructionList *lb = new KSegConstructionList(doc, sp);

    sp->setOrientation(Horizontal);
    sp->setOpaqueResize(true);

    setCentralWidget(sp);

    resize(640, 480);
    lb->resize(50, 480);
  }

  connect(view, SIGNAL(statusBarMessage(const QString &)), statusBar(), SLOT(message(const QString &)));
  connect(view, SIGNAL(updateMenus()), this, SLOT(updateMenus()));

  fileMenu = new QPopupMenu(this);
  newMenu = new QPopupMenu(this);
  editMenu = new QPopupMenu(this);
  colorMenu = new QPopupMenu(this);
  pointstyleMenu = new QPopupMenu(this);
  linestyleMenu = new QPopupMenu(this);
  fontMenu = new QPopupMenu(this);
  measureMenu = new QPopupMenu(this);
  transformMenu = new QPopupMenu(this);
  playMenu = new QPopupMenu(this);
  quickPlayMenu = new QPopupMenu(this);
  if(doc->isConstruction())
    constructionMenu = new QPopupMenu(this);

/*
  fileMenu->insertTearOffHandle();
  newMenu->insertTearOffHandle();
  editMenu->insertTearOffHandle();
  measureMenu->insertTearOffHandle();
  transformMenu->insertTearOffHandle();
  if(doc->isConstruction()) constructionMenu->insertTearOffHandle();
*/

  playMenu->insertTearOffHandle();
  colorMenu->insertTearOffHandle();
  pointstyleMenu->insertTearOffHandle();
  linestyleMenu->insertTearOffHandle();
  fontMenu->insertTearOffHandle();

  QPopupMenu *helpMenu = new QPopupMenu(this);;

  connect(fileMenu, SIGNAL(aboutToShow()), this, SLOT(updateRecentList()));
  connect(newMenu, SIGNAL(aboutToShow()), this, SLOT(updateNewMenu()));
  connect(editMenu, SIGNAL(aboutToShow()), this, SLOT(updateEditMenu()));
  connect(measureMenu, SIGNAL(aboutToShow()), this, SLOT(updateMeasureMenu()));
  connect(transformMenu, SIGNAL(aboutToShow()), this, SLOT(updateTransformMenu()));
  connect(playMenu, SIGNAL(aboutToShow()), this, SLOT(updatePlayMenu()));
  connect(quickPlayMenu, SIGNAL(aboutToShow()), this, SLOT(updateQuickPlayMenu()));
  if(doc->isConstruction())
    connect(constructionMenu, SIGNAL(aboutToShow()), this, SLOT(updateConstructionMenu()));

  menuBar()->insertItem(tr("&File"), fileMenu);
  menuBar()->insertItem(tr("&Edit"), editMenu);
  menuBar()->insertItem(tr("&New"), newMenu);
  menuBar()->insertItem(tr("Meas&ure"), measureMenu);
  menuBar()->insertItem(tr("&Transform"), transformMenu);
  menuBar()->insertItem(tr("Pla&y"), playMenu);
  if(doc->isConstruction())
    menuBar()->insertItem(tr("Const&ruction"), constructionMenu);
  menuBar()->insertSeparator();
  menuBar()->insertItem(tr("&Help"), helpMenu);

  //file menu init:
  fileMenu->insertItem(tr("&New Sketch"), this, SLOT(newSketch()), CTRL+Key_N);
  fileMenu->insertItem(tr("Ne&w Construction"), this, SLOT(newConstruction()));
  fileMenu->insertItem(tr("New &View"), this, SLOT(newView()));
  fileMenu->insertItem(tr("&Open..."), this, SLOT(fileOpen()), CTRL+Key_O);
  fileMenu->insertItem(tr("&Save"), this, SLOT(fileSave()), CTRL+Key_S);
  fileMenu->insertItem(tr("Save &As..."), this, SLOT(fileSaveAs()));

  fileMenu->insertSeparator();
  
  fileMenu->insertItem(tr("&Print..."), this, SLOT(filePrint()), CTRL+Key_P);

  fileMenu->insertSeparator();
  
  fileMenu->insertItem(tr("Choose &Language..."), this, SLOT(fileChooseLanguage()));
  
  fileMenu->insertSeparator();

  fileMenu->insertItem(tr("&Close"), this, SLOT(close()), CTRL+Key_W);
  fileMenu->insertItem(tr("&Quit"), qApp, SLOT(closeAllWindows()), CTRL+Key_Q);

  fileMenu->insertSeparator();
  //recently opened files go here

  //edit menu init:
  editMenu->insertItem(tr("&Undo"), view->getDocument(), SLOT(editUndo()),
		       CTRL+Key_Z, ID_EDIT_UNDO);
  editMenu->insertItem(tr("&Redo"), view->getDocument(), SLOT(editRedo()),
		       CTRL+Key_R, ID_EDIT_REDO);

  editMenu->insertSeparator();

  editMenu->insertItem(tr("&Delete Objects"), view->getDocument(), SLOT(editDelete()),
		       CTRL+Key_Delete, ID_EDIT_DELETE);
  editMenu->insertSeparator();
  editMenu->insertItem(tr("Toggle &Label"), view->getDocument(),
		       SLOT(editToggleLabels()), CTRL+Key_L, ID_EDIT_TOGGLELABELS);
  editMenu->insertItem(tr("Hide Label"), view->getDocument(),
		       SLOT(editHideLabels()), 0, ID_EDIT_HIDELABELS);
  editMenu->insertItem(tr("Show Label"), view->getDocument(),
		       SLOT(editShowLabels()), 0, ID_EDIT_SHOWLABELS);
  editMenu->insertItem(tr("Change La&bel..."), view->getDocument(),
		       SLOT(editChangeLabel()), 0, ID_EDIT_CHANGELABEL);
  editMenu->insertItem(tr("&Hide Objects"), view->getDocument(), SLOT(editHide()),
		       CTRL+Key_H, ID_EDIT_HIDE);
  editMenu->insertItem(tr("&Unhide All"), view->getDocument(), SLOT(editShowHidden()),
		       CTRL+Key_U, ID_EDIT_SHOWHIDDEN);

  editMenu->insertSeparator();

  editMenu->insertItem(tr("&Color"), colorMenu, ID_EDIT_COLOR);
  editMenu->insertItem(tr("Line &Style"), linestyleMenu, ID_EDIT_LINESTYLE);
  editMenu->insertItem(tr("&Point Style"), pointstyleMenu, ID_EDIT_POINTSTYLE);
  editMenu->insertItem(tr("&Font"), fontMenu, ID_EDIT_FONT);

  editMenu->insertSeparator();

  editMenu->insertItem(tr("Change &Number of Samples..."), view->getDocument(),
		       SLOT(editChangeNumberOfSamples()), 0,
		       ID_EDIT_CHANGE_NUMBER_OF_SAMPLES);
  

  //edit color menu init:
  QPixmap p(100, 20);

  p.fill(black);
  colorMenu->insertItem(p, this, SLOT(editColorBlack()), 0, ID_EDIT_COLOR_BLACK);
  p.fill(gray);
  colorMenu->insertItem(p, this, SLOT(editColorGray()), 0, ID_EDIT_COLOR_GRAY);
  p.fill(red);
  colorMenu->insertItem(p, this, SLOT(editColorRed()), 0, ID_EDIT_COLOR_RED);
  p.fill(green);
  colorMenu->insertItem(p, this, SLOT(editColorGreen()), 0, ID_EDIT_COLOR_GREEN);
  p.fill(blue);
  colorMenu->insertItem(p, this, SLOT(editColorBlue()), 0, ID_EDIT_COLOR_BLUE);
  p.fill(yellow);
  colorMenu->insertItem(p, this, SLOT(editColorYellow()), 0, ID_EDIT_COLOR_YELLOW);
  p.fill(magenta);
  colorMenu->insertItem(p, this, SLOT(editColorPurple()), 0, ID_EDIT_COLOR_PURPLE);
  p.fill(cyan);
  colorMenu->insertItem(p, this, SLOT(editColorCyan()), 0, ID_EDIT_COLOR_CYAN);

  colorMenu->insertSeparator();

  colorMenu->insertItem(tr("&Other..."), this, SLOT(editColorOther()), 0,
			ID_EDIT_COLOR_OTHER);

  //edit pointstyle menu init:
  QPainter pa;
  pa.begin(&p);
  G_drawstyle *d;
  G_point pt(50, 10);

  QBrush b(black);

  p.fill(lightGray);
  d = G_drawstyle::match(LARGE_CIRCLE, 0, 0, &b);
  pt.draw(pa, *d);
  d->deleteReference();
  pointstyleMenu->insertItem(p, this, SLOT(editPointstyleLargecircle()), 0,
			    ID_EDIT_POINTSTYLE_LARGECIRCLE);

  p.fill(lightGray);
  d = G_drawstyle::match(MEDIUM_CIRCLE, 0, 0, &b);
  pt.draw(pa, *d);
  d->deleteReference();
  pointstyleMenu->insertItem(p, this, SLOT(editPointstyleMediumcircle()), 0,
			    ID_EDIT_POINTSTYLE_MEDIUMCIRCLE);

  p.fill(lightGray);
  d = G_drawstyle::match(SMALL_CIRCLE, 0, 0, &b);
  pt.draw(pa, *d);
  d->deleteReference();
  pointstyleMenu->insertItem(p, this, SLOT(editPointstyleSmallcircle()), 0,
			    ID_EDIT_POINTSTYLE_SMALLCIRCLE);


  //edit linestyle menu init:
  QPen solid(SolidLine), dashed(DashLine), dotted(DotLine);
  QPen thin(black, 0), normal(black, 2), thick(black, 3);

  p.fill(lightGray);  
  pa.setPen(solid);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleSolid()), 0,
			    ID_EDIT_LINESTYLE_SOLID);

  p.fill(lightGray);  
  pa.setPen(dashed);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleDashed()), 0,
			    ID_EDIT_LINESTYLE_DASHED);

  p.fill(lightGray);  
  pa.setPen(dotted);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleDotted()), 0,
			    ID_EDIT_LINESTYLE_DOTTED);

  linestyleMenu->insertSeparator();

  p.fill(lightGray);  
  pa.setPen(thin);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleThin()), 0,
			    ID_EDIT_LINESTYLE_THIN);

  p.fill(lightGray);  
  pa.setPen(normal);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleNormal()), 0,
			    ID_EDIT_LINESTYLE_NORMAL);

  p.fill(lightGray);  
  pa.setPen(thick);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleThick()), 0,
			    ID_EDIT_LINESTYLE_THICK);


  //edit font menu init:
  fontMenu->insertItem("10", this, SLOT(editFont10()), 0, ID_EDIT_FONT_10);
  fontMenu->insertItem("12", this, SLOT(editFont12()), 0, ID_EDIT_FONT_12);
  fontMenu->insertItem("14", this, SLOT(editFont14()), 0, ID_EDIT_FONT_14);
  fontMenu->insertItem("24", this, SLOT(editFont20()), 0, ID_EDIT_FONT_20);
  fontMenu->insertItem("30", this, SLOT(editFont30()), 0, ID_EDIT_FONT_30);

  fontMenu->insertSeparator();
  
  fontMenu->insertItem(tr("&Font..."), this, SLOT(editFontFont()), 0, ID_EDIT_FONT_FONT);

  //new menu init:
  newMenu->insertItem(tr("&Intersection Point"), view->getDocument(),
		      SLOT(newIntersection()), ALT+Key_I, ID_NEW_INTERSECTION);
  newMenu->insertItem(tr("&Segment"), view->getDocument(), SLOT(newSegment()),
		      ALT+Key_S, ID_NEW_SEGMENT);
  newMenu->insertItem(tr("&Midpoint"), view->getDocument(), SLOT(newMidpoint()),
		      ALT+Key_M, ID_NEW_MIDPOINT);
  newMenu->insertItem(tr("&Ray"), view->getDocument(), SLOT(newRay()), 0, ID_NEW_RAY);
  newMenu->insertItem(tr("&Line"), view->getDocument(), SLOT(newLine()), ALT+Key_L,
		      ID_NEW_LINE);
  newMenu->insertItem(tr("&Perpendicular Line"), view->getDocument(),
		      SLOT(newPerpendicular()), ALT+Key_P, ID_NEW_PERPENDICULAR);
  newMenu->insertItem(tr("&Circle By Center And Point"), view->getDocument(),
		      SLOT(newCircle()), ALT+Key_C, ID_NEW_CIRCLE);
  newMenu->insertItem(tr("&Arc By Three Points"), view->getDocument(),
		      SLOT(newArc()), ALT+Key_A, ID_NEW_ARC);
  newMenu->insertItem(tr("Angle &Bisector"), view->getDocument(),
		      SLOT(newBisector()), ALT+Key_B, ID_NEW_BISECTOR);
  newMenu->insertItem(tr("L&ocus"), view->getDocument(),
		      SLOT(newLocus()), ALT+Key_O, ID_NEW_LOCUS);
  newMenu->insertSeparator();
  newMenu->insertItem(tr("Poly&gon"), view->getDocument(),
		      SLOT(newPolygon()), ALT+Key_G, ID_NEW_POLYGON);
  newMenu->insertItem(tr("Arc S&ector"), view->getDocument(),
		      SLOT(newArcSector()), 0, ID_NEW_ARCSECTOR);
  newMenu->insertItem(tr("Arc Se&gment"), view->getDocument(),
		      SLOT(newArcSegment()), 0, ID_NEW_ARCSEGMENT);
  newMenu->insertItem(tr("Circle I&nterior"), view->getDocument(),
		      SLOT(newCircleInterior()), 0, ID_NEW_CIRCLEINTERIOR);

  //measure menu init:
  measureMenu->insertItem(tr("&Distance"), view->getDocument(),
			  SLOT(measureDistance()), 0, ID_MEASURE_DISTANCE);
  measureMenu->insertItem(tr("&Length"), view->getDocument(),
			  SLOT(measureLength()), 0, ID_MEASURE_LENGTH);
  measureMenu->insertItem(tr("&Radius"), view->getDocument(),
			  SLOT(measureRadius()), 0, ID_MEASURE_RADIUS);
  measureMenu->insertItem(tr("&Angle"), view->getDocument(),
			  SLOT(measureAngle()), 0, ID_MEASURE_ANGLE);
  measureMenu->insertItem(tr("Ra&tio"), view->getDocument(),
			  SLOT(measureRatio()), 0, ID_MEASURE_RATIO);
  measureMenu->insertItem(tr("&Slope"), view->getDocument(),
			  SLOT(measureSlope()), 0, ID_MEASURE_SLOPE);
  measureMenu->insertItem(tr("Ar&ea"), view->getDocument(),
			  SLOT(measureArea()), 0, ID_MEASURE_AREA);

  measureMenu->insertSeparator();

  measureMenu->insertItem(tr("&Calculate..."), view->getDocument(),
			  SLOT(measureCalculate()), 0, ID_MEASURE_CALCULATE);

  //transform menu init:
  transformMenu->insertItem(tr("Choose &Vector"), view->getDocument(),
			    SLOT(transformChooseVector()), 0,
			    ID_TRANSFORM_CHOOSE_VECTOR);
  transformMenu->insertItem(tr("Choose &Mirror"), view->getDocument(),
			    SLOT(transformChooseMirror()), 0,
			    ID_TRANSFORM_CHOOSE_MIRROR);
  transformMenu->insertItem(tr("Choose &Center"), view->getDocument(),
			    SLOT(transformChooseCenter()), 0,
			    ID_TRANSFORM_CHOOSE_CENTER);
  transformMenu->insertItem(tr("Choose &Angle"), view->getDocument(),
			    SLOT(transformChooseAngle()), 0,
			    ID_TRANSFORM_CHOOSE_ANGLE);
  transformMenu->insertItem(tr("Choose Rati&o"), view->getDocument(),
			    SLOT(transformChooseRatio()), 0,
			    ID_TRANSFORM_CHOOSE_RATIO);
  transformMenu->insertItem(tr("C&lear Chosen"), view->getDocument(),
			    SLOT(transformClearChosen()), 0,
			    ID_TRANSFORM_CLEAR_CHOSEN);

  transformMenu->insertSeparator();

  transformMenu->insertItem(tr("&Translate"), view->getDocument(),
			    SLOT(transformTranslate()), 0, ID_TRANSFORM_TRANSLATE);
  transformMenu->insertItem(tr("R&eflect"), view->getDocument(),
			    SLOT(transformReflect()), 0, ID_TRANSFORM_REFLECT);
  transformMenu->insertItem(tr("&Scale"), view->getDocument(),
			    SLOT(transformScale()), 0, ID_TRANSFORM_SCALE);
  transformMenu->insertItem(tr("&Rotate"), view->getDocument(),
			    SLOT(transformRotate()), 0, ID_TRANSFORM_ROTATE);


  //construction menu init
  if(doc->isConstruction()) {
    constructionMenu->insertItem(tr("Make &Normal"), view->getDocument(),
				 SLOT(constructionMakeNormal()), 0,
				 ID_CONSTRUCTION_MAKE_NORMAL);
 
    constructionMenu->insertItem(tr("Make &Given"), view->getDocument(),
				 SLOT(constructionMakeGiven()), 0,
				 ID_CONSTRUCTION_MAKE_GIVEN);

    constructionMenu->insertItem(tr("Make &Final"), view->getDocument(),
				 SLOT(constructionMakeFinal()), 0,
				 ID_CONSTRUCTION_MAKE_FINAL);

    constructionMenu->insertItem(tr("Make &Initial"), view->getDocument(),
				 SLOT(constructionMakeInitial()), 0,
				 ID_CONSTRUCTION_MAKE_INITIAL);

    constructionMenu->insertSeparator();

    constructionMenu->insertItem(tr("&Recurse"), view->getDocument(),
				 SLOT(constructionRecurse()), 0,
				 ID_CONSTRUCTION_RECURSE);
  }

  //play menu init
  playMenu->insertItem(tr("&Quick Play"), quickPlayMenu, ID_PLAY_QUICKPLAY);


  //help menu init
  helpMenu->insertItem(tr("&About..."), this, SLOT(helpAbout()));

  statusBar()->show();
}


void KSegWindow::statusBarMessage(const QString &s)
{
  statusBar()->message(s);
}


void KSegWindow::updateMenus()
{
  if(view->getDrag() != KSegView::NO_DRAG || view->getMenusEnabled() == false) {
    menuBar()->setEnabled(false);

    int i;

    for(i = 0; i < (int)editMenu->count(); i++) {
      editMenu->setItemEnabled(editMenu->idAt(i), false);
    }

    for(i = 0; i < (int)newMenu->count(); i++) {
      newMenu->setItemEnabled(newMenu->idAt(i), false);
    }

    for(i = 0; i < (int)measureMenu->count(); i++) {
      measureMenu->setItemEnabled(measureMenu->idAt(i), false);
    }

    for(i = 0; i < (int)transformMenu->count(); i++) {
      transformMenu->setItemEnabled(transformMenu->idAt(i), false);
    }
  }
  else {
    menuBar()->setEnabled(true);
    updateRecentList();
    updateNewMenu();
    updateEditMenu();
    updateEditColorMenu();
    updateEditFontMenu();
    updateEditPointstyleMenu();
    updateEditLinestyleMenu();
    updateMeasureMenu();
    updateTransformMenu();
    updatePlayMenu();
    if(view->getDocument()->isConstruction()) updateConstructionMenu();
  }
}


void KSegWindow::updateRecentList()
{

  bool ok;
  int count = KSegProperties::getProperty("RecentListSize").toInt(&ok);
  if(!ok) {
    count = 4;
    KSegProperties::setProperty("RecentListSize", "4");
  }

  //clear it first:
  int i;
  for(i = 0; i < count; ++i) {
    if(fileMenu->indexOf(ID_FILE_RECENTLIST_START + i) != -1)
      fileMenu->removeItem(ID_FILE_RECENTLIST_START + i);
  }
  
  compressRecentList();

  //now recreate it
  for(i = 0; i < count; ++i) {
    QString name = KSegProperties::getProperty(QString("RecentFile") + QString::number(i + 1));
    if(name.isNull()) continue;
    
    fileMenu->insertItem(QString("&") + QString::number(i + 1) + ". " + name, this,
    		 SLOT(fileOpenRecent(int)), 0, ID_FILE_RECENTLIST_START + i);
  }
}


void KSegWindow::compressRecentList()
{
  int count = KSegProperties::getProperty("RecentListSize").toInt();

  hash_set<QString> recentItems;
  int skip = 0;

  int i;
  for(i = 0; i < count; ++i) {
    if(i + skip >= count) {
      KSegProperties::setProperty("RecentFile" + QString::number(i + 1), QString());
      continue;
    }

    QString cur = KSegProperties::getProperty("RecentFile" + QString::number(i + skip + 1));
    if(cur.isNull() || recentItems.count(cur)) {
      skip++;
      i--;
      continue;
    }

    if(skip)
      KSegProperties::setProperty("RecentFile" + QString::number(i + 1),
				 KSegProperties::getProperty("RecentFile" +
							     QString::number(i + skip + 1)));

    recentItems.insert(KSegProperties::getProperty("RecentFile" + QString::number(i + 1)));
  }
}


void KSegWindow::updateTransformMenu()
{
  transformMenu->setItemEnabled(ID_TRANSFORM_CHOOSE_VECTOR,
			   view->getDocument()->canTransformChooseVector());
  transformMenu->setItemEnabled(ID_TRANSFORM_CHOOSE_ANGLE,
			   view->getDocument()->canTransformChooseAngle());
  transformMenu->setItemEnabled(ID_TRANSFORM_CHOOSE_RATIO,
			   view->getDocument()->canTransformChooseRatio());
  transformMenu->setItemEnabled(ID_TRANSFORM_CHOOSE_CENTER,
			   view->getDocument()->canTransformChooseCenter());
  transformMenu->setItemEnabled(ID_TRANSFORM_CHOOSE_MIRROR,
			   view->getDocument()->canTransformChooseMirror());
  transformMenu->setItemEnabled(ID_TRANSFORM_CLEAR_CHOSEN,
			   view->getDocument()->canTransformClearChosen());

  transformMenu->setItemEnabled(ID_TRANSFORM_TRANSLATE,
			   view->getDocument()->canTransformTranslate());
  transformMenu->setItemEnabled(ID_TRANSFORM_REFLECT,
			   view->getDocument()->canTransformReflect());
  transformMenu->setItemEnabled(ID_TRANSFORM_ROTATE,
			   view->getDocument()->canTransformRotate());
  transformMenu->setItemEnabled(ID_TRANSFORM_SCALE,
			   view->getDocument()->canTransformScale());

}


void KSegWindow::updateMeasureMenu()
{
  bool tmp;

  measureMenu->setItemEnabled(ID_MEASURE_DISTANCE,
			   view->getDocument()->canMeasureDistance());

  tmp = view->getDocument()->canMeasureLength(); 
  if(!tmp || view->getDocument()->getSelected()[0]->getType() == G_SEGMENT) {
    measureMenu->changeItem(ID_MEASURE_LENGTH, tr("&Length"));
  }
  else if(view->getDocument()->getSelected()[0]->getType() == G_CIRCLE) {
    measureMenu->changeItem(ID_MEASURE_LENGTH, tr("&Circumference"));
  }
  else { //the object is an arc
    measureMenu->changeItem(ID_MEASURE_LENGTH, tr("Arc &Length"));
  }
  measureMenu->setItemEnabled(ID_MEASURE_LENGTH, tmp);


  measureMenu->setItemEnabled(ID_MEASURE_RADIUS,
			   view->getDocument()->canMeasureRadius());
  measureMenu->setItemEnabled(ID_MEASURE_ANGLE,
			   view->getDocument()->canMeasureAngle());
  measureMenu->setItemEnabled(ID_MEASURE_RATIO,
			   view->getDocument()->canMeasureRatio());
  measureMenu->setItemEnabled(ID_MEASURE_SLOPE,
			   view->getDocument()->canMeasureSlope());
  measureMenu->setItemEnabled(ID_MEASURE_AREA,
			   view->getDocument()->canMeasureArea());
  measureMenu->setItemEnabled(ID_MEASURE_CALCULATE,
			   view->getDocument()->canMeasureCalculate());
}

void KSegWindow::updateEditMenu()
{
  editMenu->setItemEnabled(ID_EDIT_UNDO, view->getDocument()->canEditUndo());
  editMenu->setItemEnabled(ID_EDIT_REDO, view->getDocument()->canEditRedo());
  editMenu->setItemEnabled(ID_EDIT_DELETE, view->getDocument()->canEditDelete());

  bool tmp;
  tmp = view->getDocument()->canEditToggleLabels();
  if(tmp && view->getDocument()->selectedCount() > 1) {
    editMenu->changeItem(ID_EDIT_TOGGLELABELS, tr("Toggle &Labels"));
    editMenu->changeItem(ID_EDIT_HIDELABELS, tr("Hide Labels"));
    editMenu->changeItem(ID_EDIT_SHOWLABELS, tr("Show Labels"));
  }
  else {
    editMenu->changeItem(ID_EDIT_TOGGLELABELS, tr("Toggle &Label"));
    editMenu->changeItem(ID_EDIT_HIDELABELS, tr("Hide Label"));
    editMenu->changeItem(ID_EDIT_SHOWLABELS, tr("Show Label"));
  }
  editMenu->setItemEnabled(ID_EDIT_TOGGLELABELS, tmp);
  editMenu->setItemEnabled(ID_EDIT_HIDELABELS, tmp);
  editMenu->setItemEnabled(ID_EDIT_SHOWLABELS, tmp);

  editMenu->setItemEnabled(ID_EDIT_CHANGELABEL,
			   view->getDocument()->canEditChangeLabel());
  editMenu->setItemEnabled(ID_EDIT_HIDE, view->getDocument()->canEditHide());
  editMenu->setItemEnabled(ID_EDIT_SHOWHIDDEN,
			   view->getDocument()->canEditShowHidden());

  tmp = view->getDocument()->canEditChangeColor();
  editMenu->setItemEnabled(ID_EDIT_COLOR, tmp);
  if(tmp) updateEditColorMenu();

  tmp = view->getDocument()->canEditChangePointstyle();
  editMenu->setItemEnabled(ID_EDIT_POINTSTYLE, tmp);
  if(tmp) updateEditPointstyleMenu();

  tmp = view->getDocument()->canEditChangeLinestyle();
  editMenu->setItemEnabled(ID_EDIT_LINESTYLE, tmp);
  if(tmp) updateEditLinestyleMenu();

  tmp = view->getDocument()->canEditChangeFont();
  editMenu->setItemEnabled(ID_EDIT_FONT, tmp);
  if(tmp) updateEditFontMenu();

  editMenu->setItemEnabled(ID_EDIT_CHANGE_NUMBER_OF_SAMPLES,
			   view->getDocument()->canEditChangeNumberOfSamples());
}


void KSegWindow::updateEditColorMenu()
{
  int i;

  if(!view->getDocument()->canEditChangeColor()) {
    for(i = 0; i < colorMenu->count(); ++i) {
      colorMenu->setItemChecked(colorMenu->idAt(i), false);
      colorMenu->setItemEnabled(colorMenu->idAt(i), false);
    }

    return;
  }
  else {
    for(i = 0; i < colorMenu->count(); ++i) {
      colorMenu->setItemEnabled(colorMenu->idAt(i), true);
    }
  }

  QColor c = view->getDocument()->getCurrentColor();

  colorMenu->setItemChecked(ID_EDIT_COLOR_BLACK, c == black);
  colorMenu->setItemChecked(ID_EDIT_COLOR_GRAY, c == gray);
  colorMenu->setItemChecked(ID_EDIT_COLOR_RED, c == red);
  colorMenu->setItemChecked(ID_EDIT_COLOR_GREEN, c == green);
  colorMenu->setItemChecked(ID_EDIT_COLOR_BLUE, c == blue);
  colorMenu->setItemChecked(ID_EDIT_COLOR_YELLOW, c == yellow);
  colorMenu->setItemChecked(ID_EDIT_COLOR_PURPLE, c == magenta);
  colorMenu->setItemChecked(ID_EDIT_COLOR_CYAN, c == cyan);

  if(c.isValid() && c != black && c != gray && c != red && c != green &&
     c != blue && c != yellow && c != magenta && c != cyan) {
    colorMenu->setItemChecked(ID_EDIT_COLOR_OTHER, true);
  }
  else {
    colorMenu->setItemChecked(ID_EDIT_COLOR_OTHER, false);
  }
}


void KSegWindow::updateEditPointstyleMenu()
{
  int i;

  if(!view->getDocument()->canEditChangePointstyle()) {
    for(i = 0; i < pointstyleMenu->count(); ++i) {
      pointstyleMenu->setItemChecked(pointstyleMenu->idAt(i), false);
      pointstyleMenu->setItemEnabled(pointstyleMenu->idAt(i), false);
    }

    return;
  }
  else {
    for(i = 0; i < pointstyleMenu->count(); ++i) {
      pointstyleMenu->setItemEnabled(pointstyleMenu->idAt(i), true);
    }
  }

  PointStyle style = view->getDocument()->getCurrentPointstyle();

  pointstyleMenu->setItemChecked(ID_EDIT_POINTSTYLE_LARGECIRCLE, style == LARGE_CIRCLE);
  pointstyleMenu->setItemChecked(ID_EDIT_POINTSTYLE_MEDIUMCIRCLE,
				 style == MEDIUM_CIRCLE);
  pointstyleMenu->setItemChecked(ID_EDIT_POINTSTYLE_SMALLCIRCLE, style == SMALL_CIRCLE);
}


void KSegWindow::updateEditLinestyleMenu()
{
  int i;

  if(!view->getDocument()->canEditChangeLinestyle()) {
    for(i = 0; i < linestyleMenu->count(); ++i) {
      linestyleMenu->setItemChecked(linestyleMenu->idAt(i), false);
      linestyleMenu->setItemEnabled(linestyleMenu->idAt(i), false);
    }

    return;
  }
  else {
    for(i = 0; i < linestyleMenu->count(); ++i) {
      linestyleMenu->setItemEnabled(linestyleMenu->idAt(i), true);
    }
  }

  PenStyle style = view->getDocument()->getCurrentPenstyle();
  int width = view->getDocument()->getCurrentPenWidth();

  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_SOLID, style == SolidLine);
  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_DASHED, style == DashLine);
  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_DOTTED, style == DotLine);

  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_THIN, width == 0);
  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_NORMAL, width == 2);
  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_THICK, width == 3);
}


void KSegWindow::updateEditFontMenu()
{
  int i;

  if(!view->getDocument()->canEditChangeFont()) {
    for(i = 0; i < fontMenu->count(); ++i) {
      fontMenu->setItemChecked(fontMenu->idAt(i), false);
      fontMenu->setItemEnabled(fontMenu->idAt(i), false);
    }

    return;
  }
  else {
    for(i = 0; i < fontMenu->count(); ++i) {
      fontMenu->setItemEnabled(fontMenu->idAt(i), true);
    }
  }

  int size = view->getDocument()->getCurrentFontsize();

  fontMenu->setItemChecked(ID_EDIT_FONT_10, size == 10);
  fontMenu->setItemChecked(ID_EDIT_FONT_12, size == 12);
  fontMenu->setItemChecked(ID_EDIT_FONT_14, size == 14);
  fontMenu->setItemChecked(ID_EDIT_FONT_20, size == 20);
  fontMenu->setItemChecked(ID_EDIT_FONT_30, size == 30);
}


void KSegWindow::updateNewMenu()
{
  bool tmp;
  int i;

  //segment
  tmp = view->getDocument()->canNewSegment();
  newMenu->setItemEnabled(ID_NEW_SEGMENT, tmp);
  if(tmp && view->getDocument()->selectedCount() > 2)
    newMenu->changeItem(ID_NEW_SEGMENT, tr("&Segments"));
  else newMenu->changeItem(ID_NEW_SEGMENT, tr("&Segment"));

  //ray
  tmp = view->getDocument()->canNewRay();
  newMenu->setItemEnabled(ID_NEW_RAY, tmp);
  if(tmp && view->getDocument()->selectedCount() > 2)
    newMenu->changeItem(ID_NEW_RAY, tr("&Rays"));
  else newMenu->changeItem(ID_NEW_RAY, tr("&Ray"));

  //bisector
  newMenu->setItemEnabled(ID_NEW_BISECTOR, view->getDocument()->canNewBisector());

  //arc
  newMenu->setItemEnabled(ID_NEW_ARC, view->getDocument()->canNewArc());

  //locus
  newMenu->setItemEnabled(ID_NEW_LOCUS, view->getDocument()->canNewLocus());
      
  //midpoint
  tmp = view->getDocument()->canNewMidpoint();
  newMenu->setItemEnabled(ID_NEW_MIDPOINT, tmp);
  if(tmp && view->getDocument()->selectedCount() > 1)
    newMenu->changeItem(ID_NEW_MIDPOINT, tr("&Midpoints"));
  else newMenu->changeItem(ID_NEW_MIDPOINT, tr("&Midpoint"));

  //intersection point
  tmp = view->getDocument()->canNewIntersection();
  newMenu->setItemEnabled(ID_NEW_INTERSECTION, tmp);
  if(tmp && !((view->getDocument()->getSelected()[0]->getType() & G_STRAIGHT) &&
     (view->getDocument()->getSelected()[1]->getType() & G_STRAIGHT)))
    newMenu->changeItem(ID_NEW_INTERSECTION, tr("&Intersection Points"));
  else newMenu->changeItem(ID_NEW_INTERSECTION, tr("&Intersection Point"));

  //circle
  tmp = view->getDocument()->canNewCircle();
  newMenu->setItemEnabled(ID_NEW_CIRCLE, tmp);
  if(tmp && view->getDocument()->getSelected()[0]->getType() !=
     view->getDocument()->getSelected()[1]->getType())
    newMenu->changeItem(ID_NEW_CIRCLE, tr("&Circle By Center And Radius"));
  else newMenu->changeItem(ID_NEW_CIRCLE, tr("&Circle By Center And Point"));

  // line/parallel
  tmp = view->getDocument()->canNewLine();
  newMenu->setItemEnabled(ID_NEW_LINE, tmp);
  if(tmp) {
    G_refs x = view->getDocument()->getSelected();

    for(i = 0; i < (int)x.count(); i++) if(x[i]->getType() & G_STRAIGHT) break;

    if(i == (int)x.count()) {
      if(x.count() > 2) newMenu->changeItem(ID_NEW_LINE, tr("&Lines"));
      else newMenu->changeItem(ID_NEW_LINE, tr("&Line"));
    }
    else {
      if(x.count() > 2) newMenu->changeItem(ID_NEW_LINE, tr("Parallel &Lines"));
      else newMenu->changeItem(ID_NEW_LINE, tr("Parallel &Line"));
    }
  }
  else newMenu->changeItem(ID_NEW_LINE, tr("&Line"));

  //perpendicular
  tmp = view->getDocument()->canNewPerpendicular();
  newMenu->setItemEnabled(ID_NEW_PERPENDICULAR, tmp);
  if(tmp && view->getDocument()->selectedCount() > 2)
    newMenu->changeItem(ID_NEW_PERPENDICULAR, tr("&Perpendicular Lines"));
  else newMenu->changeItem(ID_NEW_PERPENDICULAR, tr("&Perpendicular Line"));

  //polygon
  newMenu->setItemEnabled(ID_NEW_POLYGON, view->getDocument()->canNewPolygon());

  //arc sector
  tmp = view->getDocument()->canNewArcSector();
  newMenu->setItemEnabled(ID_NEW_ARCSECTOR, tmp);
  
  //arc segment
  newMenu->setItemEnabled(ID_NEW_ARCSEGMENT, tmp);

  //circle interior
  newMenu->setItemEnabled(ID_NEW_CIRCLEINTERIOR, view->getDocument()->canNewCircleInterior());
  
}


void KSegWindow::updatePlayMenu()
{
  while(playMenu->idAt(0) != ID_PLAY_QUICKPLAY) {
    ASSERT(playMenu->count() > 1);
    
    playMenu->removeItemAt(0);
  }

  std::vector<KSegConstruction *> c = KSegConstruction::getAllConstructions();

  int i;
  for(i = (int)c.size() - 1; i >= 0; --i) {
    if(c[i] == view->getDocument()) continue;
    
    QString f = c[i]->getFilename();
    if(f.isEmpty()) f = QString(tr("Untitled"));
    else {
      f = QFileInfo(f).fileName();
    }
    
    playMenu->insertItem(f, view, SLOT(playConstruction(int)), 0, -1, 0);
    
    int id = playMenu->idAt(0);
    
    playMenu->setItemParameter(id, i);
    playMenu->setItemEnabled(id, view->getDocument()->canPlay(i));
  }
  
  if(playMenu->count() > 1) {
    playMenu->insertSeparator(playMenu->count() - 1);
  }

  updateQuickPlayMenu();
}

void KSegWindow::updateQuickPlayMenu()
{
  quickPlayMenu->clear();

  quickPlayMenu->insertItem(tr("&Change Directory..."), this,
			    SLOT(quickPlaySetDirectory()),
			    0,
			    ID_QUICKPLAY_SET_DIRECTORY);

  quickPlayMenu->insertSeparator();

  vector<QString> files = getValidConstructionFiles();

  int i;
  for(i = 0; i < (int)files.size(); ++i) {
    ScanType s = KSegConstruction::scanConstructionFile(QFile(files[i]),
							view->getDocument()->getSelected());
    if(s == INVALID) continue;

    quickPlayMenu->insertItem(QFileInfo(files[i]).fileName(),
			      this, SLOT(quickPlay(int)));
    
    int id = quickPlayMenu->idAt(quickPlayMenu->count() - 1);
    
    quickPlayMenu->setItemParameter(id, i);
    quickPlayMenu->setItemEnabled(id, s == MATCHED);
  }

  if(quickPlayMenu->count() == 2) { //add dummy item
    quickPlayMenu->insertItem(tr("No Constructions"), this, SLOT(fileOpen())); //dummy slot

    int id = quickPlayMenu->idAt(quickPlayMenu->count() - 1);
    quickPlayMenu->setItemEnabled(id, false);
  }
}

void KSegWindow::quickPlay(int which)
{
  vector<QString> files = getValidConstructionFiles();

  //just in case:
  ScanType s = KSegConstruction::scanConstructionFile(QFile(files[which]),
						      view->getDocument()->getSelected());

  ASSERT(s == MATCHED);

  QFile f(files[which]);

  KSegConstruction temp;

  if(f.open(IO_ReadOnly)) {
    QDataStream stream(&f);
    
    stream >> temp;
    f.close();
  }

  vector<KSegConstruction *> cs = KSegConstruction::getAllConstructions();

  int c = (int)cs.size();

  while(c--) {
    if(cs[c] == &temp) break;
  }
  
  ASSERT(c != -1);
  
  view->playConstruction(c);
}

void KSegWindow::quickPlaySetDirectory()
{
  QFileDialog *fd = new QFileDialog(quickPlayDirectory, "*.sec", 0, 0, true);
  
  fd->setMode(QFileDialog::Directory);
  
  fd->show();
  
  QString newDir = fd->selectedFile();
  
  if(!newDir.isNull()) {
    quickPlayDirectory = newDir;
    KSegProperties::setProperty("QuickPlayDirectory", newDir);
  }
  
  delete fd;
}

//returns a vector of full paths for potential
//filenames in the current directory
vector<QString> KSegWindow::getValidConstructionFiles()
{
  vector<QString> toBeReturned;

  QDir myDir = QDir(quickPlayDirectory);
  myDir.setNameFilter("*.sec");
  myDir.setSorting(QDir::Name);
  myDir.setFilter(QDir::Files | QDir::Readable);

  QStringList entries = myDir.entryList();

  int i;
  for(i = 0; i < (int)entries.count(); ++i) {
    QString entry = entries[i];
    
    toBeReturned.push_back(myDir.absPath() + "/" + entry);
  }

  return toBeReturned;
}

void KSegWindow::updateConstructionMenu()
{
  ASSERT(view->getDocument()->isConstruction());

  KSegConstruction *c = (KSegConstruction *)(view->getDocument());

  constructionMenu->setItemEnabled(ID_CONSTRUCTION_MAKE_NORMAL,
				   c->canConstructionMakeNormal());
  constructionMenu->setItemEnabled(ID_CONSTRUCTION_MAKE_GIVEN,
				   c->canConstructionMakeGiven());
  constructionMenu->setItemEnabled(ID_CONSTRUCTION_MAKE_FINAL,
				   c->canConstructionMakeFinal());
  constructionMenu->setItemEnabled(ID_CONSTRUCTION_MAKE_INITIAL,
				   c->canConstructionMakeInitial());
  constructionMenu->setItemEnabled(ID_CONSTRUCTION_RECURSE,
				   c->canConstructionRecurse());
}



void KSegWindow::keyPressEvent(QKeyEvent *ev)
{
  updateMenus();
  
  QMainWindow::keyPressEvent(ev);

  view->updateStatusBar(ev->stateAfter());
}

//this is a workaround for a seeming bug...
void KSegWindow::keyReleaseEvent(QKeyEvent *ev)
{
  updateMenus();
  
  QMainWindow::keyReleaseEvent(ev);

  view->updateStatusBar(ev->stateAfter());
}


void KSegWindow::closeEvent(QCloseEvent *ev)
{
  if(view->getDocument()->isModified() == false) {
    ev->accept();
    return;
  }

  int ans = QMessageBox::warning(this, tr("Save?"),
				 tr("Would you like to save your changes?"),
				 QMessageBox::Yes, QMessageBox::No,
				 QMessageBox::Cancel);

  if(ans == QMessageBox::Cancel) {
    ev->ignore();
    return;
  }
  if(ans == QMessageBox::Yes) {
    fileSave();
    if(view->getDocument()->isModified() == true) { //check if the guy hit cancel
      ev->ignore();
      return;
    }
  }

  ev->accept();
}


void KSegWindow::newSketch()
{
  (new KSegWindow())->show();
}

void KSegWindow::newConstruction()
{
  KSegConstruction *c = new KSegConstruction;

  (new KSegWindow(c))->show();
}


void KSegWindow::newView()
{
  (new KSegWindow(view->getDocument()))->show();
}

void KSegWindow::fileSaveAs()
{
  QString fileName;

  if(view->getDocument()->isConstruction()) {
    fileName = QFileDialog::
      getSaveFileName(view->getDocument()->getFilename(), "*.sec");

    if(fileName.isNull()) return;
    
    if(!QFileInfo(fileName).fileName().contains('.'))
      fileName = fileName + ".sec";
  }
  else {
    fileName = QFileDialog::
      getSaveFileName(view->getDocument()->getFilename(), "*.seg");

    if(fileName.isNull()) return;

    if(!QFileInfo(fileName).fileName().contains('.'))
      fileName = fileName + ".seg";
  }

  QFile f(fileName);

  if(f.open(IO_WriteOnly)) {
    QDataStream stream(&f);

    stream << *(view->getDocument());
    f.close();
  }

  view->getDocument()->emitDocumentSaved(fileName);
}

void KSegWindow::fileSave()
{
  if(view->getDocument()->getFilename().isEmpty()) {
    fileSaveAs();
    return;
  }

  QString fileName = view->getDocument()->getFilename();

  QFile f(fileName);

  if(f.open(IO_WriteOnly)) {
    QDataStream stream(&f);

    stream << *(view->getDocument());
    f.close();
  }

  view->getDocument()->emitDocumentSaved(fileName);
}

void KSegWindow::fileOpenRecent(int num)
{
  num = num - ID_FILE_RECENTLIST_START;

  QString fileName = KSegProperties::getProperty(QString("RecentFile") + QString::number(num + 1));

  if(fileName.isNull()) return;

  if(openFile(fileName)) {
    int i;

    for(i = num; i > 0; --i) {
      KSegProperties::setProperty("RecentFile" + QString::number(i + 1),
				  KSegProperties::getProperty("RecentFile" + QString::number(i)));
    }

    KSegProperties::setProperty("RecentFile1", fileName);
  }
  else {
    KSegProperties::setProperty("RecentFile" + QString::number(num + 1), QString());
    compressRecentList();
  }
}


void KSegWindow::fileOpen()
{
  QString fileName = QFileDialog::getOpenFileName(QString::null, "*.seg *.sec");

  if(fileName.isNull()) return;

  //if the file is one of the recent ones, call the function
  //because it knows what to do with the recent list
  compressRecentList();

  int count = KSegProperties::getProperty("RecentListSize").toInt();

  int i;
  for(i = 0; i < count; ++i) {
    QString cur = KSegProperties::getProperty("RecentFile" + QString::number(i + 1));
    if(cur == fileName) {
      fileOpenRecent(i + ID_FILE_RECENTLIST_START);
      return;
    }
  }

  if(openFile(fileName)) {
    //add the file name to the recently used list:
    for(i = count - 1; i > 0; --i) {
      KSegProperties::setProperty("RecentFile" + QString::number(i + 1),
				  KSegProperties::getProperty("RecentFile" + QString::number(i)));
    }

    KSegProperties::setProperty("RecentFile1", fileName);
  }
}


bool KSegWindow::openFile(QString fileName)
{
  if(QFileInfo(fileName).exists() == false) {
      QMessageBox::warning(0, tr("File Not Found"),
			   tr("Sorry, I am afraid that file %1 does not exist.").arg(fileName),
			   QMessageBox::Ok, 0);
      return false;
  }

  QFile f(fileName);

  KSegWindow *win = 0;
  bool isConstruction;

  //check to make sure the file won't crash us.
  if(f.open(IO_ReadOnly)) {
    char block[4];
 
    f.readBlock(block, 4);

    if(block[0] || block[1] || block[2] || block[3] + 5 > f.size()) {
      QMessageBox::warning(0, tr("Invalid file"),
			   tr("The file %1 does not look like a valid KSEG file.").arg(fileName),
			   QMessageBox::Ok, 0);
      f.close();
      return false;
    }

    
    f.ungetch(block[3]);
    f.ungetch(block[2]);
    f.ungetch(block[1]);
    f.ungetch(block[0]);
    

    QDataStream stream(&f);

    QString tmp;
    
    stream >> tmp;

    if(tmp.startsWith(QString("KSeg Document Version "))) {
      isConstruction = false;
    }
    else if(tmp.startsWith(QString("KSeg Construction Version "))) {
      isConstruction = true;
    }
    else {
      QMessageBox::warning(0, tr("Invalid file"),
			   tr("The file %1 does not look like a valid KSEG file.").arg(fileName),
			   QMessageBox::Ok, 0);
      f.close();
      return false;
    }


    f.close();
  }

  if(f.open(IO_ReadOnly)) {
    QDataStream stream(&f);
    
    if(isConstruction) {
      win = new KSegWindow(new KSegConstruction);
    }
    else {
      win = new KSegWindow;
    }

    stream >> *(win->view->getDocument());
    f.close();
  }

  win->view->getDocument()->emitDocumentSaved(fileName);
  
  win->show();

  return true;
}


void KSegWindow::filePrint()
{
  QPrinter *pr = view->getDocument()->getPrinter();

  if(pr->setup(this)) {
    pr->setFullPage(false);

    QPainter p(pr);

    QRect docSize = view->getDocument()->getSize();

    //make sure the document is at least 400 by 400
    QPoint oldCenter = docSize.center();
    if(docSize.width() < 400) docSize.setWidth(400);
    if(docSize.height() < 400) docSize.setHeight(400);
    docSize.moveCenter(oldCenter);

    QRect viewPort = p.viewport();

    if(docSize.width() * viewPort.height() >
       docSize.height() * viewPort.width()) {
      //the docSize needs to be made taller
      int oldHeight = docSize.height();
      int h = (docSize.width() * viewPort.height()) / viewPort.width();
      docSize.setHeight(h);
      docSize.moveBy(0, -(h - oldHeight) / 2);
    }
    else {
      //the docSize needs to be made wider
      int oldWidth = docSize.width();
      int w = (docSize.height() * viewPort.width()) / viewPort.height();
      docSize.setWidth(w);
      docSize.moveBy(-(w - oldWidth) / 2, 0);

    }

    p.setWindow(docSize);
    view->getDocument()->print(p);
  }
}


void KSegWindow::fileChooseLanguage()
{
  QString fileName = QFileDialog::getOpenFileName(KSegProperties::getProperty("LanguageDir") +
						  "/" + KSegProperties::getProperty("Language"),
						  "*.qm");

  if(fileName.isNull()) return;

  QString dir = QFileInfo(fileName).dirPath(true);
  QString file = QFileInfo(fileName).fileName();

  QTranslator *t = new QTranslator(0);
  if(t->load(file, dir)) {
    KSegProperties::setProperty("LanguageDir", dir);
    KSegProperties::setProperty("Language", file);
    
    qApp->installTranslator(t);

    QMessageBox::information(0, tr("Language Change"),
			     tr("You will need to restart KSEG for the language "
				"change to fully take effect."),
			     QMessageBox::Ok, 0);

    return;
  }

  delete t;
}



void KSegWindow::editColorBlack()
{
  view->getDocument()->setCurrentColor(black);
  updateEditColorMenu();
}

void KSegWindow::editColorGray()
{
  view->getDocument()->setCurrentColor(gray);
  updateEditColorMenu();
}

void KSegWindow::editColorRed()
{
  view->getDocument()->setCurrentColor(red);
  updateEditColorMenu();
}

void KSegWindow::editColorGreen()
{
  view->getDocument()->setCurrentColor(green);
  updateEditColorMenu();
}

void KSegWindow::editColorBlue()
{
  view->getDocument()->setCurrentColor(blue);
  updateEditColorMenu();
}

void KSegWindow::editColorYellow()
{
  view->getDocument()->setCurrentColor(yellow);
  updateEditColorMenu();
}

void KSegWindow::editColorPurple()
{
  view->getDocument()->setCurrentColor(magenta);
  updateEditColorMenu();
}

void KSegWindow::editColorCyan()
{
  view->getDocument()->setCurrentColor(cyan);
  updateEditColorMenu();
}

void KSegWindow::editColorOther()
{
  QColor newColor;
  
  newColor = QColorDialog::getColor(view->getDocument()->getCurrentColor());

  if(newColor.isValid()) {
    view->getDocument()->setCurrentColor(newColor);
  }

  updateEditColorMenu();
}

//----------

void KSegWindow::editPointstyleLargecircle()
{
  view->getDocument()->setCurrentPointstyle(LARGE_CIRCLE);
  updateEditPointstyleMenu();
}


void KSegWindow::editPointstyleMediumcircle()
{
  view->getDocument()->setCurrentPointstyle(MEDIUM_CIRCLE);
  updateEditPointstyleMenu();
}


void KSegWindow::editPointstyleSmallcircle()
{
  view->getDocument()->setCurrentPointstyle(SMALL_CIRCLE);
  updateEditPointstyleMenu();
}

//---------

void KSegWindow::editLinestyleSolid()
{
  view->getDocument()->setCurrentPenstyle(SolidLine);
  updateEditLinestyleMenu();
}


void KSegWindow::editLinestyleDashed()
{
  view->getDocument()->setCurrentPenstyle(DashLine);
  updateEditLinestyleMenu();
}


void KSegWindow::editLinestyleDotted()
{
  view->getDocument()->setCurrentPenstyle(DotLine);
  updateEditLinestyleMenu();
}


void KSegWindow::editLinestyleThin()
{
  view->getDocument()->setCurrentPenWidth(0);
  updateEditLinestyleMenu();
}


void KSegWindow::editLinestyleNormal()
{
  view->getDocument()->setCurrentPenWidth(2);
  updateEditLinestyleMenu();
}


void KSegWindow::editLinestyleThick()
{
  view->getDocument()->setCurrentPenWidth(3);
  updateEditLinestyleMenu();
}

//---------

void KSegWindow::editFont10()
{
  view->getDocument()->setCurrentFontsize(10);
  updateEditFontMenu();
}
void KSegWindow::editFont12()
{
  view->getDocument()->setCurrentFontsize(12);
  updateEditFontMenu();
}
void KSegWindow::editFont14()
{
  view->getDocument()->setCurrentFontsize(14);
  updateEditFontMenu();
}
void KSegWindow::editFont20()
{
  view->getDocument()->setCurrentFontsize(20);
  updateEditFontMenu();
}
void KSegWindow::editFont30()
{
  view->getDocument()->setCurrentFontsize(30);
  updateEditFontMenu();
}

void KSegWindow::editFontFont()
{
  bool ok;
  QFont f = QFontDialog::getFont(&ok, view->getDocument()->getCurrentFont());
  if(ok) {
    view->getDocument()->setCurrentFont(f);
  }
  updateEditFontMenu();
}

void KSegWindow::helpAbout()
{
  QMessageBox::about(this, tr("About KSEG"),
		     tr("KSEG v0.3\n"
			"Copyright (C) 1999-2001 Ilya Baran\n"
			"http://www.mit.edu/~ibaran\n"
			"ibaran@mit.edu"));
}


#include "KSegWindow.moc"

