#ifndef PLUGINLOADER_H
#define PLUGINLOADER_H

#include <kore/kore.h>
#include <kore/serviceprovider.h>

namespace kore
{

class Plugin;
/**
 * Each Kore dll has to export a function with this signature, in order
 * to be used as a Kore plugin (this function is similar to main()):
 * extern "C" {
 *      Plugin* plugin(HMODULE libHandle, const char* libName, const char* libPath, int libFlags );
 * }
 * The default implementation for this function should look like:
 * Plugin* plugin(HMODULE libHandle, const char* libName, const char* libPath, int libFlags )
 * {
 *      return new Plugin( libHandle, libName, libPath, libFlags );
 * }
 * See also the PLUGIN_MAIN_HDR and PLUGIN_MAIN_BODY macros in plugin.h for defaults.
 * The PluginLoader will open the dll, and try to run the plugin(...) function
 * in order to get a Plugin off the dll.
 */
typedef Plugin* (*PluginFuncType)(HMODULE libHandle, const char* libName, const char* libPath, int libFlags );

/**
 * class kore::PluginLoader - platform-independant loader for dll's (dynamic library).
 * It provides methods for loading (openPlugin()) and initialising (runPlugin()) plugins.
 */
class KORE_API PluginLoader: public ServiceProvider
{
public:
    /**
     * Default constructor. Creates a new PluginLoader instance.
     */
    PluginLoader();
    /**
     * Destructor.
     */
    ~PluginLoader();
    /**
     * Loads a dll and gets the Plugin object for it.
     * @param libName - the platform-independant library name. Please note that
     * this is NOT the actual filename of the dll. I.e. if you use:
     * openPlugin("my_plugin"), the Plugin loader will look for a dll named
     * "libmy_plugin.so" (or "my_plugin.dll") and try to load it.
     * @param libPath - the filesystem path were to look for the plugin (optional).
     * @param libFlags - system-dependant flags to be used when loading the dll.
     * @return - the Plugin object associated to the given libName, if the dll
     * was found, NULL otherwise.
     */
    virtual Plugin* openPlugin(const char* libName, const char* libPath = 0, int libFlags = 0);
    /**
     * Loads a dll, gets the dll's Plugin object and initialises it.
     * @param libName - the platform-independant library name. Please note that
     * this is NOT the actual filename of the dll. I.e. if you use:
     * openPlugin("my_plugin"), the Plugin loader will look for a dll named
     * "libmy_plugin.so" (or "my_plugin.dll") and try to load it.
     * @param libPath - the filesystem path were to look for the plugin (optional).
     * @param libFlags - system-dependant flags to be used when loading the dll.
     * @return - the Plugin object associated to the given libName, if the dll
     * was found, NULL otherwise.
     */
    virtual Plugin* runPlugin(const char* libName, const char* libPath = 0, int libFlags = 0);
    /**
     * Unloads a previously loaded dll.
     * @param plugin - the Plugin object to be closed.
     */
    virtual void closePlugin(Plugin* plugin);
    /**
     * Gets the error message for the last failed openPlugin() or runPlugin().
     * @return - the last error message generated by openPlugin() or runPlugin().
     */
    virtual const char* lastError();

    /**
     * Helper method. It converts a platform-independant dll name (ie. "my_plugin")
     * to the corresponding platform-dependant dll filename (ie. "libmy_plugin.so",
     * or "my_plugin.dll").
     * @param libName - the platform-independant dll name.
     * @return - the platform-dependant dll filename.
     */
    virtual char* libName2fileName(const char* libName);
    /**
     * Helper method. It converts a platform-dependant dll filename
     * (ie. "libmy_plugin.so", or "my_plugin.dll") to the
     * corresponding platform-independant dll filename  (ie. "my_plugin")
     * @param fileName - the platform-dependant dll filename.
     * @return - the platform-independant dll name.
     */
    virtual char* fileName2libName(const char* fileName);
private:
    // current PLDR version
    const Version* _loaderVersion;
    // Kernel API version required by PLDR
    const Version* _loaderAPIVersion;
    // PLDR info
    const Info* _loaderInfo;
    // basic PLDR service
    const Service* _loaderService;
    // last error string
    char _lastError[100];
};

};

#endif
