/***************************************************************************
 *   Copyright (C) 2003 by Sylvain Joyeux                                  *
 *   sylvain.joyeux@m4x.org                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <config.h>
#ifndef USE_RPM

#include "dpkg.h"
#include "debug.h"

#include <klocale.h>
#include <kdebug.h>

#include <qstringlist.h>
#include <qregexp.h>



Dpkg::Dpkg(QObject *parent, const char *name)
 : PackageManager(parent, name)
{
  connect(&m_process, SIGNAL(readReady (KProcIO *)), this, SLOT(readReady(KProcIO*)));
}

Dpkg::~Dpkg()
{
}

int Dpkg::capabilities( int query ) const
{
	if ( (query & SEARCH_FILE) && (query & OFFLINE) )
  	return query | INSTALLED_ONLY;
	if ( (query & LIST_FILES) && (query & OFFLINE) )
  	return query | INSTALLED_ONLY;
	if ( query & ONLINE )
  	return query;
	return NOT_SUPPORTED;
}

void Dpkg::readReady(KProcIO*)
{
  bool partial;

  QString newline;
  QStringList lines;
  while(m_process.readln(newline, true, &partial) != -1)
  {
    if (partial) m_buffer += newline;
    else
    {
      newline.truncate(newline.length());
      QString line(m_buffer + newline);
      lines << line;
      m_buffer = "";
    }

  }

  (this->*m_receive)(lines);
}

bool Dpkg::search( const QString & file )
{
  m_process.resetAll();
  m_buffer = QString::null;

  m_process.clearArguments();
  m_process << "dpkg" << "-S" << file;
  m_receive = &Dpkg::receiveSearch;
  return m_process.start(KProcess::Block, KProcess::Stdout );
}

void Dpkg::receiveSearch( const QStringList & line )
{
  static QRegExp rx_notfound("dpkg: (.*) not found");
  // the format of the dpkg -S answer is
  // package1[, package2[, package3...]]: file
  for (QStringList::ConstIterator i = line.begin(); i != line.end(); ++i)
  {
    //kdDebug(DEBUG_ZONE) << *i << endl;
    if ((*i).isEmpty()) continue;
    if (rx_notfound.exactMatch(*i))
    {
      emit token("error", i18n("%1 not found").arg(rx_notfound.cap(1)));
      continue;
    }

    int semicolon = (*i).find(':');
    if (semicolon == -1)
    {
      kdDebug(DEBUG_ZONE) << "receiveSearch unmatched line : " << *i << endl;
      continue;
    }
    QStringList packages = QStringList::split(',', (*i).left(semicolon));
    QString file = (*i).right( (*i).length() - semicolon - 1 );

    emit token("file", file.stripWhiteSpace());

    for (QStringList::ConstIterator j = packages.begin(); j != packages.end(); ++j)
      emit token("package", (*j).stripWhiteSpace());
  }
}

bool Dpkg::list( const QString & package )
{
  m_process.resetAll();
  m_buffer = QString::null;

  m_process.clearArguments();
  m_process << "dpkg" << "-L" << package;
  m_receive = &Dpkg::receiveList;
  return m_process.start(KProcess::Block, KProcess::Stdout );
}

void Dpkg::receiveList( const QStringList & line )
{
  static QRegExp rx_notfound("Package (.*) is not installed");
  for (QStringList::ConstIterator i = line.begin(); i != line.end(); ++i)
  {
    if (rx_notfound.search(*i) > -1)
      emit token("error", i18n("Package %1 is not installed").arg(rx_notfound.cap(1)));
    else if ((*i).startsWith("/"))
      emit token("file", *i);
  }
}

static const QString
  html_form_begin("\n<form action=\"http://packages.debian.org/cgi-bin/search_contents.pl\" method=\"GET\">\n"
                  "<table class=\"query\">\n");

static QString make_title(const QString& title)
{	return "\t<tr><td class=\"title\" colspan=\"2\">" + title + "</td></tr>\n"; }
static const QString
  html_form_end("<tr>\n"
                "\t<td class=\"button\" colspan=\"2\">\n"
                	"\t\t<input type=\"submit\" value=\"%1\">\n"
  								"\t\t<input type=\"hidden\" name=\"searchmode\" value=\"searchfilesanddirs\">\n"
                  "\t\t<input type=\"hidden\" name=\"case\" value=\"insensitive\">\n"
          			"\t</td>\n"
                "</tr>\n"
                "</table>\n"
                "</form>\n");

static const QString
  html_form_line_begin("<tr>\n"
                 "\t<td><label for=\"%1\">%2</label></td>\n"
                 "\t<td>\n");
static const QString
	html_form_line_end("</td>\n</tr>\n");

static const QString html_form_combo("<select name=\"%1\" id=\"%2\">");

static QString make_form_text(const QString& type, const QString& label)
{
	return
  	html_form_line_begin.arg(type).arg(label)
   + QString("<input type=\"text\" name=\"%1\" id=\"%2\">").arg(type).arg(type)
   + html_form_line_end;
}

static QString begin_form_combo(const QString& type, const QString& label)
{
	return
  	html_form_line_begin.arg(type).arg(label)
   + QString("\t<select name=\"%1\" id=\"%2\">\n").arg(type).arg(type);
}
static QString make_form_option(const QString& name, const QString& text)
{ return "\t\t<option value=" + name + ">" + text + "</option>\n"; }
static QString end_form_combo()
{ return "\t</select>\n\t</td>\n</tr>\n"; }

QString Dpkg::getOnlineForm()
{
	QString buffer;
  QTextOStream stream(&buffer);
  stream
  	<< html_form_begin
    << make_title( i18n("packages.debian.org"))

    << make_form_text("word", i18n("File to search"))
    << begin_form_combo("arch", i18n("Architecture"))
      << make_form_option("i386", i18n("Intel x86"))
      << make_form_option("amd64", i18n("AMD64"))
      << make_form_option("m68k", i18n("Motorola 680x0"))
      << make_form_option("alpha", i18n("Alpha"))
      << make_form_option("sparc", i18n("SPARC"))
      << make_form_option("powerpc", i18n("PowerPC"))
      << make_form_option("arm", i18n("ARM"))
      << make_form_option("hppa", i18n("HP PA/RISC"))
      << make_form_option("ia64", i18n("Intel IA-64"))
      << make_form_option("mips", i18n("MIPS"))
      << make_form_option("mipsel", i18n("MIPS (DEC)"))
      << make_form_option("s390", i18n("IBM S/390"))
    << end_form_combo()
    << begin_form_combo("version", i18n("Version"))
    <<	make_form_option("stable", "stable")
    <<	make_form_option("testing", "testing")
    <<	make_form_option("unstable", "unstable")
    << end_form_combo()

    << html_form_end.arg(i18n("Go online!"));

  return buffer;
}

#include "dpkg.moc"
#endif

