/*
* page_export_mjpeg.cc -- Notebook Firewire/AVI/Still Frame Export Page Object
* Copyright (C) 2002 Charles Yates <charles.yates@pandora.be>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fstream>
#include <iostream>
#include <iomanip>
#include <vector>
#include <sstream>
using std::cerr;
using std::endl;

#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "page_export_mjpeg.h"
#include "kino_av_pipe.h"
#include "preferences.h"
#include "kino_common.h"
#include "frame.h"
#include "page_editor.h"
#include "message.h"

/** Constructor for page.

\param _exportPage page to which this page belongs
\param _common KinoCommon object
*/

ExportMJPEG::ExportMJPEG( PageExport *_exportPage, KinoCommon *_common ) :
		Export( _exportPage, _common )
{
	cerr << "> Creating ExportMJPEG Page" << endl;

	/* Get pointers to own controls */
	fileEntry
	= GTK_ENTRY( lookup_widget( common->getWidget(), "entry_export_mjpeg_file" ) );

	mjpegFormat
	= GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( lookup_widget( exportPage->getWidget(),
	                                      "optionmenu_export_mjpeg_format" ) ) ) );
	deinterlaceMenu
	= GTK_MENU( gtk_option_menu_get_menu( GTK_OPTION_MENU( lookup_widget( common->getWidget(),
	                                      "optionmenu_export_mjpeg_deinterlace" ) ) ) );
	mjpegVideo
	= GTK_ENTRY( lookup_widget( exportPage->getWidget(),
	                            "entry_mjpeg_video" ) );
	mjpegAudio
	= GTK_ENTRY( lookup_widget( exportPage->getWidget(),
	                            "entry_mjpeg_audio" ) );
	mjpegMultiplexer
	= GTK_ENTRY( lookup_widget( exportPage->getWidget(),
	                            "entry_mjpeg_multiplexer" ) );
	splitCheck
	= GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(),
	                                    "checkbutton_export_mjpeg_split" ) );
	cleanupCheck
	= GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(),
					"checkbutton_export_mjpeg_cleanup" ) );
	authorCheck
	= GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(),
	        			"checkbutton_export_mjpeg_author" ) );
}

/** Destructor for page.
 */

ExportMJPEG::~ExportMJPEG()
{
	cerr << "> Destroying ExportMJPEG Page" << endl;
}

/** start exporting MJPEG
 */
enum export_result
ExportMJPEG::doExport( PlayList * playlist, int begin, int end, int every,
                       bool preview )
{
	cerr << ">>> ExportMJPEG::startExport" << endl;

	// TODO: Fix this buffer stuff.
	gchar *filename;
	enum export_result status = EXPORT_RESULT_SUCCESS;

	filename = g_strdup( gtk_entry_get_text( fileEntry ) );

	if ( !( strcmp( filename, "" ) && strpbrk( filename, "\'" ) == NULL ) )
	{
		modal_message( _( "You must enter a filename which doesn't contain quotes." ) );
		g_free( filename );
		return EXPORT_RESULT_FAILURE;
	}
	unsigned char *pixels = new unsigned char[ FRAME_MAX_WIDTH * FRAME_MAX_HEIGHT * 4 ];
	Frame* frame = GetFramePool()->GetFrame();

	// Get a sample frame to obtain recording info
	// TODO: Sample frames are problematic...
	playlist->GetFrame( begin, *frame );
	frame->decoder->quality = DV_QUALITY_BEST;

	// Get all video and audio info required
	int width = frame->GetWidth();
	int height = frame->GetHeight();
	bool isPAL = frame->IsPAL();
	bool isWide = frame->IsWide();
	AudioInfo info;
	frame->GetAudioInfo( info );
	short channels = info.channels;
	int frequency = info.frequency;

	// Force audio resampling to that of the first frame
	AudioResample *resampler = AudioResampleFactory::createAudioResample(
	                               AUDIO_RESAMPLE_SRC_SINC_BEST_QUALITY, frequency );

	// vars for collecting audio and video pipe commands
	bool split = false;
	bool author = false;
	bool Usercleanup = true;
	kino_video_pipe type = PIPE_VIDEO_MJPEG;
	gchar *audiopipe = NULL;
	gchar *videopipe = NULL;
	gchar *multiplex = NULL;
	gchar *cleanup = NULL;

	// list to collect the names of the generated mpeg2 files
	// this collection only proceeds if we are authoring an
	// xml file for use with dvdauthor
	std::vector<std::string> nameList;

	// MPEG and DIVX translations to commands
	// What started off as a simple addition is getting pretty damn messy :-/ - ho hum
	// Essentially the important things to remember are that only the 5 variables above
	// are populated in the following kludge... (this should move into seprate methods)
	const char *videopipeInput = "";
	int videoformat = 0;
	int deinterlaceRequest = 0;
	const char *audioencode = "";
	const char *multiplexer = "";

	videopipeInput = gtk_entry_get_text( mjpegVideo );
	GtkWidget *active_item = gtk_menu_get_active( mjpegFormat );
	videoformat
	= g_list_index ( GTK_MENU_SHELL ( mjpegFormat ) ->children, active_item );
	audioencode = gtk_entry_get_text( mjpegAudio );
	multiplexer = gtk_entry_get_text( mjpegMultiplexer );
	active_item = gtk_menu_get_active( deinterlaceMenu );
	deinterlaceRequest
	= g_list_index ( GTK_MENU_SHELL ( deinterlaceMenu ) ->children, active_item );
	split = gtk_toggle_button_get_active( splitCheck );
	Usercleanup = gtk_toggle_button_get_active( cleanupCheck );
	/* only necessary to check this option if we are creating DVD video */
	if( videoformat == 6 )
	{
		author = gtk_toggle_button_get_active( authorCheck );
	}
	
	/* force deinterlace on MPEG-1 formats if not specified by user */
	if ( videoformat < 3 && deinterlaceRequest == 0 )
		deinterlaceRequest = 1;

	char deinterlace[ 30 ] = "";
	char interlace = '0';
	char audioOptions[ 20 ] = "";

	switch ( deinterlaceRequest )
	{
	case 0:
		interlace = '1';
		break;
	case 1:
		strcpy( deinterlace, "yuvdenoise -F -f |" );
		break;
	case 2:
		strcpy( deinterlace, "yuvdenoise -F |" );
		break;
	case 3:
		type = PIPE_VIDEO_DEINTERLACED_MJPEG;
		break;
	}

	if ( !strstr( videopipeInput, "yuvscaler" ) )
	{
		char scale[ 20 ] = "";
		char bitrate[ 20 ] = "";

		switch ( videoformat )
		{
		case 1:
			strcpy( scale, "VCD" );
			break;
		case 2:
			strcpy( scale, "VCD" );
			strcpy( bitrate, "-b 1152" );
			break;
		case 3:
			strcpy( bitrate, "-b 4000" );
			break;
		case 4:
			strcpy( scale, "SVCD" );
			break;
		case 5:
			strcpy( scale, "SVCD" );
			strcpy( bitrate, "-b 2500" );
			break;
		case 6:
			videoformat = 8;
			strcpy( scale, "" );
			if ( !strstr( audioencode, " -r " ) )
				sprintf( audioOptions, "%s -r 48000", audioOptions );
			if ( !strstr( audioencode, " -b " ) )
				sprintf( audioOptions, "%s -b 192", audioOptions );
			break;
		}

		// If the bitrate is specified then remove the defaults above
		if ( strstr( videopipeInput, " -b " ) || strstr( videopipeInput, "--video-bitrate" ) )
			strcpy( bitrate, "" );

		// Yuck (!) - builds the video pipe depending on data collected above...
		if ( strcmp( scale, "" ) )
			videopipe = g_strdup_printf(
			         "%s yuvscaler -v 0 -O %s -n %c | %s -f %d -I %c -n %c -a %c %s -o \'%s\'.mpv",
			         deinterlace,
			         scale, isPAL ? 'p' : 'n', videopipeInput, videoformat, interlace,
			         isPAL ? 'p' : 'n', isWide ? '3' : '2', bitrate, filename );
		else
			videopipe = g_strdup_printf( "%s %s -f %d -I %c -n %c -a %c %s -o \'%s\'.mpv",
			         deinterlace, videopipeInput, videoformat, interlace,
			         isPAL ? 'p' : 'n', isWide ? '3' : '2', bitrate, filename );
	}
	else
	{
		videopipe = g_strdup_printf( "%s %s -f %d -I %c -n %c -o \'%s\'.mpv\n",
		         deinterlace, videopipeInput, videoformat, interlace,
		         isPAL ? 'p' : 'n', filename );
	}
	audiopipe = g_strdup_printf( "|%s %s -o \'%s\'.mp2", audioencode, audioOptions, filename );
	if ( strcmp( multiplexer, "" ) )
	{
		multiplex = g_strdup_printf( "%s -f %d -o \'%s\'%%s.mpeg \'%s\'.mpv \'%s\'.mp2",
		         multiplexer, videoformat, filename, filename, filename );
		cleanup = g_strdup_printf( "rm -f \'%s\'.mpv \'%s\'.mp2 \'%s\'.wav",
		         filename, filename, filename );
	}

	KinoAudioPipe *wav = KinoAudioFactory::CreateAudioPipe( PIPE_AUDIO_WAV );
	KinoVideoPipe *video = KinoVideoFactory::CreateVideoPipe( type );
	KinoAVPipe *mpeg = new KinoAVPipe( wav, video );

	// Iterate through each scene (opening and closing audio pipes as requested)
	int scene = 0;

	for ( int sceneBegin = begin; sceneBegin <= end && exportPage->isExporting;
	        scene ++ )
	{
		gchar *full;
		char sceneString[ 20 ];
		int sceneEnd = end;
		bool userMplexSplit = ( strstr( multiplex, "-S" ) || strstr( multiplex, "--max-segment-size" ) );

		// Determine if we need to split by scene or not
		// when splitting: Output is generated as fileNNN_MMM.mpeg where NNN is the
		// kino scene number and MMM is the mplex split
		// when not splitting: Output is generated file as fileMMM.mpeg
		sceneString[0] = '\0';
		if ( split )
		{
			sprintf( sceneString, "%03d", scene );
			if ( userMplexSplit )
				strcat( sceneString, "_%03d" );
			sceneEnd = playlist->FindEndOfScene( sceneBegin );
			if ( sceneEnd > end )
				sceneEnd = end;
		}
		else if ( userMplexSplit )
		{
			strcpy( sceneString, "%03d" );
		}

		if ( author )
		{
		  	char tmpFilename[512];

		   	if ( split && userMplexSplit ) {
             	snprintf( tmpFilename, sizeof( tmpFilename ), "%s%03d_001.mpeg", filename, scene );
      		} else if ( split ) {
		  		snprintf( tmpFilename, sizeof( tmpFilename ), "%s%03d.mpeg", filename, scene );
      		} else if ( userMplexSplit ) {
		  		snprintf( tmpFilename, sizeof( tmpFilename ), "%s001.mpeg", filename );
      		} else {
		  		snprintf( tmpFilename, sizeof( tmpFilename ), "%s.mpeg", filename );
			}

			nameList.push_back( std::string( tmpFilename ) );
		}

		cerr << ">>> Generated video pipe '" << videopipe << "'" << endl;
		cerr << ">>> Generated audio pipe '" << audiopipe << "'" << endl;

		mpeg->OpenAudio( audiopipe, channels, frequency, 2 );
		mpeg->OpenVideoPipe( videopipe, width, height );

		int imagesize = width * height * 3;

		/* Iterate over frames in scene */
		for ( int i = sceneBegin; i <= sceneEnd && exportPage->isExporting;
		        i += every )
		{
			/* Call innerLoopUpdate */
			innerLoopUpdate( i, begin, end, every );

			playlist->GetFrame( i, *frame );

			resampler->Resample( *frame );
			if ( !mpeg->OutputAudioFrame( resampler->output, resampler->size ) )
			{
				modal_message( _( "Error writing to KINO/MJPEG audio filter - aborting." ) );
				status = EXPORT_RESULT_FAILURE;
				break;
			}

			if ( exportPage->isExporting )
			{
				if ( type == PIPE_VIDEO_MJPEG )
					frame->ExtractYUV( pixels );
				else
					frame->ExtractRGB( pixels );
				if ( !mpeg->OutputVideoFrame( pixels, imagesize ) )
				{
					modal_message( _( "Error writing to KINO/MJPEG video filter - aborting." ) );
					status = EXPORT_RESULT_FAILURE;
					break;
				}
			}

			/* Update progressbar ... */
			exportPage->updateProgress( ( gfloat ) ( i + every - begin )
			                            / ( gfloat ) ( end + 1 - begin ) );
		}

		if ( status == EXPORT_RESULT_SUCCESS && !exportPage->isExporting )
			status = EXPORT_RESULT_ABORT;

		if ( !split )
			exportPage->isExporting = false;

		// Try to finish the stream to create a valid mpeg if possible
		mpeg->CloseAudio();
		mpeg->CloseVideo();

		if ( multiplex != NULL && strcmp( multiplex, "" ) )
		{
			// expand the template to accomodate the scene splits
			full = g_strdup_printf( multiplex, sceneString );
			cerr << ">>> Executing '" << full << "'" << endl;
			if ( 0 != mpeg->ExecuteCommand( full ) )
			{
				// If mplex error, use user setting
				//do_cleanup = Usercleanup;
			}
			g_free( full );
		}

		if ( Usercleanup )
				mpeg->ExecuteCommand( cleanup );
		
		// Move to start of next scene
		sceneBegin = sceneEnd + 1;
	}

	if ( author )
     	createAuthorXml( filename, nameList, split, isPAL );

	g_free( filename );
	g_free( audiopipe );
	g_free( videopipe );
	g_free( multiplex );
	g_free( cleanup );
	delete mpeg;
	delete video;
	delete wav;
	delete resampler;
	delete pixels;

	GetFramePool()->DoneWithFrame( frame );
	
	return status;
}


/* added 2004-01-08
 * Greg Hookey <g@deadly.ca>
 */
void
ExportMJPEG::createAuthorXml( const char *filename,
							  std::vector<std::string>& nameList,
							  bool split, bool isPal )
{
	if( filename == NULL )
	{
	  std::cerr << ">>> ExportMJPEG::createAuthorXml (file name null)" << std::endl;
	  return;
	}

	char filenameBuffer[512];

	snprintf( filenameBuffer, sizeof( filenameBuffer ), "%s-dvdauthor.xml", filename );

	std::cerr << ">>> ExportMJPEG::createAuthorXml (filename: " << filenameBuffer << ")" << std::endl;

	std::stringstream xml;

	xml << "<?xml version=\"1.0\"?>" << std::endl;
	xml << "<dvdauthor>" << std::endl;

	xml << "\t<vmgm>" << std::endl;
	xml << "\t\t<menus>" << std::endl;
	xml << "\t\t\t<video />" << std::endl;
	xml << "\t\t\t<audio />" << std::endl;
	xml << "\t\t\t<subpicture lang=\"en\" />" << std::endl;
	xml << "\t\t</menus>" << std::endl;
	xml << "\t</vmgm>" << std::endl;

	xml << "\t<titleset>" << std::endl;
	xml << "\t\t<titles>" << std::endl;
	xml << "\t\t\t<pgc pause=\"0\">" << std::endl;

	std::vector<std::string>::iterator iter = nameList.begin();

	if( split == true )
	{
		/* output the list list of vobs */
		while( iter != nameList.end() )
		{
			xml << "\t\t\t\t<vob file=\"" << *iter << "\" chapters=\"\" pause=\"0\" />" << std::endl;
			iter++;
		}
	}
	else
	{
		/* retrieve the vector of scenes, assuming that the vector
		 * doesn't have the 0 entry.
		 */
		std::vector<int> sceneStarts = this->common->getPageEditor()->GetScene();
		std::vector<int>::iterator sceneIter = sceneStarts.begin();
		std::stringstream chapterList;

		chapterList << "0";

		for( sceneIter = sceneStarts.begin(); sceneIter != sceneStarts.end() - 1; ++sceneIter )
		{
			std::string tmpSceneStr;
			tmpSceneStr = this->common->getTime().parseFramesToString( *sceneIter, SMIL::Time::TIME_FORMAT_SMPTE );
			chapterList << "," << tmpSceneStr.substr( 1, 7 ) << "." << std::setw( 2 ) << std::setfill( '0' ) <<
				static_cast< int >( atof( tmpSceneStr.substr( 9, 2 ).c_str() ) / ( isPal ? 25 : 29.97 ) * 100 );
		}

		/* output the single vob, checking for chapter marks */
		xml << "\t\t\t\t<vob file=\"" << *iter << "\" chapters=\"" <<
		chapterList.str() << "\" pause=\"0\" />" << std::endl;
	}

	xml << "\t\t\t</pgc>" << std::endl;
	xml << "\t\t</titles>" << std::endl;
	xml << "\t</titleset>" <<std::endl;

	xml << "</dvdauthor>" << std::endl;

	std::cerr << xml.str() << std::endl;

	std::ofstream out( filenameBuffer );

	if( out.is_open() == true )
	{
		out << xml.str();
		out.flush();
	}

	out.close(); 
}

extern "C"
{
	void
	on_button_export_mjpeg_file_clicked    (GtkButton       *button,
                                            gpointer         user_data)
	{
		const char *filename = common->getFileToSave( _("Enter a File Name to Save As") );
		gtk_widget_grab_focus( lookup_widget( GTK_WIDGET( button ), "entry_export_mjpeg_file" ) );
		if ( strcmp( filename, "" ) )
			gtk_entry_set_text( GTK_ENTRY( lookup_widget( GTK_WIDGET( button ), "entry_export_mjpeg_file" ) ), filename );
	}
}
