/*
 * page_export_stills.cc Notebook Firewire/AVI/Still Frame Export Page Object
 * Copyright (C) 2001 Dan Dennedy <dan@dennedy.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <vector>
#include <iostream>
using std::cout;
using std::endl;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>

#include "page_export_stills.h"
#include "preferences.h"
#include "kino_common.h"
#include "frame.h"
#include "page_editor.h"
#include "message.h"

extern "C" {
	#include "support.h"
	#include "callbacks.h"
	#include "interface.h"
	#include "commands.h"
}

/** Constructor for page.

  	\param common	common object to which this page belongs
*/

ExportStills::ExportStills( PageExport *exportPage, KinoCommon *common ) : exportQuality(80)
{
	cout << "> Creating ExportStills Page" << endl;
	this->exportPage = exportPage;
	this->common = common;
	recordButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "togglebutton_export_stills_record" ) );
	stopButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "togglebutton_export_stills_stop" ) );

	fileEntry = GTK_ENTRY( gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( 
		lookup_widget( common->getWidget(), "gnome_fileentry_export_stills" ) ) ) );
	allToggle = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "radiobutton_export_stills_all") );
	currentToggle = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "radiobutton_export_stills_current") );
	rangeToggle = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "radiobutton_export_stills_range" ) );
	startSpin = GTK_ENTRY( lookup_widget( exportPage->getWidget(), "spinbutton_export_stills_range_start" ) );
	endSpin = GTK_ENTRY( lookup_widget( exportPage->getWidget(), "spinbutton_export_stills_range_end" ) );	

}

/** Destructor for page.
*/

ExportStills::~ExportStills() {
	cout << "> Destroying ExportStills Page" << endl;
}

/** Start of page.
*/

void ExportStills::start() {
	cout << ">> Starting ExportStills" << endl;
	
}

/** Define active widgets.
*/

gulong ExportStills::activate() {
	cout << ">> ExportStill activate" << endl;
	return SCENE_LIST;
}

/** Leaving the page
*/

void ExportStills::clean() {
	cout << ">> Leaving ExportStills" << endl;
}


/** start exporting still frames
*/
void ExportStills::startExport() {
    static unsigned char pixels[720*576*4];
	static Frame frame;
	GdkImlibSaveInfo saveInfo = {0, 0, 0, 0, 0, 0};
	GdkImlibImage *image = NULL;
	int begin = 0;
	int end = 0;
	char file[512];
	char filename[512];
	char extension[8];

	gtk_toggle_button_set_active( recordButton, TRUE );
	gtk_toggle_button_set_active( stopButton, FALSE );
	exportPage->exportMutex = false;

	strcpy( file, gtk_entry_get_text( fileEntry ) );

	if ( strcmp(file, "") ) {

		if (gtk_toggle_button_get_active( allToggle )) {
			begin = 0;
			end = common->getPlayList()->GetNumFrames() - 1;
		} else if (gtk_toggle_button_get_active( currentToggle )) {
			begin = end = common->g_currentFrame;
		} else if (gtk_toggle_button_get_active( rangeToggle )) {
			begin = atoi( gtk_entry_get_text( startSpin ) );
			end = atoi( gtk_entry_get_text( endSpin ) );
		}
		saveInfo.quality = (int) ( (float) exportQuality / 100.0 * 256.0 );
		exportPage->resetProgress();
	
		/* parse the base filename and extension */
		strcpy( filename, file);
		/* make sure a file extension is supplied */
		char *tmp = strrchr( filename, '.');
		if (tmp == NULL) {
			modal_message( "You must enter a filename with an extension" );
		} else {
			tmp[0] = '\0';
			strcpy( extension, strrchr( file, '.'));
		}
	
		for (int i = begin; i <= end && exportPage->isExporting; i++ ) {
			common->getPlayList()->GetFrame( i, frame );
		    frame.ExtractRGB( pixels );
			image = gdk_imlib_create_image_from_data
		                      (pixels, NULL, 720, frame.IsPAL() ? 576 : 480);
			sprintf( file, "%s_%10.10i%s", filename, i, extension);
		    gdk_imlib_save_image( image, file, &saveInfo );
		    gdk_imlib_destroy_image( image );
			exportPage->updateProgress( (gfloat) (i - begin) / (gfloat) (end - begin) );
		}
		
		exportPage->isExporting = false;
		exportPage->exportMutex = true;
		gtk_toggle_button_set_active( recordButton, FALSE );
		gtk_toggle_button_set_active( stopButton, TRUE );
		exportPage->exportMutex = false;
		
	} else {
		modal_message( "You must enter a filename." );

		exportPage->isExporting = false;
		exportPage->exportMutex = true;
		gtk_toggle_button_set_active( recordButton, FALSE );
		gtk_toggle_button_set_active( stopButton, TRUE );
		exportPage->exportMutex = false;
	}
}


/** stop the exporting still frames
*/
void ExportStills::stopExport() {
	/* do nothing, let the export page clear the export flag
	   to fall out of the loop.
	*/
}

/** put the scene begin and end frame numbers into spinners

    \param i the numerical index position of the scene in the playlist1
*/
void ExportStills::selectScene( int i ) {
	int begin = 0;
	int end = 0;
	GtkAdjustment *adjust = gtk_spin_button_get_adjustment( GTK_SPIN_BUTTON( startSpin ) );
	
	gtk_toggle_button_set_active( rangeToggle, TRUE );
	
	cout << ">>> ExportStills selectScene " << i << endl;
	vector <int> scene = common->getPageEditor()->GetScene();

	begin = i == 0 ? 0 : scene[ i - 1 ];
	adjust->lower = 0;
	adjust->upper = scene[ scene.size() - 1 ];
	gtk_spin_button_set_value( GTK_SPIN_BUTTON( startSpin ), begin );
	gtk_signal_emit_by_name( GTK_OBJECT( adjust ), "changed" );

	adjust = gtk_spin_button_get_adjustment( GTK_SPIN_BUTTON( endSpin ) );
	adjust->lower = 0;
	adjust->upper = scene[ scene.size() - 1 ];
	end = scene[i];	
	gtk_spin_button_set_value( GTK_SPIN_BUTTON( endSpin ), end );
	gtk_signal_emit_by_name( GTK_OBJECT( adjust ), "changed" );
}

// callbacks
extern "C" {
	extern KinoCommon *common;

	void setExportQuality( int i ) {
		common->getPageExport()->getPageExportStills()->setExportQuality( i );
	}
	
	void
	on_hscale_export_stills_value_changed_event  (GtkWidget       *widget,
												gpointer         user_data) {
		setExportQuality( (int) GTK_ADJUSTMENT(widget)->value );
	}
                                        
	void
	on_togglebutton_export_stills_stop_toggled
	                                        (GtkToggleButton *togglebutton,
	                                        gpointer         user_data)
	{
		stopExport();
	}
	
	
	void
	on_togglebutton_export_stills_record_toggled
	                                        (GtkToggleButton *togglebutton,
	                                        gpointer         user_data)
	{
		startExport();
	}
	
	
	gboolean
	on_spinbutton_export_stills_range_start_focus_in_event
	                                        (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
		gtk_toggle_button_set_active( 
			GTK_TOGGLE_BUTTON( lookup_widget( widget, "radiobutton_export_stills_range" ) ),
			TRUE );
		return FALSE;
	}
	
	
	gboolean
	on_spinbutton_export_stills_range_end_focus_in_event
	                                        (GtkWidget       *widget,
	                                        GdkEventFocus   *event,
	                                        gpointer         user_data)
	{
		gtk_toggle_button_set_active( 
			GTK_TOGGLE_BUTTON( lookup_widget( widget, "radiobutton_export_stills_range" ) ),
			TRUE );
		return FALSE;
	}

}