/*
 * page_export_avi.cc Notebook Firewire/AVI/Still Frame Export Page Object
 * Copyright (C) 2001 Dan Dennedy <dan@dennedy.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <iostream>
using std::cout;
using std::endl;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>

#include "page_export_avi.h"
#include "preferences.h"
#include "filehandler.h"
#include "message.h"

extern "C" {
	#include "support.h"
	#include "callbacks.h"
	#include "interface.h"
	#include "commands.h"
}

/** Constructor for page.

    \param exportPage the export page object to bind to
  	\param common	common object to which this page belongs
*/

ExportAVI::ExportAVI( PageExport *exportPage, KinoCommon *common ) : writer(NULL)
{
	cout << "> Creating ExportAVI Page" << endl;
	this->exportPage = exportPage;
	this->common = common;
	recordButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "togglebutton_export_avi_record" ) );
	stopButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "togglebutton_export_avi_stop" ) );

	fileEntry = GTK_ENTRY( gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( 
		lookup_widget( common->getWidget(), "gnome_fileentry_export_avi" ) ) ) );
	formatDV1Toggle = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "radiobutton_export_avi_type1") );
	formatDV2Toggle = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "radiobutton_export_avi_type2") );
	autoSplitToggle = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "checkbutton_export_avi_autosplit" ) );
	timeStampToggle = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(),  "checkbutton_export_avi_timestamp" ) );
	framesEntry = GTK_ENTRY( lookup_widget( exportPage->getWidget(), "entry_export_avi_frames" ) );
	everyEntry = GTK_ENTRY( lookup_widget( exportPage->getWidget(), "entry_export_avi_every" ) );	

}

/** Destructor for page.
*/

ExportAVI::~ExportAVI() {
	cout << "> Destroying ExportAVI Page" << endl;
}

/** Start of page.
*/

void ExportAVI::start() {
	char s[20];
	cout << ">> Starting ExportAVI" << endl;
	
	if (common->getConfig().fileFormat == AVI_DV1_FORMAT) {
		gtk_toggle_button_set_active( formatDV1Toggle, TRUE );
	} else if(common->getConfig().fileFormat == AVI_DV2_FORMAT) {
		gtk_toggle_button_set_active( formatDV2Toggle, TRUE);
	}
	gtk_toggle_button_set_active( autoSplitToggle, common->getConfig().autoSplit);
	gtk_toggle_button_set_active( timeStampToggle, common->getConfig().timeStamp);
	gtk_entry_set_text( fileEntry, common->getConfig().file);
	sprintf(s, "%d", common->getConfig().frames);
	gtk_entry_set_text( framesEntry, s);
	sprintf(s, "%d", common->getConfig().every);
	gtk_entry_set_text( everyEntry, s);
}

/** Define active widgets.
*/

gulong ExportAVI::activate() {
	return 0;
}

/** Leaving the page
*/

void ExportAVI::clean() {
	cout << ">> Leaving ExportAVI" << endl;
	if (writer) {
		delete writer;
		writer = NULL;
	}
}


/** start exporting an AVI
*/
void ExportAVI::startExport() {
	gtk_toggle_button_set_active( recordButton, TRUE );
	gtk_toggle_button_set_active( stopButton, FALSE );
	exportPage->exportMutex = false;

	strcpy( common->getConfig().file, gtk_entry_get_text( fileEntry ) );
	
	if ( strcmp(common->getConfig().file, "") ) { 

		if (gtk_toggle_button_get_active( formatDV1Toggle ))
			common->getConfig().fileFormat = AVI_DV1_FORMAT;
		if (gtk_toggle_button_get_active( formatDV2Toggle ))
			common->getConfig().fileFormat = AVI_DV2_FORMAT;
		common->getConfig().autoSplit = gtk_toggle_button_get_active( autoSplitToggle );
		common->getConfig().timeStamp = gtk_toggle_button_get_active( timeStampToggle );
		common->getConfig().frames = atoi( gtk_entry_get_text( framesEntry ) );
		common->getConfig().every = atoi( gtk_entry_get_text( everyEntry ) );
		exportPage->resetProgress();
		
		writer = new AVIHandler( common->getConfig().fileFormat );
		writer->SetMaxFrameCount( common->getConfig().frames );
		writer->SetAutoSplit( common->getConfig().autoSplit );
		writer->SetTimeStamp( common->getConfig().timeStamp );
		writer->SetEveryNthFrame( common->getConfig().every );
		writer->SetBaseName( common->getConfig().file );
	
		int last = common->getPlayList()->GetNumFrames();
		static Frame frame;
	
	    try {
	
			common->getPlayList()->GetFrame( 0, frame );
			writer->SetSampleFrame(frame);
	
	        for (int i = 0; i < last && exportPage->isExporting; ++i) {
	       	
	            common->getPlayList()->GetFrame( i, frame );
	            writer->WriteFrame( frame );
	            exportPage->updateProgress( (gfloat) (i + 1) / (gfloat) last );
	        }
	        writer->Close();
	
	    } catch(string s) {
	        cout << "Could not save AVI file " << common->getConfig().file << ", because an exception has occurred: " << endl;
	        cout << s << endl;
	    }
	
		exportPage->isExporting = false;
		exportPage->exportMutex = true;
		gtk_toggle_button_set_active( recordButton, FALSE );
		gtk_toggle_button_set_active( stopButton, TRUE );
		exportPage->exportMutex = false;
		delete writer;
		writer = NULL;
		
	} else {
		modal_message( "You must enter a stem filename without an extension.");

		exportPage->isExporting = false;
		exportPage->exportMutex = true;
		gtk_toggle_button_set_active( recordButton, FALSE );
		gtk_toggle_button_set_active( stopButton, TRUE );
		exportPage->exportMutex = false;
	}
}


/** stop exporting an AVI
*/
void ExportAVI::stopExport() {
	/* do nothing, let the export page clear the export flag
	   to fall out of the loop.
	*/
}


extern "C" {
	void
	on_togglebutton_export_avi_record_toggled
	                                        (GtkToggleButton *togglebutton,
	                                        gpointer         user_data)
	{
		startExport();
	}
	
	
	void
	on_togglebutton_export_avi_stop_toggled
	                                        (GtkToggleButton *togglebutton,
	                                        gpointer         user_data)
	{
		stopExport();
	}
	
}
