/*
 * page_export_1394.cc Notebook Firewire/AVI/Still Frame Export Page Object
 * Copyright (C) 2001 Dan Dennedy <dan@dennedy.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <iostream>
using std::cout;
using std::endl;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>

#include "page_export_1394.h"
#include "preferences.h"
#include "ieee1394io.h"
#include "message.h"

extern "C" {
	#include "support.h"
	#include "callbacks.h"
	#include "interface.h"
	#include "commands.h"
}

/** Constructor for page.

    \param exportPage the export page object to bind to
  	\param common	common object to which this page belongs
*/

Export1394::Export1394(PageExport *exportPage, KinoCommon *common ) : exportReader(NULL), exportWriter(NULL)
{
	cout << "> Creating Export1394 Page" << endl;
	this->exportPage = exportPage;
	this->common = common;
	recordButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "togglebutton_export_1394_record" ) );
	stopButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "togglebutton_export_1394_stop" ) );
	previewButton = GTK_TOGGLE_BUTTON( lookup_widget( common->getWidget(), "togglebutton_export_1394_preview" ) );
    entry_cip_n = GTK_ENTRY( lookup_widget( common->getWidget(), "entry_export_1394_cip_n") );
    entry_cip_d = GTK_ENTRY( lookup_widget( common->getWidget(), "entry_export_1394_cip_d") );
    entry_syt_offset = GTK_ENTRY( lookup_widget( common->getWidget(), "entry_export_1394_syt_offset") );
}

/** Destructor for page.
*/

Export1394::~Export1394() {
	cout << "> Destroying Export1394 Page" << endl;
}

/** Start of page.
*/

void Export1394::start() {
	static raw1394handle_t  handle;
	gchar s[512];

	cout << ">> Starting Export1394" << endl;

	sprintf( s, "%d", common->getConfig().cip_n );
	gtk_entry_set_text( entry_cip_n, s );
	sprintf( s, "%d", common->getConfig().cip_d );
	gtk_entry_set_text( entry_cip_d, s );
	sprintf( s, "%d", common->getConfig().syt_offset );
	gtk_entry_set_text( entry_syt_offset, s );

#ifdef LIBRAW1394_OLD
		if ((handle = raw1394_get_handle())) {
#else
		if ((handle = raw1394_new_handle())) {
#endif
			if (raw1394_set_port(handle, 0) >= 0) {
    
				gtk_widget_set_sensitive( GTK_WIDGET( entry_cip_n ), TRUE);
				gtk_widget_set_sensitive( GTK_WIDGET( entry_cip_d ), TRUE);
				gtk_widget_set_sensitive( GTK_WIDGET( entry_syt_offset ), TRUE);

                raw1394_destroy_handle(handle);
                handle = NULL;
		        exportReader = new IEEE1394Reader;
				exportReader->Open();
            } else
            modal_message("Error setting the IEEE 1394 port (host adapater); all IEEE 1394 options are disabled.");
        } else 
            modal_message("The IEEE 1394 subsystem is not loaded; all IEEE 1394 options are disabled.");

}

/** Define active widgets.
*/

gulong Export1394::activate() {
	return 0;
}

/** Leaving the page
*/

void Export1394::clean() {
	cout << ">> Leaving Export1394" << endl;
	if (exportReader) {
		exportReader->Close();
		delete exportReader;
		exportReader = NULL;
	}
	if (exportWriter) {
		delete exportWriter;
		exportWriter = NULL;
	}
	common->getConfig().Save();
}

void Export1394::doExport() {
	int last = common->getPlayList()->GetNumFrames();
	static Frame frame;

	assert( exportWriter );
	if (exportWriter->viddev >= 0 ) {
		
		/* pre-roll */
		common->getPlayList()->GetFrame( 0, frame );
		for (int i = 0; i < last && i <= 75 && exportPage->isExporting; i++) {
			exportWriter->SendFrame( frame );
		}

		for (int i = 0; i < last && exportPage->isExporting; i++) {
	
			common->getPlayList()->GetFrame( i, frame );
			exportWriter->SendFrame( frame );
			exportPage->updateProgress( (gfloat) (i + 1) / (gfloat) last );
		}
	}

	exportPage->isExporting = false;
	exportReader->AVCStop( common->getConfig().phyID );
	exportPage->exportMutex = true;
	gtk_toggle_button_set_active( recordButton, FALSE );
	gtk_toggle_button_set_active( previewButton, FALSE );
	gtk_toggle_button_set_active( stopButton, TRUE );
	gtk_widget_set_sensitive( GTK_WIDGET(previewButton), TRUE );
	gtk_widget_set_sensitive( GTK_WIDGET(recordButton), TRUE );
	exportPage->exportMutex = false;
	delete exportWriter;
	exportWriter = NULL;
}


/** start the exporting DV over 1394
*/
void Export1394::startExport() {
	gtk_toggle_button_set_active( recordButton, TRUE );
	gtk_toggle_button_set_active( stopButton, FALSE );
	gtk_toggle_button_set_active( previewButton, TRUE );
	gtk_widget_set_sensitive( GTK_WIDGET(previewButton), FALSE );
	exportPage->exportMutex = false;

	common->getConfig().cip_n = atoi( gtk_entry_get_text( entry_cip_n ));
	common->getConfig().cip_d = atoi( gtk_entry_get_text( entry_cip_d ));
	common->getConfig().syt_offset = atoi( gtk_entry_get_text( entry_syt_offset ));
	exportPage->resetProgress();
	exportWriter = new IEEE1394Writer;
	
	exportReader->AVCRecord( common->getConfig().phyID );
	doExport();
}


/** stop the exporting DV over 1394
*/
void Export1394::stopExport() {
	/* do nothing, PageExport will clear the export flag and let the loop fallout */
}

/** preview export DV over 1394 without telling the camera to record
*/
void Export1394::previewExport() {
	if ( !exportPage->exportMutex  && !exportPage->isExporting ) {
		exportPage->exportMutex = true;
		exportPage->isExporting = true;
		gtk_toggle_button_set_active( stopButton, FALSE );
		gtk_toggle_button_set_active( recordButton, FALSE );
		gtk_widget_set_sensitive( GTK_WIDGET(recordButton), FALSE );
		exportPage->exportMutex = false;

		common->getConfig().cip_n = atoi( gtk_entry_get_text( entry_cip_n ));
		common->getConfig().cip_d = atoi( gtk_entry_get_text( entry_cip_d ));
		common->getConfig().syt_offset = atoi( gtk_entry_get_text( entry_syt_offset ));
		exportPage->resetProgress();
		exportWriter = new IEEE1394Writer;
	
		doExport();
	}
}	

extern "C" {
	extern KinoCommon *common;
	
	void previewExport( void ) {
		common->getPageExport()->getPageExport1394()->previewExport();
	}
	
	void
	on_togglebutton_export_1394_preview_toggled
	                                        (GtkToggleButton *togglebutton,
	                                        gpointer         user_data)
	{
		previewExport();
	}
	
	
	void
	on_togglebutton_export_1394_record_toggled
	                                        (GtkToggleButton *togglebutton,
	                                        gpointer         user_data)
	{
		startExport();
	}
	
	
	void
	on_togglebutton_export_1394_stop_toggled
	                                        (GtkToggleButton *togglebutton,
	                                        gpointer         user_data)
	{
		stopExport();
	}
	
}