// taim.cpp
//
// Kit AIM client
//
// For copyright and license, see accompanying documentation

#include "config.h"
#include <qdatetime.h>
#include <qmessagebox.h>
#include <qstring.h>
#include <qtimer.h>
#include "tbuddylist.h"
#include <klocale.h>
#include <qregexp.h>

/* For ScreenSaver stuff */

extern "C"
{

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/scrnsaver.h>

#include <time.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

}

#include "kitsocket.h"
#include "taim.h"

// * class TAim - public *
TAim::TAim(QObject *parent, const char *name)
	: QObject(parent, name)
{
	status = TAIM_OFFLINE;
	socket = 0;
	username = QString::null;
	r_password = QString::null;
	userInfo = QString::null;

	serverConfig.permitStatus = 1;
	serverConfig.buddyList.reset();
	serverConfig.permitList.reset();
	serverConfig.denyList.reset();

	currentConfig.permitStatus = 1;
	currentConfig.buddyList.reset();
	currentConfig.permitList.reset();
	currentConfig.denyList.reset();

	keepAlive = false;
	usingCustomServer = false;
  useX11forIdle = true;
  idleTimeBeforeSignal = 30;
	
	updateIdleTime = false;
	idleTime = 0;
	isIdle = false;

	useSocks = false;
	useSocksAuth = false;

	timerLock = false;
	if (timerLock != true){	
		timerLock = true;	  
		QTimer::singleShot( 60000, this, SLOT(updateIdleness()) );
	}
}

TAim::~TAim()
{
	emit endProgress();
	setStatus(TAIM_OFFLINE);
}
void TAim::setBuddyList(const TBuddyList &a){
  TBuddy buddy;
  TBuddyList newList = a;
  /*
  // reset all buddies
  for(int i = 0; i < newList.getCount(); i++){
    newList.setStatus(i, TAIM_OFFLINE);
    newList.setUserClass(i, 0);
    newList.setSignonTime(i, 0);
    newList.setIdleTime(i, 0);
    newList.setEvil(i, 0);
  }
  // get the current values
  for(int i = 0; i < currentConfig.buddyList.getCount(); i++){
    int j = newList.getNum( currentConfig.buddyList.getName(i) );
    if(j != -1){
      currentConfig.buddyList.get(&buddy, i);
      //newList.setStatus(j, buddy.status);
      newList.setUserClass(j, buddy.userClass);
      newList.setSignonTime(j, buddy.signonTime);
      newList.setIdleTime(j, buddy.idleTime);
      newList.setEvil(j, buddy.evil);
    }
  }
  */	
  currentConfig.buddyList = newList;
  sendConfig();
  emit configChanged();
  timerLock = false;

  // Lets keep track of these.
  setServerBuddyList(a);
}
void TAim::setPermitList(const TBuddyList &a)
{
	currentConfig.permitList = a;
	sendPermissions();
}
void TAim::setDenyList(const TBuddyList &a)
{
	currentConfig.denyList = a;
	sendPermissions();
}
void TAim::setPermissions(int a)
{
	currentConfig.permitStatus = ( 1 <= a && 4 >= a ) ? a : 1;
	sendPermissions();
}
void TAim::doKeepAlive(int doit)
{
	keepAlive = doit ? true : false;
	if(socket != 0) socket->setKeepAlive(keepAlive);
}
void TAim::doIdleTime(bool doit)
{
	updateIdleTime = doit;
}

void TAim::setIdleTime(int time){
  if( time >= 60*10 ){
    isIdle = true;
    QString data;
    data.sprintf("toc_set_idle %i", time);
    socket->writeData(data);
  }
}

void TAim::useCustomServer(int use)
{
	usingCustomServer = use ? true : false;
}
void TAim::setCustomServer(const QString &s, int sp, const QString &a, int ap)
{
	server = s;
	serverPort = sp;
	authorizer = a;
	authorizerPort = ap;
}
void TAim::useSocksServer(bool a)
{
	//useSocks = a;
}
void TAim::setSocksProxy(const QString &h, unsigned short p, bool a, const QString &name, const QString &pwd)
{
	//socksServer = h;
	//socksPort = p;
	//useSocksAuth = a;
	//socksName = name;
	//socksPassword = pwd;
}

void TAim::setServerBuddyList(const TBuddyList &a)
{
	serverConfig.buddyList = a;
	setConfig();
}
void TAim::setServerPermitList(const TBuddyList &a)
{
	serverConfig.permitList = a;
	setConfig();
}
void TAim::setServerDenyList(const TBuddyList &a)
{
	serverConfig.denyList = a;
	setConfig();
}
void TAim::setServerPermissions(int a)
{
	serverConfig.permitStatus = a;
	setConfig();
}

// * class TAim - public slots *
void TAim::setStatus(int newstatus)
{
	// this guards against a double delete
	if(newstatus == status)
    return;
	status = newstatus;
	if(socket)
	{
    delete socket;
		socket = 0;
	}
	if(status == TAIM_ONLINE) tocConnect();
	emit statusChanged(status);
}
void TAim::setUserName(const QString &a)
{
	if(a == username) return;
	if(status != TAIM_OFFLINE) setStatus(TAIM_OFFLINE);

	username = a;
	serverConfig.permitStatus = 1;
	serverConfig.buddyList.reset();
	serverConfig.permitList.reset();
	serverConfig.denyList.reset();
	
	emit userNameChanged();
}
void TAim::setPassword(const QString &a)
{
	r_password = tocRoast(a);
}
void TAim::setUserInfo(const QString &a)
{
	userInfo = a;
	sendUserInfo();
	endIdleness();
}
void TAim::requestUserInfo(const QString &name)
{
	if(!socket) return;
	QString data;
	data = QString::fromLatin1("toc_get_info %1").arg(tocNormalize(name));
	socket->writeData(data);
	endIdleness();
}

void TAim::sendIM(QString message, QString target, bool isAuto, bool endIdleNess)
{
  if(!socket) return;
	QString outMessage = message;
	if(isAuto){
    QString currentDate = QDate::currentDate().toString();
		QString currentTime = QTime::currentTime().toString();
    outMessage.replace( QRegExp("%d"), currentDate );
    outMessage.replace( QRegExp("%t"), currentTime );
    outMessage.replace( QRegExp("%n"), target );
	}

  QString data;
  data = QString::fromLatin1("toc_send_im %1 %2").arg(tocNormalize(target)).arg(tocProcess(outMessage));
  if(isAuto){
    QString currentDate = QDate::currentDate().toString();
		QString currentTime = QTime::currentTime().toString();
    data.replace( QRegExp("%d"), currentDate );
    data.replace( QRegExp("%t"), currentTime );
    data.replace( QRegExp("%n"), target );
		data += " auto";
	}
  //printf("string: %s\n", data.data());
  socket->writeData(data);
  if (isAuto == false && endIdleNess == true)
	  endIdleness();
}

void TAim::sendWarning(QString target, bool isAnonymous)
{
	if(!socket) return;
	QString data;
	data.sprintf("toc_evil %s %s", tocNormalize(target).latin1(), isAnonymous ? "anon" : "norm");
	socket->writeData(data);
	endIdleness();
}
void TAim::changePassword(QString oldPassword, QString newPassword)
{
	if(!socket) return;
	QString data;
	data.sprintf("toc_change_passwd %s %s", tocNormalize(oldPassword).latin1(),tocNormalize(newPassword).latin1() );
	socket->writeData(data);
	endIdleness();
}

//toc_chat_join <Exchange> <Chat Room Name>
void TAim::chatJoin(QString name, int Exchange){
  if(!socket) return;

  QString data;
  data.sprintf("toc_chat_join %i %s", Exchange, tocProcess(name).latin1());
  socket->writeData(data);
}

//toc_chat_send <Chat Room ID> <Message>
void TAim::chatSend(int room, QString message){
  if(!socket) return;
  
  QString data;
  data.sprintf("toc_chat_send %i %s", room, tocProcess(message).latin1());
  socket->writeData(data);
}

//toc_chat_whisper <Chat Room ID> <dst_user> <Message>
void TAim::chatWhisper(int room, QString dst_bud, QString message){
  if(!socket) return;
 
  QString data;
  data.sprintf("toc_chat_whisper %i %s %s", room, tocProcess(dst_bud).latin1(), tocProcess(message).latin1());
  printf("whisper data %s\n", data.latin1());
  socket->writeData(data);
}

//toc_chat_invite <Chat Room ID> <Invite Msg> <buddy1> [<buddy2> [<buddy3> [...]]]
void TAim::chatInvite(int room, QString message, QString buddy){
  if(!socket) return;

  // tocProcess each buddy.
  QString data;
  int i = -1;
  QString newbuddylist = "";
  while (buddy.length() > 0){
    i = buddy.find(",");
    if(i != -1 && i != 0){
      newbuddylist += tocProcess(buddy.left(i)) + " ";
      buddy.remove(0, i + 2);
    }
    else{
      newbuddylist += tocProcess(buddy);
      buddy.remove(0, buddy.length());
		}

  }
  data.sprintf("toc_chat_invite %i %s %s", room, tocProcess(message).latin1(), newbuddylist.latin1());
  socket->writeData(data);
}

//toc_chat_leave <Chat Room ID>
void TAim::chatLeave(int room){
  if(!socket) return;

  QString data;
  data.sprintf("toc_chat_leave %i", room);
  socket->writeData(data);
}

//toc_chat_accept <Chat Room ID>
void TAim::chatAccept(int room){
  if(!socket) return;
  
  QString data;
  data.sprintf("toc_chat_accept %i", room);
  socket->writeData(data);
}


void TAim::setAwayMessage(const QString &message)
{
	if(!socket) return;
 
	QString awaymessage = message;
	if (awaymessage.length() > 1000){
		awaymessage = awaymessage.mid(0, 950);
	  awaymessage = awaymessage + "... Im me for the rest of this away message.";
	}

	QString data;
	data.sprintf("toc_set_away %s", tocProcess(awaymessage).latin1());
	socket->writeData(data);
	if(message == QString::null || message.length() == 0 || message == "") endIdleness();
}

void TAim::requestDirectory(const QString &target)
{
	if(!socket) return;
	QString data = QString::fromLatin1("toc_get_dir %1").arg(tocNormalize(target));
	socket->writeData(data);
	endIdleness();
}

void TAim::setDirectory(const directory &dir)
{
	if(!socket) return;
	QString data = QString::fromLatin1("toc_set_dir %1:%2:%3:%4:%5:%6:%7:%8:");
	data.arg(tocProcess(dir.firstName));
	data.arg(tocProcess(dir.middleName));
	data.arg(tocProcess(dir.lastName));
	data.arg(tocProcess(dir.maidenName));
	data.arg(tocProcess(dir.city));
	data.arg(tocProcess(dir.state));
	data.arg(tocProcess(dir.country));
	data.arg(tocProcess(dir.email));
	if(dir.allowWebSearches) data += "T";
	socket->writeData(data);
	endIdleness();
}

void TAim::searchDirectory(const directory &dir)
{
	if(!socket) return;
	QString data = QString::fromLatin1("toc_dir_search %1:%2:%3:%4:%5:%6:%7:%8");
	data.arg(tocProcess(dir.firstName));
	data.arg(tocProcess(dir.middleName));
	data.arg(tocProcess(dir.lastName));
	data.arg(tocProcess(dir.maidenName));
	data.arg(tocProcess(dir.city));
	data.arg(tocProcess(dir.state));
	data.arg(tocProcess(dir.country));
	data.arg(tocProcess(dir.email));
	socket->writeData(data);
	endIdleness();
}

void TAim::cancelProgress(void)
{
	setStatus(TAIM_OFFLINE);
	emit endProgress();
	setStatus(TAIM_OFFLINE);
}

// * class TAim - protected slots *
// *******************************************************************************
#define tocConnectError(msg2) { setStatus(TAIM_OFFLINE);\
                                emit endProgress();\
                                emit displayError(QString(msg2));\
                                return; }

void TAim::tocConnect(void)
{
	// don't bother if already connected
	if(socket) return;

	serverConfig.buddyList.reset();
	serverConfig.permitList.reset();
	serverConfig.denyList.reset();
	oldEvil = 0;
	lastListSent = QString::null;
	lastPermissionSent = -1;
	lastInfoSent = QString::null;
	lastConfigSent = QString::null;

	// reset all buddies -- they'll be refreshed on the UPDATE_BUDDYs after configChanged
	int i = -1;
	while( ++i < currentConfig.buddyList.getCount() )
	{
		currentConfig.buddyList.setStatus(i, TAIM_OFFLINE);
		currentConfig.buddyList.setUserClass(i, 0);
		currentConfig.buddyList.setSignonTime(i, 0);
		currentConfig.buddyList.setIdleTime(i, 0);
		currentConfig.buddyList.setEvil(i, 0);
	}

	if( username.isNull() )
		tocConnectError(("No username specified."));

	if( r_password.isNull() )
		tocConnectError(("No password specified."));

	emit initProgress(7, 0, ("Initializing connection to server..."));

	// initialize connection
	QString usingServer = usingCustomServer ? server : QString(TOC_HOST);
	unsigned short usingPort = usingCustomServer ? serverPort : TOC_PORT;
	socket = new KitSocket(usingServer, usingPort);
	
        if( !socket->connect() )
		tocConnectError(("Failed to connect."));

	socket->setKeepAlive(keepAlive);
	QObject::connect( socket, SIGNAL(readData()), this, SLOT(tocConnect1()) );
	QObject::connect( socket, SIGNAL(disconnected()), this, SLOT(onDisconnect()) );

	connect(socket, SIGNAL(debugWriteData(QString)), this, SIGNAL(debugDataIn(QString)));

	emit updateProgress(1, "Waiting for reply...");
}
void TAim::tocConnect1(void)
{
	if(!socket) return;
	char data[8192];

	emit updateProgress(2, ("Signing on..."));
	
	QObject::disconnect( socket, SIGNAL(readData()), this, SLOT(tocConnect1()) );
	QObject::connect( socket, SIGNAL(readData()), this, SLOT(tocConnect2()) );

	int i = socket->read((char *)&data, 8192);
	
	if(i == -1)
		tocConnectError(("No data to read.  This shouldn't happen, since this function is called only by the emit readData."));
	
	if(i != SFLAP_SIGNON)
		tocConnectError(("Missed signon frame from server"));

	socket->writeSignon( 1, 1, tocNormalize(username) );

	emit updateProgress(3, ("Sending username and password..."));
	
	if(usingCustomServer)
	{
	  QString temp;
	  temp.sprintf("%i", authorizerPort);
	  tocSignon(authorizer, temp, TOC_LANG);
	}
	else
	  tocSignon(TOC_AUTH_HOST, TOC_AUTH_PORT, TOC_LANG);
	
	emit updateProgress(4, ("Waiting for authorization..."));
}
void TAim::tocConnect2(void)
{
	if(!socket) return;
	QObject::disconnect( socket, SIGNAL(readData()), this, SLOT(tocConnect2()) );
	QString data;

	socket->read(data);

	if(data.left(6) == "ERROR:")
	{
		doError(data);
		setStatus(TAIM_OFFLINE);
		emit endProgress();
		return;
	}
	else if(data.left(8) != "SIGN_ON:")
		tocConnectError(("Response to tocSignon is not SIGN_ON"));

	emit updateProgress(5, ("Checking for configuration..."));
	QObject::connect( socket, SIGNAL(readData()), this, SLOT(tocConnect3()) );
	QObject::connect( socket, SIGNAL(readData()), this, SLOT(tocServer()) );
}
void TAim::tocConnect3(void)
{
	if(!socket) return;
	QObject::disconnect( socket, SIGNAL(readData()), this, SLOT(tocConnect3()) );

	emit updateProgress(6, ("Sending configuration..."));

	sendPermissions();
	sendUserInfo();
	sendConfig();

	emit updateProgress(7, ("Completing signon..."));

	tocInitDone();
	emit endProgress();

	// we need this to update the idle time, as necessary
	idleTime = time(0);
	isIdle = false;
}

void TAim::onDisconnect(void)
	tocConnectError(("Disconnected."));
// ***************************************************************************

#define case_tocServer( x, y ) if( data.left( y ) == x ) {
#define end_case_tocServer return; }

void TAim::tocServer(void)
{
	if(!socket) return;
	QString data;
	int i;

	socket->read(data);
	emit(debugDataIn(data));
	case_tocServer("SIGN_ON", 7)
		sendConfig();
		// tell TSocket to resume writing
		socket->setPaused(false);
		tocInitDone();
	end_case_tocServer
	case_tocServer("CONFIG", 6)
		tocParseConfig(data, &(serverConfig.buddyList), &(serverConfig.permitList), &(serverConfig.denyList), &(serverConfig.permitStatus));

  TBuddyList slist = serverBuddyList();
  TBuddyList list = buddyList();
  list += slist;
  setBuddyList(list);
  emit reloadList();

	end_case_tocServer
	case_tocServer("NICK", 4)
	  doNick(data);	
	end_case_tocServer
	case_tocServer("IM_IN", 5)
		doIM(data);
	end_case_tocServer
	case_tocServer("UPDATE_BUDDY", 12)
		doUpdate(data);
	end_case_tocServer
	case_tocServer("ERROR", 5)
		doError(data);
  end_case_tocServer
	case_tocServer("ADMIN_PASSWD_STATUS",19)
	  doPasswordMessage(data);
	end_case_tocServer
	case_tocServer("EVILED", 6)
		doEviled(data);
	end_case_tocServer
	case_tocServer("CHAT_JOIN", 9)
		doChatJoin(data);	
	end_case_tocServer
	case_tocServer("CHAT_INVITE", 11)
		doChatInvite(data);
	end_case_tocServer
        case_tocServer("CHAT_IN", 7)
		doChatIn(data);	
	end_case_tocServer
	case_tocServer("CHAT_UPDATE_BUDDY", 17)
		doChatUpdateBuddy(data);
	end_case_tocServer
	case_tocServer("CHAT_LEFT", 9)
		doChatLeft(data);
	end_case_tocServer
	case_tocServer("GOTO_URL", 8)
		// strip the GOTO_URL
		i = data.find(':');
		if(i > 0) data.remove(0, i + 1);
		// strip the suggested window title
		i = data.find(':');
		if(i > 0) data.remove(0, i + 1);
		// TODO: hack hack hack
		if(usingCustomServer)
			data.sprintf("http://%s:%i/%s", server.latin1(), serverPort, data.latin1());
		else
			data.sprintf("http://" TOC_HOST ":%i/%s", TOC_PORT, data.latin1());
		// send the url
		emit userInfoReceived(data);
	end_case_tocServer
	case_tocServer("PAUSE", 5)
		socket->setPaused(true);
	end_case_tocServer
  case_tocServer("RVOUS_PROPOSE",13)
	  
	end_case_tocServer
		
}
void TAim::endIdleness(void)
{
  if(!socket) return;
  idleTime = time(0);
  if(isIdle)
  {
    isIdle = false;
    socket->writeData( "toc_set_idle 0" );
  }
}

void TAim::updateIdleness(void){
  timerLock = false; 
  if(!socket){
    if (timerLock != true){	
      timerLock = true;	  
      QTimer::singleShot( 60000, this, SLOT(updateIdleness()) );
    }
    return;
  }
  
  // set idle after 10 minutes
  int difference = (int)difftime(time(0), idleTime);
  if ( useX11forIdle == true ) { // If using X-Windows
    static XScreenSaverInfo *mit_info = NULL;
    static Display *d = NULL;
    if (!d)
      d = XOpenDisplay((char *)NULL);
    if (d != NULL){
      if (mit_info == NULL)
        mit_info = XScreenSaverAllocInfo();
      XScreenSaverQueryInfo(d, DefaultRootWindow(d), mit_info);
		  difference = (mit_info->idle) / 1000;
			//XCloseDisplay(d);
		}
		// If we were idle and someone played with keyboard/mouse,
    if( isIdle && (difference < 600)){
      // then end idleness
      endIdleness();
		}
  }
  if( !isIdle && (difference >= 60*10)){
    isIdle = true;
    QString data;
    data.sprintf("toc_set_idle %i", difference);
    socket->writeData(data);
  }
	
  if (idleTimeBeforeSignal == difference/60){
    emit (idleTimeReached());
  }

  const int numberOfBuddies = currentConfig.buddyList.getCount();
  for (int currentNum=0; currentNum < numberOfBuddies; currentNum++) {
    TBuddy *tempBuddy = NULL; 
    tempBuddy = currentConfig.buddyList.getByNum(currentNum);
    if (tempBuddy != NULL){
      if (tempBuddy->idleTime > 0){
        tempBuddy->idleTime = (tempBuddy->idleTime)+1;
       
        // Update the idle time string 
        int minute = tempBuddy->idleTime;
    	int hour=0;
    	while (minute > 59){
      	  hour++;
      	  minute = minute-60;
    	}

    	QString mid = "h ";
    	QString post = "m";
    	if (idleStringWColon == true){
      	  mid = ":";
      	  post = "";
        }

    	if (minute < 10)
      	  mid += "0";
    	tempBuddy->idleTimeString =  QString("%1%3%2%4").arg(hour).arg(minute).arg(mid).arg(post);
    
    	if (hour <= 0 && minute <= 0)
      	  tempBuddy->idleTimeString = "";
        if (tempBuddy->status ==  TAIM_ONLINE){
          emit buddyChanged(currentNum);
        }
      }
    }
  } 
	
  // check again in a minute
  if (timerLock != true){	
    timerLock = true;	  
    QTimer::singleShot( 60000, this, SLOT(updateIdleness()) );
  }
}

// * TAim -- private *
void TAim::tocSignon(const QString &auth_host, const QString &auth_port, const QString &lang)
{
  QString data;
  //data.sprintf("toc_signon %s %s  %s %s %s \"TIK:\\$Revision: 1.62 $\"", auth_host.latin1(), auth_port.latin1(), (tocNormalize(username)).latin1(), r_password.latin1(), TOC_LANG); 
  
  data.sprintf("toc_signon %s %s  %s %s %s \"penguin\"", auth_host.latin1(), auth_port.latin1(), (tocNormalize(username)).latin1(), r_password.latin1(), TOC_LANG); 
  

  // frame is set up, now send it
  socket->writeData(data);
}
void TAim::tocInitDone(void){
  QString data = "toc_init_done";
  socket->writeData(data);
}
void TAim::setConfig(void){
  if(!socket) return;
  QString conf, data;
  conf = tocWriteConfig( &(serverConfig.buddyList), &(serverConfig.permitList), &(serverConfig.denyList), serverConfig.permitStatus);
  data.sprintf("toc_set_config \"%s\"", conf.latin1());
  if(data != lastConfigSent )
  {
    socket->writeData(data);
    lastConfigSent = data;
  }
}
void TAim::sendConfig(void){
  if(!socket) return;
  QString data;
  // send buddyList
  data = "toc_add_buddy";
  for(int i = 0; i < currentConfig.buddyList.getCount(); i++){
    data += " " + tocNormalize(currentConfig.buddyList.getName(i));
  }
  if(data != lastListSent && data != QString("toc_add_buddy")){
    socket->writeData(data);
    if (lastListSent == QString::null)
			sendPermissions();
		lastListSent = data;
  }
}

void TAim::tocremoveBuddy(const TBuddyList &a, const QString &buddyName) {
  TBuddyList newList = a;
  currentConfig.buddyList = newList;
  if(!socket) return;
  QString data;
  // remove buddy
  data = "toc_remove_buddy " + tocNormalize(buddyName);
  socket->writeData(data);
}

void TAim::tocaddBuddy(const TBuddyList &a, const QString &buddyName) {
  TBuddyList newList = a;
  currentConfig.buddyList = newList;
  if(!socket) return;
  QString data;
  // remove buddy
  data = "toc_add_buddy " + tocNormalize(buddyName);
  socket->writeData(data);
}

void TAim::sendPermissions(void){
  if(!socket) return;
  QString data;
  // set permit mode
  switch(lastPermissionSent)
  {
    case TOC_PERMITSOME:
    case TOC_PERMITALL:
      socket->writeData("toc_add_permit");
      break;
		case TOC_DENYALL:
		case TOC_DENYSOME:
			socket->writeData("toc_add_deny");
			break;
	}
	switch(currentConfig.permitStatus)
	{
		case TOC_PERMITSOME:
			data = "toc_add_permit";
			for(int i = 0; i < currentConfig.permitList.getCount(); i++)
			{
				data += " " + tocNormalize( currentConfig.permitList.getName(i) );
			}
			socket->writeData(data);
			break;
		case TOC_DENYALL:
			socket->writeData("toc_add_permit");
			break;
		case TOC_DENYSOME:
			data = "toc_add_deny";
			for(int i = 0; i < currentConfig.denyList.getCount(); i++)
			{
				data += " " + tocNormalize( currentConfig.denyList.getName(i) );
			}
			socket->writeData(data);
		case TOC_PERMITALL:
			socket->writeData("toc_add_deny");
			break;
	}
	lastPermissionSent = currentConfig.permitStatus;
}
void TAim::sendUserInfo(void)
{
	if(!socket) return;
	QString data;
	data.sprintf("toc_set_info %s", tocProcess(userInfo).latin1());
	if(data != lastInfoSent)
	{
		socket->writeData(data);
		lastInfoSent = data;
	}
}
void TAim::doUpdate(const QString &data)
{
  TBuddy buddy;
  QString workstr;
  QString holding;
  int i;

  workstr = data;

  // skip the "UPDATE_BUDDY:"
  workstr.remove(0, 13);

  // name
  i = workstr.find(':');
  buddy.name = workstr.left(i);
  // skip past the colon
  workstr.remove(0, i + 1);

  // status:  T or F
  bool newStatus = (workstr[0] == 'T');
  if( newStatus != TAIM_ONLINE && buddy.status == TAIM_ONLINE ){
    emit (userSignedOff(buddy.name));
  }
  buddy.status = newStatus;
	
  // skip the status and the colon
  workstr.remove(0, 2);

  // evil amount : 0 to 100
  i = workstr.find(':');
	holding = workstr.left(i);
	buddy.evil = holding.toInt();
	workstr.remove(0, i + 1);
	
	// signontime:  seconds since unix epoch
	i = workstr.find(':');
	holding = workstr.left(i);
	buddy.signonTime = holding.toInt();
	int lastOn = holding.toInt();
	if ((buddy.status == TAIM_ONLINE)  && (lastOn > 918025725)){
	  buddy.lastOn = lastOn;
	}  
	else
	  buddy.lastOn = 0;  
	
	// The signed off, get current time.
	if (buddy.status == 0){
		QDate theDate(1970,1,1);
		QDateTime dt(theDate);
		int sec = dt.secsTo(QDateTime::currentDateTime());
		buddy.lastOn = sec + (5*60*60);
	}
	workstr.remove(0, i + 1);

  QTime theTime(0,0,0,0);
  QString lastOnString = "";
  if (buddy.status != 0){
     lastOnString = "Now";
  }
  else  
  {
    QDate date(1970,1,1);
    QDateTime dateTime;
    dateTime.setTime_t( buddy.lastOn );
    theTime = dateTime.time();
    //printf("date:%s\n", ((dateTime.date()).toString()).latin1());
    date = dateTime.date();
    
    if (theTime.minute() < 10)
      lastOnString = QString("%1 %2:0%3").arg(date.dayName(date.dayOfWeek())).arg(theTime.hour()).arg(theTime.minute());
    else	
      lastOnString = QString("%1 %2:%3").arg(date.dayName(date.dayOfWeek())).arg(theTime.hour()).arg(theTime.minute());
    if (buddy.lastOn == 0){
      lastOnString = "Unknown";
    }	
  }
  buddy.lastOnString = lastOnString;
			
  // idle time: minutes
  i = workstr.find(':');
  holding = workstr.left(i);
  buddy.idleTime = holding.toInt();
  workstr.remove(0, i + 1);
  // If the buddy is not online then he has no idle time.
  if (buddy.status == 0){
    buddy.idleTime = 0;
  }
	
  QString idleText = "";
  int minute = buddy.idleTime;
  int hour=0;
  while (minute > 59){
    hour++;
    minute = minute-60;
  }

  QString mid = "";
  QString post = "";
  if (idleStringWColon == true){
    mid = ":";
  }
  else{
    mid = "h ";
    post = "m";
  }

  if (minute < 10){
    mid += "0";
  }
  idleText =  QString("%1%3%2%4").arg(hour).arg(minute).arg(mid).arg(post);
  if (hour == 0 && minute == 0)
    idleText = "";
  buddy.idleTimeString = idleText;

	
	// user class "A " == AOL  " A" == oscar admin  " U" == unconfirmed  " O" == normal
	if(workstr[0] == 'A')
		buddy.userClass = TOC_USER_AOL;
	else
	{
		switch((char)(QChar)workstr[1])
		{
			case 'A':
				buddy.userClass = TOC_USER_ADMIN;
				break;
			case 'U':
				buddy.userClass = TOC_USER_UNCONFIRMED;
				break;
			case 'O':
			default:
				buddy.userClass = TOC_USER_NORMAL;
    }
  }
  switch((char)(QChar)workstr[2]){
	  case 'U':
		  buddy.userClass = TOC_USER_AWAY;
		  break;
  }

  // store the information
  int currentNum = currentConfig.buddyList.getNum(buddy.name);
  if(currentNum != -1)
  {
    TBuddy *bud = new TBuddy;
    currentConfig.buddyList.get(bud, currentNum);
    
    if ( currentConfig.buddyList.getStatus(currentNum) == 0 && buddy.status == 1){
	emit userSignedOn(buddy.name);
	emit userSignedOn(buddy.name, buddy.alias);
    }
    if (bud->idleTime > 0 && buddy.idleTime == 0 && buddy.status == 1)
      emit userNotIdle(buddy.name);	    
    if (bud->userClass == TOC_USER_AWAY && buddy.userClass != TOC_USER_AWAY && buddy.status == 1)
      emit userNotAway(buddy.name);	
    delete bud;

    currentConfig.buddyList.setStatus(currentNum, buddy.status);
    currentConfig.buddyList.setUserClass(currentNum, buddy.userClass);
    currentConfig.buddyList.setSignonTime(currentNum, buddy.signonTime);
    currentConfig.buddyList.setIdleTime(currentNum, buddy.idleTime);
		currentConfig.buddyList.setIdleTimeString(currentNum, buddy.idleTimeString);
    currentConfig.buddyList.setEvil(currentNum, buddy.evil);
    if (buddy.lastOn > 0){
      currentConfig.buddyList.setLastOn(currentNum, buddy.lastOn);
		}
	  currentConfig.buddyList.setLastOnString(currentNum, buddy.lastOnString);
    
		// Just to make sure that we have the right name.
    TBuddy *tempbud =  currentConfig.buddyList.getByNum(currentNum);
    if (tempbud->name == tempbud->alias)
			tempbud->alias = buddy.name;
    tempbud->name = buddy.name;

    emit buddyChanged(currentNum);
  }
}

void TAim::doNick(const QString &data){
	QString workstr = data;
	// skip NICK:
	workstr.remove(0, 5);
	emit nick(workstr);
}

void TAim::doIM(const QString &data)
{
  //printf("string: %s\n", data.latin1());
  QString workstr;
	QString senderName;
	bool isAuto;
	int i;
	
	workstr = data;
	// skip IM_IN:
	workstr.remove(0, 6);

	// get username
	i = workstr.find(':');
	// if no colons, then there's something wrong
	if(i == -1)
		return;
	else
	{
		senderName = workstr.left(i);
		workstr.remove(0, i + 1);
	}
	// get whether it's auto
	isAuto = ( (char)(QChar)workstr[0] == 'T' ) ? true : false;
	workstr.remove(0, 2);
	// the rest is the message
	emit IMReceived(workstr, senderName, isAuto);
}
void TAim::doEviled(const QString &data)
{
	QString workstr;
	int newEvil;
	int i;

	workstr = data;
	// skip EVILED:
	workstr.remove(0, 7);

	i = workstr.find(':');
	if(i == -1) return;
	
	newEvil = workstr.left(i).toInt();
	// remove new evil
	workstr.remove(0, i + 1);

	// check if it's anonymous, and if it's a warning or a reduction
	if(newEvil > oldEvil)
		emit warningReceived(newEvil, (workstr.isEmpty()) ? QString::null : workstr);
	else
		emit warningReduced(newEvil, (workstr.isEmpty()) ? QString::null : workstr);
}

//CHAT_JOIN:<Chat Room Id>:<Chat Room Name>
void TAim::doChatJoin(const QString &data){
  QString work = data;
  work.remove(0,10);
  
  int i;
  int room = 0;
  QString name;
	
  i = work.find(':');
  if(i != -1){
    room = (work.left(i)).toInt();
    name = work.right(work.length()-i-1);
  }
  emit (chat_joined(room,name));
}

//CHAT_IN:<Chat Room Id>:<Source User>:<Whisper? T/F>:<Message>
void TAim::doChatIn(const QString &data){
  QString work = data;
  work.remove(0,8);
  
  int i;
  int room = 0;
  QString buddy;
  bool wisper;
  QString message;
	
  i = work.find(':');
  if(i != -1){
    room = (work.left(i)).toInt();
    work = work.right(work.length()-i-1);
  }

  i = work.find(':');
  if(i != -1){
    buddy = work.left(i);
    work = work.right(work.length()-i-1);
  }
  wisper = ( (char)(QChar)work[0] == 'T' ) ? true : false;
  message = work.mid(2, work.length()-2);
  emit (chat_in(room, buddy, wisper, message));
}

//CHAT_UPDATE_BUDDY:<Chat Room Id>:<Inside? T/F>:<User 1>:<User 2>...
void TAim::doChatUpdateBuddy(const QString &data){
  QString work = data;
  work.remove(0,18);
  
  int i;
  int room = 0;
  bool inside;
  QString buddy;
	
  i = work.find(':');
  if(i != -1){
    room = (work.left(i)).toInt();
    work = work.right(work.length()-i-1);
  }
  inside = ( (char)(QChar)work[0] == 'T' ) ? true : false;
  work = work.mid(2, work.length()-2);
  while (i != -1){
    i = work.find(':');
    if(i != -1){
      buddy = work.left(i);
      emit (chat_updateBuddy(room, inside, buddy));
      work = work.right(work.length()-i-1);
    }
    else{
      if (work.length() > 0){
        emit (chat_updateBuddy(room, inside, work));
      }
    }
  }
}

//CHAT_INVITE:<Chat Room Name>:<Chat Room Id>:<Invite Sender>:<Message>
void TAim::doChatInvite(const QString &data){
  QString work = data;
  work.remove(0,12);
  
  int i;

  QString name = "Kinkatta Error";
  int room = 0;
  QString buddy;
  QString message;
	
  i = work.find(':');
  if(i != -1){
    name = work.left(i);
    work = work.right(work.length()-i-1);
  }
  i = work.find(':');
  if(i != -1){
    room = (work.left(i)).toInt();
    work = work.right(work.length()-i-1);
  }
  i = work.find(':');
  if(i != -1){
    buddy = work.left(i);
    message = work.right(work.length()-i-1);
  }
  emit (chat_invite(name,room,buddy,message));
}

//CHAT_LEFT:<Chat Room Id>
void TAim::doChatLeft(const QString &data){
  QString work = data;
  work.remove(0,10);
  int roomLeft = work.toInt();
  emit (chat_left(roomLeft));
}

void TAim::doPasswordMessage(const QString &data){
  QString work = data;
  work.remove(0,20);
  if (work == "0"){
    QMessageBox::information(0, "Kinkatta - Message",("Password updated successfully."), QMessageBox::Ok);
  }
  else{
    QString error = "Error in changing the password.";	
    emit (displayError(error));
  }
}

void TAim::doError(const QString &data)
{
	int errorno = 0;
	int i;
	QString workstr;
	QString message;
	QString display;

	workstr = data;

	// skip ERROR:
	workstr.remove(0, 6);

	// extract error number
	// QString message used as a temporary here
	// after this, workstr holds the parameter
	i = workstr.find(':');
	if(i != -1)
	{
		message = workstr.left(i);
		errorno = message.toInt();
		workstr.remove(0, i + 1);
	}
	else
	{
		errorno = workstr.toInt();
		workstr.truncate(0);
	}
#define doError_errorCase( x, y ) case x: message = y; break;

	switch(errorno)
	{
		// general errors
		doError_errorCase(901, i18n("%1 is not currently available."));
		doError_errorCase(902, i18n("Warning of %1 not currently available."));
		doError_errorCase(903, i18n("A message has been dropped; you are exceeding the server speed limit."));
		// admin errors
		doError_errorCase(911, i18n("Error validating input."));
		doError_errorCase(912, i18n("Invalid account"));
		doError_errorCase(913, i18n("Error encountered while processing request"));
		doError_errorCase(914, i18n("Service unavailable"));
   
		// chat errors
		doError_errorCase(950, i18n("Chat in %1 is unavailable."));
		// Message and information errors
		doError_errorCase(960, i18n("You are sending messages too fast to %1."));
		doError_errorCase(961, i18n("You missed a message from %1 because\nthe message was too big."));
		doError_errorCase(962, i18n("You missed a message from %1 because\nthe message was sent too fast."));
		// Directory errors
		doError_errorCase(970, i18n("Search failed."));
		doError_errorCase(971, i18n("Search returned too many matches."));
		doError_errorCase(972, i18n("Search needs more qualifiers."));
		doError_errorCase(973, i18n("Directory service temporarily unavailable."));
		doError_errorCase(974, i18n("Email lookup is restricted."));
		doError_errorCase(975, i18n("Keyword ignored."));
		doError_errorCase(976, i18n("No Keywords."));
		doError_errorCase(977, i18n("Language not supported."));
		doError_errorCase(978, i18n("Country not supported."));
		doError_errorCase(979, i18n("Unknown directory failure: %1"));
		// Authorization errors
		doError_errorCase(980, i18n("Incorrect screen name or password.  Login failed."));
		doError_errorCase(981, i18n("The service is temporarily unavailable."));
		doError_errorCase(982, i18n("Your warning level is currently too high to sign on."));
		doError_errorCase(983, i18n("You have been connecting and disconnecting too frequently.\nWait 10 minutes and try again.  If you continue to try,\nyou will need to wait even longer."));
		doError_errorCase(989, i18n("Unknown signon failure: %1"));
	}
#undef doError_errorCase

	// if parameter exists, use it
	if(workstr.length() > 0)
		display = message.arg(workstr);
	else
		display = message;

	// stop = 8
	emit displayError(display);
}

#include "taim.moc"


