

#  Copyright Mission Critical Linux, 2000

#  Kimberlite is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the
#  Free Software Foundation; either version 2, or (at your option) any
#  later version.

#  Kimberlite is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with Kimberlite; see the file COPYING.  If not, write to the
#  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
#  MA 02139, USA.
#
# $Revision: 1.6 $
#
# Author: Gregory P. Myrdal <Myrdal@MissionCriticalLinux.Com>
#
#------------------------------------------------------------

#
# KornShell library for device functions.
#
# Requires globals.ksh to be sourced in.
#

#
#
#
startDevice ()
{
	if [ $# -ne 1 ]; then
	  logAndPrint $LOG_ERR "Usage: startDevice deviceID"
	fi

	typeset id=$1
	typeset mp dev fstype opts dev_owner dev_group dev_mode 

	#
	# Get the device
	#
	dev=$(getSvcDevice $DB $SVC_ID $id)
	case $? in
	  0) dev="-d $dev";;		# found it
	  *) logAndPrint $LOG_ERR "\
startDevice: Cannot find device $id for service $SVC_NAME in $DB, err=$?"
	    return $FAIL ;;
	esac

	#
	# Get the mount point, if it exists.
	#
	mp=$(getSvcMountPoint $DB $SVC_ID $id)
	case $? in
	  0) 
	     case "$mp" in 
	     ""|"[ 	]*")
	       mp="";;			# getconfig returns white space
	     *)
	       mp="-m $mp";;		# found it
	     esac
	     ;;
	  2) mp="";;		# raw device, erase "not found" from getconfig
	  *) logAndPrint $LOG_ERR "\
startDevice: Cannot get mount point $id for service $SVC_NAME, err=$?"
	     return $FAIL ;;
	esac

	#
	# Get the filesystem type, if specified.
	#
	fstype=$(getSvcMountFstype $DB $SVC_ID $id)
	case $? in 
	  0) 
	     case "$fstype" in 
	     ""|"[ 	]*")
	       fstype="";;		# getconfig returns white space
	     *)
	       fstype="-t $fstype";;	# found it
	     esac
	     ;;
	  2) fstype="";;	   # getconfig returns "not found", erase it 
	  *) logAndPrint $LOG_ERR "\
startDevice: Cannot get mount fstype $id for service $SVC_NAME, err=$?"
	     return $FAIL ;;
        esac

	#
	# Get the mount options, if they exist.
	#
	opts=$(getSvcMountOptions $DB $SVC_ID $id)
	case $? in
	  0) 
	     case "$opts" in 
	     ""|"[ 	]*")
	       opts="";;		# getconfig returns white space
	     *)
	       opts="-o $opts";;	# found it
	     esac
	     ;;
	  2) opts="";;		# getconfig returns "not found", erase it 
	  *) logAndPrint $LOG_ERR "\
startDevice: Cannot get mount options $id for service $SVC_NAME, err=$?"
	     return $FAIL ;;
	esac

	#
	# Get the device owner, if it exists.
	#
	dev_owner=$(getSvcDeviceOwner $DB $SVC_ID $id)
	case $? in
	  0) 
	     case "$dev_owner" in 
	     ""|"[ 	]*")
	       dev_owner="";;			# getconfig returns white space
	     *)
	       dev_owner="-u $dev_owner";;	# found it
	     esac
	     ;;
	  2) dev_owner="";;	# getconfig returns "not found", erase it 
	  *) logAndPrint $LOG_ERR "\
startDevice: Cannot get device owner $id for service $SVC_NAME, err=$?"
	     return $FAIL ;;
	esac

	#
	# Get the device group, if it exists.
	#
	dev_group=$(getSvcDeviceGroup $DB $SVC_ID $id)
	case $? in
	  0) 
	     case "$dev_group" in 
	     ""|"[ 	]*")
	       dev_group="";;			# getconfig returns white space
	     *)
	       dev_group="-g $dev_group";;	# found it
	     esac
	     ;;
	  2) dev_group="";;	# getconfig returns "not found", erase it 
	  *) logAndPrint $LOG_ERR "\
startDevice: Cannot get device group $id for service $SVC_NAME, err=$?"
	     return $FAIL ;;
	esac

	#
	# Get the device mode, if it exists.
	#
	dev_mode=$(getSvcDeviceMode $DB $SVC_ID $id)
	case $? in
	  0) 
	     case "$dev_mode" in 
	     ""|"[ 	]*")
	       dev_mode="";;			# getconfig returns white space
	     *)
	       dev_mode="-p $dev_mode";;	# found it
	     esac
	     ;;
	  2) dev_mode="";;	# getconfig returns "not found", erase it 
	  *) logAndPrint $LOG_ERR "\
startDevice: Cannot get device mode $id for service $SVC_NAME, err=$?"
	     return $FAIL ;;
	esac

	#
	# Try the mount.  
	#
	# Note: Even if we do not have a mount point we call mount in
	# the case that we have any permissions to set on a device.
	#
	logAndPrint $LOG_DEBUG "Running $MOUNT_SCRIPT start $dev $mp $fstype $opts $dev_owner $dev_group $dev_mode"
	$MOUNT_SCRIPT start $dev $mp $fstype $opts $dev_owner $dev_group $dev_mode
	if [ $? -ne 0 ]; then
	  return $FAIL
	fi

	return $SUCCESS
}

#
#
#
startDevices()
{
	typeset TMPFILE=/tmp/$MYNAME.startDevices.$$
	typeset -i n=$MIN_TOKEN_ID
	typeset -i ret_val
	typeset mp mount_point
	typeset id
	typeset dev

	#
	# Get all mounts points in the correct heirarchical order.
	# Save the mount point ID so we can grab other associated
	# attributes for that mount later.
	#
	while : ; do

	  #
	  # We loop on devices looking for devices that have mount points
	  #
	  dev=$(getSvcDevice $DB $SVC_ID $n)
	  case $? in
	    0) : ;;		# found it, let it fall through
	    2) break ;;		# done getting devices, no more left
	    *) logAndPrint $LOG_ERR "\
startDevices: Cannot get device $n of service $SVC_NAME, err=$?"
	       return $FAIL;;
	  esac

	  mp=$(getSvcMountPoint $DB $SVC_ID $n)
	  case $? in
	    0) : ;;		# found a mount point
	    2) ((n=n+1))	# no mount point, go to next device
	       continue ;;
	    *) logAndPrint $LOG_ERR "\
startDevices: Cannot get mount point $n of service $SVC_NAME, err=$?"
	       return $FAIL ;;
	  esac

	  print "$mp $n" >> $TMPFILE
	  ((n=n+1))
	done

	if [ -s $TMPFILE ]; then
	  #
	  # Get all of the attributes for this mount and attempt to mount it.
	  #
	  sort $TMPFILE |&
	  while read -p mount_point id
	  do
	    startDevice $id
	    if [ $? -ne $SUCCESS ]; then
	      rm -f $TMPFILE
	      return $FAIL
	    fi
	  done
	fi

	((n=MIN_TOKEN_ID))
	cat /dev/null > $TMPFILE

	#
	# Start all of the devices that are not mounts
	#
	while : ; do
	  dev=$(getSvcDevice $DB $SVC_ID $n)
	  if [ $? -ne 0 ]; then
	    break		# done getting devices
	  fi

	  mp=$(getSvcMountPoint $DB $SVC_ID $n)
	  if [ $? -eq 0 ]; then
	    ((n=n+1))
	    continue		# already started
	  fi

	  print "$dev $n" >> $TMPFILE
	  ((n=n+1))
	done

	if [ -s $TMPFILE ]; then
	  cat $TMPFILE |&
	  while read -p dev id
	  do
	    startDevice $id
	    if [ $? -ne $SUCCESS ]; then
	      rm -f $TMPFILE
	      return $FAIL
	    fi
	  done
	fi
      
	rm -f $TMPFILE
	return $SUCCESS
}

#
#
#
stopDevice ()
{
	if [ $# -ne 1 ]; then
	  logAndPrint $LOG_ERR "Usage: stopDevice deviceID"
	fi

	typeset id=$1
	typeset mp dev opts dev_owner dev_group dev_mode

	#
	# Get the device
	#
	dev=$(getSvcDevice $DB $SVC_ID $id)
	case $? in
	  0) dev="-d $dev";;		# found it
	  *) logAndPrint $LOG_ERR "\
stopDevice: Cannot find device for $mp in $DB, err=$?"
	     return $FAIL ;;
	esac

	#
	# Get the mount point, if it exists.
	#
	mp=$(getSvcMountPoint $DB $SVC_ID $id)
	case $? in
	  0) mp="-m $mp";;		# found it
	  2) return $SUCCESS;;		# nothing to unmount
	  *) logAndPrint $LOG_ERR "\
stopDevice: Cannot get mount point $id for $SVC_NAME, err=$?"
	     return $FAIL ;;
	esac

	#
	# Get the force unmount setting if there is a mount point.
	#
	if [ -n "$mp" ]; then
	  force_umount=$(getSvcForceUnmount $DB $SVC_ID $id)
	  case $? in
	    0) 				# found it
	      case $force_umount in
	        $YES_STR)
	          force_umount="-f" ;;
	        *)
	          force_umount="" ;;
	      esac
	      ;;
	    2) force_umount="" ;;	# no force umount set
	    *) logAndPrint $LOG_ERR "\
stopDevice: Cannot get force umount setting $id for $SVC_NAME, err=$?"
	       return $FAIL ;;
	  esac
	fi

	#
	# Try the umount (the mount script does retries if needed).
	#
	logAndPrint $LOG_DEBUG "Running $MOUNT_SCRIPT stop $dev $mp $force_umount"
	$MOUNT_SCRIPT stop $dev $mp $force_umount
	if [ $? -ne 0 ]; then
	  return $FAIL
	fi

	return $SUCCESS
}

#
#
#
stopDevices()
{
	typeset TMPFILE=/tmp/$MYNAME.stopDevices.$$
	typeset -i n=MIN_TOKEN_ID
	typeset -i ret_val
	typeset mp mount_point
	typeset id
	typeset dev

	#
	# Get all mounts points in the correct heirarchical order.
	# Save the mount point ID so we can grab other associated
	# attributes for that mount later.
	#
	while : ; do

	  #
	  # We loop on devices looking for devices that have mount points
	  #
	  dev=$(getSvcDevice $DB $SVC_ID $n)
	  case $? in
	    0) : ;;		# found it, let it fall through
	    2) break ;;		# done getting devices, no more left
	    *) logAndPrint $LOG_ERR "\
stopDevice: Cannot get device $n of service $SVC_NAME, err=$?"
	       return $FAIL;;
	  esac

	  mp=$(getSvcMountPoint $DB $SVC_ID $n)
	  case $? in
	    0) : ;;		# found a mount point
	    2) ((n=n+1))	# no mount point, go to next device
	       continue ;;
	    *) logAndPrint $LOG_ERR "\
stopDevice: Cannot get mount point $n of service $SVC_NAME, err=$?"
	       return $FAIL ;;
	  esac

	  print "$mp $n" >> $TMPFILE
	  ((n=n+1))
	done

	#
	# Sort the mount points in reverse order and umount them
	#
	if [ -s $TMPFILE ]; then
	  sort -r $TMPFILE |&
	  while read -p mount_point id
	  do
	    stopDevice $id
	    if [ $? -ne $SUCCESS ]; then
	      rm -f $TMPFILE
	      return $FAIL
	    fi
	  done
	fi

	#
	# Stop all of the devices that are not mounts
	#
	((n=MIN_TOKEN_ID))
	cat /dev/null > $TMPFILE
	while : ; do
	  dev=$(getSvcDevice $DB $SVC_ID $n)
	  case $? in
	    0) : ;;		# found it, let it fall through
	    2) break ;;		# done getting devices, no more left
	    *) logAndPrint $LOG_ERR "\
stopDevice: Cannot get device $n of service $SVC_NAME, err=$?"
	       rm -f $TMPFILE
	       return $FAIL;;
	  esac

	  mp=$(getSvcMountPoint $DB $SVC_ID $n)
	  case $? in
	    0) ((n=n+1))	# Already stopped ... skip to next device
	       continue ;;
	    2) : ;;		# Did not find a mount, let it fall through
	    *) logAndPrint $LOG_ERR "\
stopDevice: Cannot get mount point $n of service $SVC_NAME, err=$?"
	       rm -f $TMPFILE
	       return $FAIL ;;
	  esac

	  print "$dev $n" >> $TMPFILE
	  ((n=n+1))
	done

	cat $TMPFILE |&
	if [ -s $TMPFILE ]; then
	  while read -p dev id
	  do
	    stopDevice $id
	    if [ $? -ne $SUCCESS ]; then
	      rm -f $TMPFILE
	      return $FAIL
	    fi
	  done
	fi

	rm -f $TMPFILE
	return $SUCCESS
}

