/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/*
 * $Id: diskcomms.c,v 1.5 2000/08/23 15:13:22 burke Exp $
 *
 * Copyright 2000, Mission Critical Linux, LLC
 *
 * author: Jeff Moyer <moyer@missioncriticallinux.com>
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>
#include <sys/syslog.h>
#include <errno.h>
#include <unistd.h>
#include <sys/time.h>

#include "diskcomms.h"
#include <logger.h>


#ifdef DEBUG
#define Dprintf(fmt,args...) printf(fmt,##args)
#define DBG_ASSERT(x)        assert(x)
#else
#define Dprintf(fmt,args...)
#define DBG_ASSERT(x)
#endif

extern quorumdConnectionSt *cnx_list;
extern time_t last_sm_contact, last_powerd_contact;
extern pid_t powerd_daemon_pid, sm_daemon_pid;
extern msg_handle_t listenFD;

extern void fill_generic_hdr(generic_msg_hdr *hdr, int command, int len);
extern int getNodeState(int nodeNum);
extern int notifySMStartup(void);
extern void processConfigChange(void);


static const char *version __attribute__ ((unused)) = "$Revision: 1.5 $";

int
quorumdAddConn(int fd)
{
    quorumdConnectionSt *new;

    new = (quorumdConnectionSt*)malloc(sizeof(quorumdConnectionSt));
    if (new == NULL) {
	Dprintf("out of memory.  bummer\n");
	return -1;
    }

    memset(new, 0, sizeof(quorumdConnectionSt));
    new->fd = fd;

    if (cnx_list) {
	new->next = cnx_list;
	new->prev = NULL;
	cnx_list->prev = new;
    }

    cnx_list = new;
    return 0;
}


int
quorumdDelConn(quorumdConnectionSt *cnxp)
{
    DBG_ASSERT(cnxp);

    if (cnxp->prev == NULL) { /* first element of list */
	cnx_list = cnxp->next;
	if (cnx_list)
	    cnx_list->prev = NULL;
	free(cnxp);
	return 0;
    }

    cnxp->prev->next = cnxp->next;
    free(cnxp);
    return 0;
}

/*
 * Take a command received in the quorumdBody and do the necessary
 * processing on it.  We need not close the file descriptor, as that
 * is done by the caller.
 *
 * If all goes well, we will _always_ return 1.  If we ever return 0, it
 * means that stopQuorumd() failed.  I don't know that this can happen.
 */
int
process_disk_command(quorumdConnectionSt *cnxp)
{
    DiskMessageSt sendMsgBuf, rcvMsgBuf;

    int retval;
    int proceed=1;
    pid_t new_pid;

    switch (cnxp->msg.hdr.command) {

    /*
     * Returns an array representing the state of all nodes
     * in the cluster.  Principaly used by monitoring utility.
     */
    case DISK_NODE_STATES:
	clulog(LOG_DEBUG, "quorumdBody: DISK_NODE_STATES\n");
	fill_generic_hdr(&sendMsgBuf.hdr, DISK_NODE_STATES, 
			 DISK_MESSAGE_SIZE);
	for (retval=0; retval < MAX_NODES; retval++) {
	    sendMsgBuf.data.nodeStates.states[retval] = getNodeState(retval);
	}
	retval = msg_send(cnxp->fd, &sendMsgBuf, DISK_MESSAGE_SIZE);
	if (retval != DISK_MESSAGE_SIZE){
	    clulog(LOG_ERR, "quorumdBody: msg_send failed in DISK_NODE_STATES,"
		   " %d.\n", retval);
	}
	break;
    /*
     * The service manager is nofifying us that its now
     * running.  After SM sends us this message it is 
     * waiting for host up / down notification messages
     * to cause it to initiate starting services.
     * This message is also sent periodically to indicate that the SM
     * daemon is still alive.
     */
    case DISK_SM_ALIVE:
	/*
	 * This needs to be an authenticated message.
	 */
	if (!cnxp->secure) {
	    clulog(LOG_CRIT, "** process_disk_command: Received DISK_SM_ALIVE "
		   "message from unauthorized source. **\n");
	    break;
	}
	last_sm_contact = time(NULL);
	new_pid = cnxp->msg.data.daemonPid;
	clulog(LOG_DEBUG, "quorumdBody: DISK_SM_ALIVE, pid=%ld.\n", new_pid);
	/*
	 * Don't send any notifications to SM until we conclude the 
	 * state of the partner node.  This is necessary because one of the
	 * first things SM is going to do is to take out the synchrnoization
 	 * lock.  BUT, if the partner node is down and went down uncleanly,
	 * when we conclude its down, we will reset the partner's lock to
	 * being clear.  This must be done before notifying SM to insure that
	 * all the lock states are initialized properly.
	 */
	if ((new_pid != 0) && (sm_daemon_pid != new_pid)) {
	    if (notifySMStartup() == 0) {
	        clulog(LOG_INFO, "quorumdBody: new SM at pid=%ld.\n", new_pid);
	        sm_daemon_pid = new_pid;
	    }
        }
	break;
    case DISK_SM_EXITING:
	/*
	 * This needs to be an authenticated message.
	 */
	if (!cnxp->secure) {
	    clulog(LOG_CRIT,"** process_disk_command: Received DISK_SM_EXITING"
		   " message from unauthorized source. **\n");
	    break;
	}
	/*
	 * Clean shutdown case. SM is telling us that its
	 * going away as an indication of a clean shutdown.
	 */
	clulog(LOG_DEBUG, "quorumdBody: DISK_SM_EXITING\n");
	proceed = 0;
	break;
    case DISK_POWERD_ALIVE:
	/*
	 * This needs to be an authenticated message.
	 */
	if (!cnxp->secure) {
	    clulog(LOG_CRIT, "*** process_disk_command: Received "
		  "DISK_POWERD_ALIVE message from unauthorized source. ***\n");
	    break;
	}
	/*
	 * Message from powerd sent periodicaly to indicate that its alive.
	 * Update timestamp to mark last activity.
	 */
	new_pid = cnxp->msg.data.daemonPid;
	clulog(LOG_DEBUG, "quorumdBody: DISK_POWERD_ALIVE, pid=%ld.\n", new_pid);
	last_powerd_contact = time(NULL);
	if ((new_pid != 0) && (powerd_daemon_pid != new_pid)) {
	    clulog(LOG_INFO, "quorumdBody: new Powerd at pid=%ld.\n", new_pid);
	    powerd_daemon_pid = new_pid;
        }
	break;
    case DISK_CONFIG_CHANGE:
	/*
	 * Message sent by configuration utilities to indicate a change
	 * in the configuration settings stored on the quorum partition within
 	 * the "configuration database" portion. 
	 */
	clulog(LOG_DEBUG, "quorumdBody: DISK_CONFIG_CHANGE\n");
	processConfigChange();
	break;
    default: // unrecognized command
	clulog(LOG_ERR, "quorumdBody: unrecognized command %d.\n",
		rcvMsgBuf.hdr.command);
	break;
    }

    return proceed;
}


/*
 * int process_message(quorumdConnectionSt *cnxp)
 *
 * Return Values:
 *
 *  1 - No errors, but there is more data to read, so don't close the cnx.
 *  0 - No errors, and all data was read.  Close the connection.
 * -1 - An error occurred, close the connection.
 */
#define check_return(x)    \
do {                       \
    if ((x) <= 0) {        \
        msg_close(sockfd); \
        return -1;         \
    }                      \
} while (0)

int
process_message(quorumdConnectionSt *cnxp)
{
    ssize_t        msglen=-1, bytes_ready=-1, bytes_left=-1;
    int            sockfd=-1, auth=0;
    char           *msg = NULL;
    char           *readbufp;

    sockfd = cnxp->fd;
    msg = (char *)&cnxp->msg;

    if (cnxp->offset) {
	readbufp = (char *)msg+cnxp->offset;
	bytes_left = sizeof(DiskMessageSt) - cnxp->offset;
    } else {
	readbufp = msg;
	bytes_left = sizeof(DiskMessageSt);
    }

    bytes_ready = msg_peek(sockfd, readbufp, bytes_left);
    check_return(bytes_ready);

    msglen = msg_receive(sockfd, readbufp, bytes_left, &auth);
    check_return(msglen);

    cnxp->secure = auth;

    if ((msglen + cnxp->offset) != sizeof(DiskMessageSt)) {
	cnxp->offset += msglen;
	return 1; /* Need more data */
    }

    return 0;
}


int
process_requests(int secs)
{
	int                nfds=-1;
	int                proceed = 1, maxfd=-1, ret = -1;
	fd_set             readfds;
	time_t             comms_begin_time, comms_elapsed_time;
	struct timeval     tv;
	msg_handle_t       acceptFD = -1;
	quorumdConnectionSt *cnxp, *cnxp_curr;
	static int         dbg_num_conns=0;

	tv.tv_sec = secs;
	tv.tv_usec = 0;

	comms_begin_time = time(NULL);

	while (tv.tv_sec) {

		FD_ZERO(&readfds);

		FD_SET(listenFD, &readfds);
		maxfd = listenFD;

		for (cnxp = cnx_list; cnxp != NULL; cnxp = cnxp->next) {
			FD_SET(cnxp->fd, &readfds);

			if (cnxp->fd > maxfd)
				maxfd = cnxp->fd;
		}

		nfds = select(maxfd+1, &readfds, NULL, NULL, &tv);
		if (nfds <= 0)
			continue;

		/*
		 * Process requests which are already pending first.
		 */
		cnxp = cnx_list;
		while (cnxp && (nfds > 0)) {
		    if (FD_ISSET(cnxp->fd, &readfds)) {
			    nfds--;
			    ret = process_message(cnxp);
			    switch (ret) {
			    case -1:
				    clulog(LOG_ERR, "process_incoming"
					   "_requests: Error receiving "
					   "data on connection %d.\n", 
					   cnxp->fd);
				    cnxp_curr = cnxp;
				    cnxp = cnxp->next;
				    quorumdDelConn(cnxp_curr);
				    dbg_num_conns--;
				    break;

			    case 0:
				    clulog(LOG_DEBUG, "process_incoming"
					   "_requests: processing "
					   "request from connection %d.\n",
					   cnxp->fd);
				    proceed = process_disk_command(cnxp);
				    cnxp_curr = cnxp;
				    cnxp = cnxp->next;
				    msg_close(cnxp_curr->fd);
				    quorumdDelConn(cnxp_curr);
				    dbg_num_conns--;

				    if (proceed == 0) /* High Priority */
					    return proceed;
				    continue;

			    case 1:
				    clulog(LOG_INFO, "process_incoming"
					   "_requests: Waiting for "
					   "more data on connection %d\n",
					   cnxp->fd);
			    default:
				    clulog(LOG_ERR, "process_incoming"
					   "_requests: Got invalid return"
					   " code from process_message.");
				    break;
			    }
		    }
		    if (cnxp)
			cnxp = cnxp->next;
		}

		if (FD_ISSET(listenFD, &readfds)) {
			if ((acceptFD = msg_accept(listenFD)) < 0) {
				clulog(LOG_ERR,"process_incoming_requests: "
				       "error in accept. %s\n", strerror(errno));
				continue;
			}
			quorumdAddConn(acceptFD);
			acceptFD = -1;
			dbg_num_conns++;
		}

		/*
		 * Recalculate our timeslice.
		 */
		comms_elapsed_time = time(NULL) - comms_begin_time;
		if (comms_elapsed_time >= secs)
		        break;
		else
			tv.tv_sec = secs - comms_elapsed_time + 1;
	} /* while(tv.tv_sec | tv.tv_usec) */

	return proceed;
}

void
close_open_comms(void)
{
	quorumdConnectionSt *cnxp, *curr;

	cnxp = cnx_list;
	while (cnxp) {
		if (cnxp->fd > 0)
			msg_close(cnxp->fd);
		curr = cnxp;
		cnxp = cnxp->next;
		quorumdDelConn(curr);
	}
}
/*
 * Local variables:
 *  c-basic-offset: 4
 *  c-indent-level: 4
 *  tab-width: 8
 * End:
 */
