/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/

/* RPS-10 power switch library.

   author: Ron Lawrence <lawrence@missioncriticallinux.com>
*/

#include <power.h>
#include <parseconf.h>
#include <clusterdefs.h>
#include "plock.h"

#include <unistd.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>
#include <stdio.h>
#include <sys/time.h>
#include <string.h>
#include <errno.h>
#include <regex.h>
#include <signal.h>

#include <clucfg.h>

static const char *version __attribute__ ((unused)) = "$Id: power_rps10.c,v 1.16 2000/11/10 19:29:48 burke Exp $";

/*#define LOG_STDERR*/
/*#define USE_LOGGING*/

#if defined(LOG_STDERR)
#define PR(format, args...) fprintf(stderr, "PWR_RPS10: " format, ## args)
#elif defined(USE_LOGGING)
#include <sys/syslog.h>
#include <msgsvc.h>
#include <logger.h>
#define PR(format, args...) clulog(LOG_DEBUG, format, ## args)
#else
#define PR(format, args...)
#endif 


enum PWR_boolean PWR_RPS10_configure(char *config_file_name);
enum PWR_boolean PWR_RPS10_set(char* option, char* value);
enum PWR_boolean PWR_RPS10_get(char* option, char *dflt, char** value);
enum PWR_boolean PWR_RPS10_init(PWR_result *status);
enum PWR_boolean PWR_RPS10_release(void);
PWR_result  PWR_RPS10_status(void);
PWR_result  PWR_RPS10_reboot(void);
PWR_result  PWR_RPS10_off(void);
PWR_result  PWR_RPS10_on(void);


/* Implementation Details */

#define BUFFER_LEN 255
#define DFLT_TMP_BUFFER_SZ 32

#define STREQ(s1,s2) (strcmp((s1),(s2)) == 0)

#define ZEROLIMIT 7             /* The number of times to call read before
                                   giving up when it is returning zero's. */
#define ERRORLIMIT 7            /* The number of times to call read
                                   when it is returning errors before
                                   giving up. */
#define READLIMIT 21            /* The maximum number of times we will
                                   attempt to read from serial_fd on
                                   any given go.  To prevent lossage
                                   due to unpredicted pathological
                                   continuous looping. */

static int initialized = 0;
static int locked = 0;

static regex_t on_off_regex;
static regex_t complete_regex;
static regex_t ready_regex;

static int regex_allocated=0;

/* the name of the serial device file. */ 
// XXX - have no default device; thats a configuration error.
#define DFLT_NO_DEVICE "none"
static char serial_device_file[BUFFER_LEN] = DFLT_NO_DEVICE;

/* Serial device modes. Things like baud, parity, stop bits, etc. */

#define TTY_LOCK_D "/var/lock"

static int serial_fd = 0;
static struct termios oldtio, newtio;
static char buffer[BUFFER_LEN];

static long timeout = 2;
static long init_timeout = 10;

/* When set, the following variable causes dtr to be toggled on the
   serial line when opening and closing.  This causes the power switch
   to "restart", and, after a delay, to print RPS-10 Ready.  The
   default, set here, is to perform the reset.  This might be a bad
   thing for programs that repeatedly call pswitch through its shell
   interface.  

   this might be a reasonable thing to control via a command line
   switch from, e.g. pswitch.  That is not currently (Wed Apr 26
   14:25:20 2000) being done.
*/
static int perform_reset = 1;

#define BAUDRATE B9600

/* Implementation Variables */

static PWR_result status;

static char dflt_config_filename[BUFFER_LEN] = CLU_CONFIG_FILE; 

/* Private functions */

static PWR_result internal_status(long);

#define SET(where, what)        (where |= (what))
#define RESET(where, what)      (where &= ~(what))
#define ISSET(where, what)      (where & (what))
#define SETIFF(condition, where, what)          \
  {if((condition)) {                            \
    SET((where),(what));}                       \
  else {                                        \
    RESET((where),(what));                      \
  }}

/* Private: send a command to the switch. 
*/
static int send_command(char c, long ttimeout) {
  int length, result;
  fd_set wfds, xfds;
  struct timeval tv;
  int retval;

  FD_ZERO(&wfds);
  FD_ZERO(&xfds);

  FD_SET(serial_fd, &wfds);
  FD_SET(serial_fd, &xfds);

  tv.tv_sec = ttimeout;
  tv.tv_usec = 0;

  retval = select(serial_fd+1, NULL, &wfds, &xfds, &tv);
  switch(retval) {
  case -1:                      /* Error */
    return 0;
  case 1:                       /* something happened on our fd */
    if (FD_ISSET(serial_fd, &xfds))
      return 0;                 /* an exception occurred */
    break;
  case 0:                       /* timeout */
    return 0;
  }
  snprintf(buffer, BUFFER_LEN, "%c%c%c%c%c%c0%c%c", 
           0x02, 0x18, 0x18, 0x02, 0x18, 0x18, c, 0x0D);
  length = strlen(buffer);
  result = write(serial_fd, buffer, length);
  if (result == length)
    return 1;
  else
    return 0;
}

/* Private: Read a response from the switch. 
*/
static int read_onoff(PWR_result *result, long ttimeout, int rebooted) {
  int ready = 0;
  int count, readresult;
  fd_set rfds, xfds;
  struct timeval tv;
  int retval;
  int rc;
  regmatch_t match[2];
  int zerocount = 0, ecount = 0, readcount=0;
  int onoff_regex_count = 0;
  
  if(initialized)
    SET(*result, PWR_INIT);
  else
    RESET(*result, PWR_INIT);

  FD_ZERO(&rfds);
  FD_ZERO(&xfds);

  FD_SET(serial_fd, &rfds);
  FD_SET(serial_fd, &xfds);

  tv.tv_sec = ttimeout;
  tv.tv_usec = 0;

  retval = select(serial_fd+1, &rfds, NULL, &xfds, &tv);
  switch(retval) {
  case -1:                    /* Error */
    RESET(*result, PWR_TIMEOUT);
    SET(*result, PWR_ERROR);
    return 0;
    break;
  case 1:                     /* something happened on our fd. */
    if( FD_ISSET(serial_fd, &xfds) ) { 
      /* an exception happened */
      RESET(*result, PWR_TIMEOUT); 
      SET(*result, PWR_ERROR); 
      return 0; 
    } 
    break;
  case 0:                     /* timedout */
    tcflush(serial_fd, TCIOFLUSH);
    SET(*result, PWR_TIMEOUT);
    SET(*result, PWR_ERROR);
    return 0;
    break;
  }
  count = 0;
  while(!ready) {
    readcount++;
    if(readcount > READLIMIT) {
      /* This is a catchall failsafe to prevent unpredicted
         pathologies from causing effectively infinite loops.  This
         while loop is the only loop in this module; by setting a
         limit on the number of times we iterate through it, we reduce
         the likelihood that we will fail to return when called. */
      RESET(*result, PWR_TIMEOUT);
      SET(*result, PWR_ERROR);
      PR("Too many reads(%d) from %s\n", readcount, serial_device_file);
      return 0;
    }
    readresult = read( serial_fd, buffer + count, BUFFER_LEN - count);
    if(-1 == readresult) {
#ifdef DEBUG
      perror( __FUNCTION__ " read error: ");
#endif
      if(errno != EAGAIN) ecount ++;
      if(ecount > ERRORLIMIT) {
        RESET(*result, PWR_TIMEOUT);
        SET(*result, PWR_ERROR);
        PR("Errors reading from %s\n", serial_device_file);
        return 0;
      }
      else {
        sleep(2);
        continue;
      }
    }
    if(0 == readresult) {
      PR("Zero bytes read\n");

      zerocount++;
      if(zerocount > ZEROLIMIT) {
        RESET(*result, PWR_TIMEOUT);
        SET(*result, PWR_ERROR);
        PR("Can't read from %s\n", serial_device_file);
        return 0;
      }
      else {
        sleep(2);
        continue;
      }
    }
    count += readresult;
    buffer[count] = 0;
    if(! strcmp(buffer, "\n" )) {
      PR("! strcmp(buffer, \"\\n\" )\n");
      count = 0;
      continue;
    } else {
      PR("read a string: %s\n", buffer);
      /* Expecting On/Off string. */
      rc = regexec(&on_off_regex, buffer, 2, match, 0);
      if(REG_NOMATCH != rc) {
        PR("matched on.off regex, %s\n", buffer);
        onoff_regex_count++;
        
        if(! strncmp(buffer + match[1].rm_so, "On", strlen("On")))
          SET(*result, PWR_SWITCH);
        else
          RESET(*result, PWR_SWITCH);
        if(rebooted && 2 == onoff_regex_count ) {
          /* Device doesn't ever get around to printing Complete after
             a reboot.  Lame.  This is an admittedly cheezy
             work-around. */
          RESET(*result, PWR_TIMEOUT);
          RESET(*result, PWR_ERROR);
          return 1;
        }
        count = 0;
      }
      /* Expecting Complete */
      rc = regexec(&complete_regex, buffer, 2, match, 0);
      if(REG_NOMATCH != rc) {
        PR("matched complete\n");
        ready = 1;
        break;
      }
      /* Expecting RPS-10 Ready */
      rc = regexec(&ready_regex, buffer, 2, match, 0);
      if(REG_NOMATCH != rc) {
        PR("matched RPS ready\n");
        RESET(*result,PWR_ERROR);
        RESET(*result,PWR_TIMEOUT);
        RESET(*result,PWR_INIT); /* Treat this as if initialization is
                                    not yet fully completed. */
        return 1;
      }
    }
  }
  RESET(*result, PWR_TIMEOUT);
  RESET(*result, PWR_ERROR);
  return 1;
}

/* Private: Get the status from the switch, constrained by the given
   timeout. 
*/
static PWR_result internal_status(long ttimeout) {
  int rc;
  PWR_result result = 0;

  rc = send_command('?', ttimeout);
  PR("sent command ?\n");
  if(!rc) {
    PR("send returned bad rc returning result: %d\n",result);
    SET(result, PWR_ERROR);
    return result;
  }
  read_onoff(&result, ttimeout,0);
  PR( "read onoff, result: %d\n",result);
  return result;
}

/* Private: send a command to the switch.  Return the result. 
*/
static PWR_result internal_command(char c) {
  int rc;
  PWR_result result=0;

  if(initialized) {
    SET(result, PWR_INIT);
    rc = send_command(c, timeout);
    PR("sent command %c\n", c);
    if(!rc) {
      PR("send returned bad rc returning result: %d\n",result);
      SET(result, PWR_ERROR);
      return result;
    }
    if( 'T' == c) {
      read_onoff(&result, timeout,1);
    }
    else {
      read_onoff(&result, timeout,0);
    }
    PR( "read onoff, result: %d\n",result);
  }
  else {
    PR(__FUNCTION__ " not initialised");
    SET(result, PWR_ERROR);
  }
  SETIFF(initialized, result, PWR_INIT);
  return result;
}

/* Private: Reset the serial device. */
static void serial_reset(void) {
  struct termios tty, old;
  int sec = 2;

  tcgetattr(serial_fd, &tty);
  tcgetattr(serial_fd, &old);
  cfsetospeed(&tty, B0);
  cfsetispeed(&tty, B0);
  tcsetattr(serial_fd, TCSANOW, &tty);
  if (sec>0) {
    sleep(sec);
    tcsetattr(serial_fd, TCSANOW, &old);
  }
}

/* Private: Open the serial device.  Set the parameters on the serial
   line.  If the device cannot be opened, we return some negative
   error code.  
*/
static int open_serial(char * device_file) {
  int rc;

  serial_fd = open(device_file, O_RDWR | O_NOCTTY | O_NDELAY );
  if (serial_fd <0) {
#ifdef DEBUG
    perror(device_file);
#endif
    return -1;
  }
  rc = tcgetattr(serial_fd, &oldtio);
  if(rc < 0) {
    PR( "tcgetattr(serial_fd, &oldtio) failed\n");
    return rc;
  }
  memset(&newtio, 0, sizeof(newtio));

  newtio.c_cflag = BAUDRATE | CRTSCTS | CS8 | CLOCAL | CREAD;
  newtio.c_iflag =  0;
  newtio.c_oflag = 0;
  newtio.c_lflag = ICANON;

  newtio.c_cc[VINTR]    = 0;
  newtio.c_cc[VQUIT]    = 0;
  newtio.c_cc[VERASE]   = 0;
  newtio.c_cc[VKILL]    = 0;
  newtio.c_cc[VEOF]     = 4;
  newtio.c_cc[VTIME]    = 0;
  newtio.c_cc[VMIN]     = 1;
  newtio.c_cc[VSWTC]    = 0;
  newtio.c_cc[VSTART]   = 0;
  newtio.c_cc[VSTOP]    = 0;
  newtio.c_cc[VSUSP]    = 0;
  newtio.c_cc[VEOL]     = 0;
  newtio.c_cc[VREPRINT] = 0;
  newtio.c_cc[VDISCARD] = 0;
  newtio.c_cc[VWERASE]  = 0;
  newtio.c_cc[VLNEXT]   = 0;
  newtio.c_cc[VEOL2]    = 0;

  rc = tcsetattr(serial_fd, TCSANOW, &newtio);
  if(rc < 0) {
    PR( "tcsetattr(serial_fd, TCSANOW, &newtio) failed\n");
    return rc;
  }
  rc = tcflush(serial_fd, TCIFLUSH);
  if(rc < 0) {
    PR("tcflush(serial_fd, TCIFLUSH) failed\n");
    return rc;
  }
  /* Here we drop and raise dtr on the serial line.  This will make
     certain that the switch is being accessed in a known state.  The
     downside is that the initialization will take longer than it
     would in the case where the switch was already listening on the
     line.  This could potentially be a big problem, esp. for programs
     that are accessing the device through the scriptable command
     interface.

     The following is some code that is apparently POSIX'ly correct.
     It was taken (almost) verbatim from minicom sources.  
  */
  if(perform_reset) {
    serial_reset();
  }
  fcntl(serial_fd, F_SETFL, O_NONBLOCK);
  return 0;
}

/****************************************************************************/

/* Public Interface Functions */

/* Read a configuration file for settings and parameters. 
*/
enum PWR_boolean PWR_RPS10_configure(char *config_file_name) {
  char *pstr;
  /* when a configuration filename hasn't been passed in...
   */
  if( NULL == config_file_name ||
      ! strcmp(config_file_name, "")) {
    /* If a config file has already been read, it may have an entry
       that tells us where to read a config file from.
    */
    CFG_Get("power%configfile", dflt_config_filename, &pstr);
    /* Read the file specified, or the default configu file.
     */
    return CFG_OK == CFG_ReadFile(pstr);
  }
  else {
    return CFG_OK == CFG_ReadFile(config_file_name);
  }
}

/* Set or query the configuration parameters of the switch interface. */
enum PWR_boolean PWR_RPS10_set(char* option, char* value) {
  CFG_status rc;
  rc = CFG_Set(option, value);
  if(CFG_OK == rc) {
    return PWR_TRUE;
  }
  else {
    return PWR_FALSE;
  }
}

enum PWR_boolean PWR_RPS10_get(char* option, char* dflt, char** value) {
  CFG_status rc;
  rc = CFG_Get(option, dflt, value);
  if(CFG_OK == rc || CFG_DEFAULT == rc) {
    return PWR_TRUE;
  }
  else {
    return PWR_FALSE;
  }
}

/* Initialize the power switch communication paths. */
enum PWR_boolean PWR_RPS10_init(PWR_result *s) {
  char sdflt[DFLT_TMP_BUFFER_SZ];
  char *svalue;
  long lvalue;
  int lock_result;
  CluCfg              *cfg;
  char                *file = NULL;

  if(!initialized) {
    /* Compile regular expressions. */
    if(!regex_allocated) {
      regcomp(&on_off_regex,
              "[ \t\n]*Plug 0 (On|Off)[ \t\n]*",
              REG_EXTENDED);
      regcomp(&complete_regex,
              "[ \t\n]*Complete[ \t\n]*",
              REG_EXTENDED);
      regcomp(&ready_regex,
              "[ \t\n]*RPS-10 Ready[ \t\n]*",
              REG_EXTENDED);
      regex_allocated=1;
    }
    /* Check to see if a device has been set, possibly from the
       command line. */
    snprintf(sdflt, DFLT_TMP_BUFFER_SZ, "%s", DFLT_NO_DEVICE);
    PWR_get("power%device", sdflt, &svalue);
    if( STREQ(svalue,DFLT_NO_DEVICE)) {
      /* Read configuration variables. */
      cfg = get_clu_cfg(file);
      if (cfg != NULL) {
        strcpy(serial_device_file, 
               cfg->nodes[cfg->lid].powerSerialPort);
        free(cfg);
      }
      else {
        strcpy(serial_device_file, DFLT_NO_DEVICE);
      }
    } else {
      strcpy(serial_device_file, svalue);
    }
    snprintf(sdflt, DFLT_TMP_BUFFER_SZ, "%ld", init_timeout);
    PWR_get("power%init_timeout", sdflt, &svalue);
    sscanf(svalue, "%ld", &lvalue);
    init_timeout = lvalue;

    snprintf(sdflt, DFLT_TMP_BUFFER_SZ, "%ld", timeout);
    PWR_get("power%timeout", sdflt, &svalue);
    sscanf(svalue, "%ld", &lvalue);
    timeout = lvalue;
  
    if (!locked) {
      /* Create a lock file for the serial device. */
      if (strcmp(serial_device_file, DFLT_NO_DEVICE) == 0) {
        PR("serialPort not specified\n");
        SET(status, PWR_ERROR);
      }
      else if((lock_result = pwr_lock_serial_device(serial_device_file)) < 0) {
        PR("can't lock %s, lock result: %d\n", 
           serial_device_file,
           lock_result);
        SET(status, PWR_ERROR);
      }
      else {
        PR("locked %s\n", serial_device_file);
        locked = 1;
        /* Open and query the serial device. */
        if(open_serial(serial_device_file) < 0) {
          SET(status,PWR_ERROR);
          *s = status;
          return PWR_FALSE;
        }
      }
    } else { /* locked */
      /*
       * This situation is basically the following:  we already called
       * this initialization routine, but we were unable to contact the
       * power switch.  Thus, we have it locked, we need only cycle dtr
       * and hope that we can talk to it.
       */
      if (serial_fd > 0)
        serial_reset();
      else
        open_serial(serial_device_file);
    }
    status = internal_status(init_timeout);
    if(!(status & PWR_ERROR) && 
       !(status & PWR_TIMEOUT) && 
       !(status & PWR_INIT)) {
      /* The switch isn't fully initialized, yet it is not in error,
         nor has it timed out.  This can only mean that we have just
         received the RPS-10 Ready string.  Send the status command
         again; the switch should now be ready to act on
         commands. 
      */
      status = internal_status(init_timeout);
    }
  }
  else {
    status = internal_status(timeout);
  }
  if( ISSET(status, PWR_TIMEOUT) || ISSET(status, PWR_ERROR)) {
    initialized = 0;
    RESET(status, PWR_INIT);
    *s = status;
    /* 'close()' will cause the program to hang if the fd hasn't been
       tcflush'ed beforehand. */
    tcflush(serial_fd, TCIOFLUSH);
    close(serial_fd);
    return PWR_FALSE;
  } else {
    initialized = 1;
    SET(status,   PWR_INIT);
    *s = status;
    return PWR_TRUE;
  }
}

/* Give back resources to the system.  Flush the serial port, and
   close the file.  This must be done prior to calling exit.  
*/
enum PWR_boolean PWR_RPS10_release(void) {
  if(initialized) {
    if(regex_allocated) {
      regfree(&on_off_regex);
      regfree(&complete_regex);
      regex_allocated = 0;
    }
    if(pwr_unlock_serial_device(serial_device_file) < 0 ) {
      PR("failed to unlock serial device\n");

    }
    /* Toggle dtr on the serial line.  This way the switch will know
       we are no longer talking to it. 
    */
    if(perform_reset) {
      serial_reset();
    }
    /* 'close()' will cause the program to hang if the fd hasn't been
       tcflush'ed beforehand. */
    tcflush(serial_fd, TCIOFLUSH);
    close(serial_fd);
  }
  initialized = 0;
  return 1;
}

/* Query the switch to determine its current status, and its readiness
   to carry out additional commands.  Returns a PWR_result that
   contains flags that tell whether the switch is open or closed.
*/
PWR_result PWR_RPS10_status(void) {
  return internal_status(timeout);
}

/* Send a command to the switch that will cause the system powered by
   the switch to reboot.  Normally, this means that the switch will be
   opened for some period of time, after which it will be closed
   again.  
*/
PWR_result PWR_RPS10_reboot(void) {
  return internal_command('T');
}

/* Send a command to the switch that will cause the switch to open,
   securing power to the effected system. 
*/
PWR_result PWR_RPS10_off(void) {
  return internal_command('0');
}

/* Send a command to the switch that will cause the switch to close,
   supplying power to eht effected system. 
*/
PWR_result PWR_RPS10_on(void) {
  return internal_command('1');
}

