/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.

  Modifications to power_rps10.c Copyright Computer Generation Incorporated, 2000

*/

/*

   Nightware RPC100S power switch library, copied from power_rps10.c
   
   Original author: Ron Lawrence <lawrence@missioncriticallinux.com>
   Modifications contributed by: Eric Z. Ayers <eric.ayers@compgen.com>
      Computer Generation, Incorporated.
      
   The Nightware RPS-100S is manufactured by:
   
      Micro Energetics Corp
      +1 703 250-3000
      http://www.nightware.com/

   Thank you to David Hicks of Micro Energetics Corp. for providing
   a demo unit to write this software.
      
   This switch has a very simple protocol, 
   You issue a command and it  gives a response.
   Sample commands are conveniently documented on a sticker on the
      bottom of the device.
      
   The switch accepts a single command of the form

   //0,yyy,zzz[/m][/h]<CR>
   
     Where yyy is the wait time before activiting the relay.
           zzz is the relay time.

     The default is that the relay is in a default state of ON, which
     means that  usually yyy is the number of seconds to wait
     before shutting off the power  and zzz is the number of seconds the
     power remains off.  There is a dip switch to change the default
     state to 'OFF'.  Don't set this switch. It will screw up this code. 

     An asterisk can be used for zzz to specify an infinite switch time.
     The /m /and /h command options will convert the specified wait and
     switch times to either minutewes or hours. 
   
   A response is either
    <cr><lf>OK<cr><lf>
       or
    <cr><lf>Invalid Entry<cr><lf>


   As far as THIS software is concerned, we have to implement 4 commands:

   status     -->    //0,0,BOGUS; # Not a real command, this is just a
                                  #   probe to see if switch is alive
   open(on)   -->    //0,0,0;     # turn power to default state (on)
   close(off) -->    //0,0,*;     # leave power off indefinitely
   reboot     -->    //0,0,10;    # immediately turn power off for 10 seconds.

   and expect the response 'OK' to confirm that the unit is operational.



*/


/*#define RPC100S_STATUS_CMD  "0" */ /* This will actually turn the device ON */

#define RPC100S_STATUS_CMD  "BOGUS" /* just to see if the switch is alive */
#define RPC100S_ON_CMD      "0"
#define RPC100S_OFF_CMD     "*"
#define RPC100S_REBOOT_CMD  "10"

#include <power.h>
#include <parseconf.h>
#include <clusterdefs.h>
#include "plock.h"

#include <unistd.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>
#include <stdio.h>
#include <sys/time.h>
#include <string.h>
#include <errno.h>
#include <regex.h>
#include <signal.h>

#include <clucfg.h>

static const char *version __attribute__ ((unused)) = "$Id: power_nw_rpc100s.c,v 1.4 2000/11/10 19:29:48 burke Exp $";

#ifndef NDEBUG
#define PR(format, args...) fprintf(stderr, format, ## args)
#else
#define PR(format, args...)
#endif 


enum PWR_boolean PWR_NW_RPC100S_configure(char *config_file_name);
enum PWR_boolean PWR_NW_RPC100S_set(char* option, char* value);
enum PWR_boolean PWR_NW_RPC100S_get(char* option, char *dflt, char** value);
enum PWR_boolean PWR_NW_RPC100S_init(PWR_result *status);
enum PWR_boolean PWR_NW_RPC100S_release(void);
PWR_result  PWR_NW_RPC100S_status(void);
PWR_result  PWR_NW_RPC100S_reboot(void);
PWR_result  PWR_NW_RPC100S_off(void);
PWR_result  PWR_NW_RPC100S_on(void);


/* Implementation Details */

#define BUFFER_LEN 255
#define DFLT_TMP_BUFFER_SZ 32

#define ZEROLIMIT 7             /* The number of times to call read before
                                   giving up when it is returning zero's. */
#define ERRORLIMIT 7            /* The number of times to call read
                                   when it is returning errors before
                                   giving up. */
#define READLIMIT 21            /* The maximum number of times we will
                                   attempt to read from serial_fd on
                                   any given go.  To prevent lossage
                                   due to unpredicted pathological
                                   continuous looping. */

static int gbl_initialized = 0;
static int locked = 0;

static regex_t ok_regex;
static regex_t invalid_regex;

static int regex_allocated=0;

/* the name of the serial device file. */ 
// XXX - have no default device; thats a configuration error.
#define DFLT_NO_DEVICE "none"
static char serial_device_file[BUFFER_LEN] = DFLT_NO_DEVICE;

/* Serial device modes. Things like baud, parity, stop bits, etc. */

#define TTY_LOCK_D "/var/lock"

static int gbl_serial_fd = 0;
static struct termios oldtio, newtio;
static char gbl_buffer[BUFFER_LEN];

static long timeout= 2 ;
static long init_timeout = 10;

/* RPS10 Note:
   
   When set, the following variable causes dtr to be toggled on the
   serial line when opening and closing.  This causes the power switch
   to "restart", and, after a delay, to print RPS-10 Ready.  The
   default, set here, is to perform the reset.  This might be a bad
   thing for programs that repeatedly call pswitch through its shell
   interface.  

   this might be a reasonable thing to control via a command line
   switch from, e.g. pswitch.  That is not currently (Wed Apr 26
   14:25:20 2000) being done.
*/
static int gbl_perform_reset = 1;

#define BAUDRATE B9600

/* Implementation Variables */

static PWR_result gbl_power_status;

static char dflt_config_filename[BUFFER_LEN] = CLU_CONFIG_FILE; 

/* Private functions */


#define SET(where, what)   (where |= (what))
#define RESET(where, what) (where &= ~(what))
#define ISSET(where, what) (where & (what))

/* Private: send a command string to the switch. 
*/
static PWR_result send_command(char *cmd, long ttimeout)
{
  int length, result;
  fd_set rfds, wfds, xfds;
  struct timeval tv;
  int retval;
  int readcount =0;
  char *p;
  regmatch_t match[2];
  int rc;
  
  /* Empty the serial driver's buffers. We want this dialog
     to be syncronous, and we don't want other commands cluttering it up.
  */
  tcflush(gbl_serial_fd, TCIOFLUSH);
  
  /* Wait for the device to be ready to write to */
  FD_ZERO(&wfds); FD_ZERO(&xfds);

  FD_SET(gbl_serial_fd, &wfds);
  FD_SET(gbl_serial_fd, &xfds);

  tv.tv_sec = ttimeout;
  tv.tv_usec = 0;

  retval = select(gbl_serial_fd+1, NULL, &wfds, &xfds, &tv);
  switch(retval) {
    case -1:
      /* error */
      SET(gbl_power_status, PWR_ERROR);
      return gbl_power_status;
      break;
    case 1:
      /* something happened on our fd. */
      if( FD_ISSET(gbl_serial_fd, &xfds) ) {
        /* an exception happened */
        SET(gbl_power_status, PWR_ERROR);
        return gbl_power_status;
      }
      break;
    case 0:
      /* timedout */
      SET(gbl_power_status, PWR_TIMEOUT);	    
      return gbl_power_status;
      break;
  }

  /*
   * Write the command to the serial port
   */
  snprintf(gbl_buffer, BUFFER_LEN, "//0,0,%s;\r\n", cmd);
           
  length = strlen(gbl_buffer);
  result = write(gbl_serial_fd, gbl_buffer, length);
  if( result != length ) {
    SET(gbl_power_status, PWR_ERROR);
  } 

  /* Now, wait for a result */
  FD_ZERO(&rfds); FD_ZERO(&xfds);

  FD_SET(gbl_serial_fd, &rfds);
  FD_SET(gbl_serial_fd, &xfds);

  tv.tv_sec = ttimeout;
  tv.tv_usec = 0;

  length =0;
  p=gbl_buffer;
  
  /* While there is data to read, let's read from the serial port */
  while ((retval = select(gbl_serial_fd+1, &rfds, NULL, &xfds, &tv)) == 1) {
    readcount++;
    /* Check for an infinite loop */
    if (readcount > READLIMIT) {
      RESET(gbl_power_status, PWR_TIMEOUT);
      SET(gbl_power_status, PWR_ERROR);
      PR("Too many reads(%d) from %s\n", readcount, serial_device_file);
      
      return gbl_power_status;
    }	  
    /*
     * Read the result string from the serial port
     */
    result = read (gbl_serial_fd, p, BUFFER_LEN-length);
    if (result <=0) {
      PR("Got result %d trying to read from serial port, errno %d:%s\n",
	 result, errno, strerror(errno));
      SET(gbl_power_status, PWR_TIMEOUT);	
      return gbl_power_status;
    }

    /* If we match one of the regexp results.  It is time to exit the loop
     */
    rc = regexec (&ok_regex, gbl_buffer, 1, match, 0);
    if (REG_NOMATCH != rc) {
      PR("matched OK regex, %s\n", gbl_buffer);
      RESET(gbl_power_status, PWR_TIMEOUT);
      RESET(gbl_power_status, PWR_ERROR);

      return gbl_power_status;
    }


    rc = regexec (&invalid_regex, gbl_buffer, 1, match, 0);
    if (REG_NOMATCH != rc) {
      PR("matched Invalid regex, %s\n", gbl_buffer);
      RESET(gbl_power_status, PWR_TIMEOUT);
      
      /* NOTE: We expect the status command to return "Invalid Entry" */
      if (!strcmp (RPC100S_STATUS_CMD, cmd)) {
	RESET(gbl_power_status, PWR_ERROR);
      } else {
	SET(gbl_power_status, PWR_ERROR);
      }

      return gbl_power_status;
    }
    
    /*
     * See if we've filled our buffer.
     */
    length += result;
    if (length >= BUFFER_LEN) {
      /* That's all we have room for... bomb out */
      PR("Didn't get a valid result after reading %d chars from serial port\n",
	 length);
      SET(gbl_power_status, PWR_TIMEOUT);	
      return gbl_power_status;      
    }


  } /* end while there is data ready on the serial port */

  /* Hmm, we couldn't match any return value to our expected responses?
     Analyze the return from select()
  */
  switch(retval) {
    case -1:
      /* error */
      SET(gbl_power_status, PWR_ERROR);
      break;
    case 1:
      /* something happened on our fd. */
      if( FD_ISSET(gbl_serial_fd, &xfds) ) {
        /* an exception happened */
        SET(gbl_power_status, PWR_ERROR);
      }
      break;
    case 0:
      /* timedout */
      SET(gbl_power_status, PWR_TIMEOUT);	    
      break;
  }

  return gbl_power_status;
  
}

static void serial_reset(void) {
  struct termios tty, old;
  int sec = 2;

  tcgetattr(gbl_serial_fd, &tty);
  tcgetattr(gbl_serial_fd, &old);
  cfsetospeed(&tty, B0);
  cfsetispeed(&tty, B0);
  tcsetattr(gbl_serial_fd, TCSANOW, &tty);
  if (sec>0) {
    sleep(sec);
    tcsetattr(gbl_serial_fd, TCSANOW, &old);
  }
}

/* Private: Open the serial device.  Set the parameters on the serial
   line.  If the device cannot be opened, we return some negative
   error code.  
*/
static int open_serial(char * device_file) {
  int rc;

  gbl_serial_fd = open(device_file, O_RDWR | O_NOCTTY | O_NDELAY );
  if (gbl_serial_fd <0) {
#ifdef DEBUG
    perror(device_file); 
#endif
    return -1;
  }

  rc = tcgetattr(gbl_serial_fd, &oldtio);
  if(rc < 0) {
    PR( "tcgetattr(gbl_serial_fd, &oldtio) failed\n");
    return rc;
  }
  memset(&newtio, 0, sizeof(newtio));

  newtio.c_cflag = BAUDRATE | CRTSCTS | CS8 | CLOCAL | CREAD;

  /* Tue May 23 11:54:02 2000 RAL changed the value of this flag set
     to all zero.  This stopped the spurious newlines from appearing
     on input. */
  /* Following is the line that I got from the serial howto, now
     commented out: */
  /* newtio.c_iflag = IGNPAR | ICRNL;*/

  newtio.c_iflag =  0;

  newtio.c_oflag = 0;
  newtio.c_lflag = ICANON;

  newtio.c_cc[VINTR]    = 0;
  newtio.c_cc[VQUIT]    = 0;
  newtio.c_cc[VERASE]   = 0;
  newtio.c_cc[VKILL]    = 0;
  newtio.c_cc[VEOF]     = 4;
  newtio.c_cc[VTIME]    = 0;
  newtio.c_cc[VMIN]     = 1;
  newtio.c_cc[VSWTC]    = 0;
  newtio.c_cc[VSTART]   = 0;
  newtio.c_cc[VSTOP]    = 0;
  newtio.c_cc[VSUSP]    = 0;
  newtio.c_cc[VEOL]     = 0;
  newtio.c_cc[VREPRINT] = 0;
  newtio.c_cc[VDISCARD] = 0;
  newtio.c_cc[VWERASE]  = 0;
  newtio.c_cc[VLNEXT]   = 0;
  newtio.c_cc[VEOL2]    = 0;

  rc = tcsetattr(gbl_serial_fd, TCSANOW, &newtio);
  if(rc < 0 ) {
    PR( "tcsetattr(gbl_serial_fd, TCSANOW, &newtio) failed\n");
    return rc;
  }
  rc = tcflush(gbl_serial_fd, TCIFLUSH);
  if(rc < 0) {
    PR("tcflush(gbl_serial_fd, TCIFLUSH) failed\n");

    return rc;
  }
  /* Here we drop and raise dtr on the serial line.  This will make
     certain that the switch is being accessed in a known state.  The
     downside is that the initialization will take longer than it
     would in the case where the switch was already listening on the
     line.  This could potentially be a big problem, esp. for programs
     that are accessing the device through the scriptable command
     interface.

     The following is some code that is apparently POSIX'ly correct.
     It was taken (almost) verbatim from minicom sources.  
  */
  if(gbl_perform_reset) {
    serial_reset();
  }
  fcntl(gbl_serial_fd, F_SETFL, O_NONBLOCK);
  return 0;
}


/****************************************************************************/

/* Public Interface Functions */

/* Read a configuration file for settings and parameters. 
*/
enum PWR_boolean PWR_NW_RPC100S_configure(char *config_file_name) {
  char *pstr;
  /* when a configuration filename hasn't been passed in... 
   */
  if( NULL == config_file_name || 
      ! strcmp(config_file_name, "")) {
    /* If a config file has already been read, it may have an entry
       that tells us where to read a config file from. 
    */
    /* 08/30/00 - Ron Lawrence says this is a vestige and can be ignored -EZA*/
    CFG_Get("power%configfile", dflt_config_filename, &pstr);
    /* Read the file specified, or the default config file. 
     */
    return CFG_OK == CFG_ReadFile(pstr);
  }
  else {
    return CFG_OK == CFG_ReadFile(config_file_name);
  }
}

/* Set or query the configuration parameters of the switch
   interface. 
*/
enum PWR_boolean PWR_NW_RPC100S_set(char* option, char* value) {
  CFG_status rc;
  rc = CFG_Set(option, value);
  if(CFG_OK == rc) {
    return PWR_TRUE;
  }
  else {
    return PWR_FALSE;
  }
}

enum PWR_boolean PWR_NW_RPC100S_get(char* option, char* dflt, char** value) {
  CFG_status rc;
  rc = CFG_Get(option, dflt, value);
  if(CFG_OK == rc || CFG_DEFAULT == rc) {
    return PWR_TRUE;
  }
  else {
    return PWR_FALSE;
  }
}

#define STREQ(s1,s2) (strcmp((s1),(s2)) == 0)

/* Initialize the power switch, and the communication paths to it. 
*/
enum PWR_boolean PWR_NW_RPC100S_init(PWR_result *s) {
  char sdflt[DFLT_TMP_BUFFER_SZ];
  char *svalue;
  long lvalue;
  int lock_result;
  CluCfg              *cfg = (CluCfg *)NULL;
  char                *file = NULL;

  if(!gbl_initialized) {
    /* Compile regular expressions. */
    if(!regex_allocated) {
      regcomp(&ok_regex,
              "[ \t\n]*OK[ \t\n]*",
              REG_EXTENDED);
      regcomp(&invalid_regex,
              "[ \t\n]*Invalid Entry[ \t\n]*",
              REG_EXTENDED);
      regex_allocated=1;
    }
    /* Check to see if a device has been set, possibly from the
       command line. */
    snprintf(sdflt, DFLT_TMP_BUFFER_SZ, "%s", DFLT_NO_DEVICE);
    PWR_get("power%device", sdflt, &svalue);
    if( STREQ(svalue,DFLT_NO_DEVICE)) {
      /* Read configuration variables. */
      cfg = get_clu_cfg(file);
      if (cfg != NULL) {
	strcpy(serial_device_file, cfg->nodes[cfg->lid].powerSerialPort);
	free(cfg);
      }
      else {
        strcpy(serial_device_file, DFLT_NO_DEVICE);
      }
    } else {
      strcpy(serial_device_file, svalue);
    }
    snprintf(sdflt, DFLT_TMP_BUFFER_SZ, "%ld", init_timeout);
    PWR_get("power%init_timeout", sdflt, &svalue);
    sscanf(svalue, "%ld", &lvalue);
    init_timeout = lvalue;

    snprintf(sdflt, DFLT_TMP_BUFFER_SZ, "%ld", timeout);
    PWR_get("power%timeout", sdflt, &svalue);
    sscanf(svalue, "%ld", &lvalue);
    timeout = lvalue;
  
    if (!locked) {
      /* Create a lock file for the serial device. */
      if (strcmp(serial_device_file, DFLT_NO_DEVICE) == 0) {
	PR("serialPort not specified\n");
	SET(gbl_power_status, PWR_ERROR);
      }
      else if((lock_result = pwr_lock_serial_device(serial_device_file)) < 0) {
	PR("can't lock %s, lock result: %d\n", 
	   serial_device_file,
	   lock_result);

	SET(gbl_power_status, PWR_ERROR);
      }
      else {
	PR("locked %s\n", serial_device_file);
	locked = 1;
	/* Open and query the serial device. */
	if(open_serial(serial_device_file) < 0) {
	  SET(gbl_power_status,PWR_ERROR);
	  *s = gbl_power_status;
	  return PWR_FALSE;
	}
      }
    } else { /* locked */
      /* RPS-10 Note:
       * This situation is basically the following:  we already called
       * this initialization routine, but we were unable to contact the
       * power switch.  Thus, we have it locked, we need only cycle dtr
       * and hope that we can talk to it.
       */
      if (gbl_serial_fd > 0)
	serial_reset();
      else
	open_serial(serial_device_file);
    }
    gbl_power_status = send_command(RPC100S_STATUS_CMD, timeout);
  }
  else {
    gbl_power_status = send_command(RPC100S_STATUS_CMD, timeout);
  }
  if( ISSET(gbl_power_status, PWR_TIMEOUT) || ISSET(gbl_power_status, PWR_ERROR)) {
    gbl_initialized = 0;
    RESET(gbl_power_status, PWR_INIT);
    *s = gbl_power_status;
    /* It has been noticed that 'close()' will cause the program to hang
       try flushing all of the i/o to this FD
     */
    tcflush(gbl_serial_fd, TCIOFLUSH);    
    close(gbl_serial_fd);
    return PWR_FALSE;
  } else {
    gbl_initialized = 1;
    SET(gbl_power_status,   PWR_INIT);
    *s = gbl_power_status;
    return PWR_TRUE;
  }
}

/* Give back resources to the system.  This is only needed if the
   power switch is loaded occasionally in some long running program.
   For the use model scenarios under which this module was designed,
   there is not normally a need to call this function. 
*/
enum PWR_boolean PWR_NW_RPC100S_release(void) {
  if(gbl_initialized) {
    if(regex_allocated) {
      regfree(&ok_regex);
      regfree(&invalid_regex);
      regex_allocated = 0;
    }
    if(pwr_unlock_serial_device(serial_device_file) < 0 ) {
      PR("failed to unlock serial device\n");

    }
    /* Toggle dtr on the serial line.  This way the switch will know
       we are no longer talking to it. 
    */
    if(gbl_perform_reset) {
      serial_reset();
    }
    /* It has been noticed that 'close()' will cause the program to hang
       try flushing all of the i/o to this FD
     */
    tcflush(gbl_serial_fd, TCIOFLUSH);
    close(gbl_serial_fd);
  }
  gbl_initialized = 0;
  return 1;
}

/* Query the switch to determine its current status, and its readiness
   to carry out additional commands.

   08/30/00 -EZA
   The original intention was to return a PWR_result that
   contains flags that tell whether the switch is open or closed,
   but with the RPC100S, it turns out that we can't really tell.
   The side effect of probing status is that power gets turned back on! 
*/
PWR_result PWR_NW_RPC100S_status(void) {

  PR("Probing NW RPC100S status.\n");
  send_command(RPC100S_STATUS_CMD, timeout);

  /* Lie and just say that the power is now on...
     set the 'closed' bit in the status
  */
  if (!ISSET(gbl_power_status, PWR_ERROR)) {
    SET(gbl_power_status, PWR_SWITCH);
  }  
  return gbl_power_status;
}

/* Send a command to the switch that will cause the system powered by
   the switch to reboot.  Normally, this means that the switch will be
   opened for some period of time, after which it will be closed
   again.  
*/
PWR_result PWR_NW_RPC100S_reboot(void) {
  PR("Turning NW RPC100S OFF then ON.\n");
  send_command(RPC100S_REBOOT_CMD, timeout);
  /* This is a case where power could be turned off. */
  if (!ISSET(gbl_power_status, PWR_ERROR)) {
    RESET(gbl_power_status, PWR_SWITCH);
  }
  return gbl_power_status;
}

/* Send a command to the switch that will cause the switch to open,
   securing power to the effected system. 
*/
PWR_result PWR_NW_RPC100S_off(void) {
  PR("Turning NW RPC100S OFF.\n");
  send_command(RPC100S_OFF_CMD, timeout);
  
  /* This is a case where power could be turned off. */
  if (!ISSET(gbl_power_status, PWR_ERROR)) {
    RESET(gbl_power_status, PWR_SWITCH);
  }
  
  return  gbl_power_status;
}

/* Send a command to the switch that will cause the switch to close,
   supplying power to eht effected system. 
*/
PWR_result PWR_NW_RPC100S_on(void) {
  PR("Turning NW RPC100S ON.\n");
  
  send_command(RPC100S_ON_CMD, timeout);
  
  /* the power is now on... set the 'closed' bit in the status */
  if (!ISSET(gbl_power_status, PWR_ERROR)) {
    SET(gbl_power_status, PWR_SWITCH);
  }  
  return gbl_power_status;
}

/* Emacs Hints
   c-basic-offset: 2
*/
