/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/* 
   getaline.c

   Provides completion for cluadmin's use of getline.c.

   author: Ron Lawrence <lawrence@missioncriticallinux.com>
   
*/

#include "getline.h"
#include <limits.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

static const char *version __attribute__ ((unused)) = "$Id:";
static char *commands[] = {   
  "apropos",             "clear",          "exit", 
  "help",                     "cluster status", 
  "cluster monitor",          "cluster loglevel", 
  "cluster reload",           "cluster name", 
  "cluster edit",             "cluster backup", 
  "cluster restore",          "cluster saveas", 
  "cluster restorefrom",      "service add", 
  "service show state",       "service show config", 
  "service show services", 
  "service modify",           "service disable", 
  "service enable",           "service delete",

  "help apropos",        "help clear",     "help exit", 
  "help help",                "help cluster status", 
  "help cluster monitor",     "help cluster loglevel", 
  "help cluster reload",      "help cluster name", 
  "help cluster edit",        "help cluster backup", 
  "help cluster restore",     "help cluster saveas", 
  "help cluster restorefrom", "help service add", 
  "help service show state",  "help service show config", 
  "help service show services", 
  "help service modify",      "help service disable", 
  "help service enable",      "help service delete", 0};

#define COMMANDS 44
#define ON 1
#define OFF 0
#define BUFFER_SIZE 1024

int getaline_ignore_tabs(char *buf, int, int*);
void detach_completer(void);
void attach_completer(void);
void load_history(void);
void save_history(void);
int getScreenSize(char *buffer, int size);
void getaline(char*, char*, unsigned int);

static char* prompt_ptr = 0;
static int recording_history = ON;

/* Compare two strings and return the length of the part of the
   strings starting at the beginning that are both the same. */
static int common_prefix_length(char *s1, 
                                char *s2) {
  unsigned int i;
  int count;

  for(i=0, count=0; i<strlen(s1); ++i) {
    if('\0' == s2[i]) 
      break;
    if(s1[i] == s2[i]) 
      count++;
    else
      break;
  }
  return count;
}

/* Grovel over a list of strings looking for the shortest common
   prefix in the list. Copy the LCP into the provided buffer. */
static void shortest_common_prefix(char** strings, 
                                   int    count, 
                                   char  *buffer) {
  int shortest_so_far = 0;
  int i;
  int shortest=INT_MAX;
  int len;

  if(0==count) 
    return;
  for(i=0; i<count; ++i) {
    len = common_prefix_length(strings[i], strings[shortest_so_far]);
    if(len < shortest) {
      shortest = len;
      shortest_so_far = i;
    }
  }
  strncpy(buffer, strings[shortest_so_far], shortest);   
  buffer[shortest] = '\0';
}

/* Complete the users input. */
static int getaline_tab_action(char *buf, 
                               int prompt_width __attribute__((unused)), 
                               int *cursor_loc) {

  unsigned int commands_list_length = COMMANDS;
  static int longest_command = 0;
  static unsigned int last_length = 0;
  static int last_code   = 0;
  int code=0;
  char **pointers = (char**) alloca(sizeof(char*)*commands_list_length);
  char *squeezed =  (char*)  alloca(sizeof(char)*(strlen(buf)+1));
  char *common_prefix = NULL;
  int flag;
  unsigned int i, j, count;

  memset(pointers, 0, commands_list_length);
  memset(squeezed, 0, strlen(buf));
  if(0 == longest_command) {
    for(i=0; i<commands_list_length; ++i) {
      int len = strlen(commands[i]);
      if(len > longest_command)
        longest_command = len;
    }
  }
  /* squeeze extra spaces out of the input buffer. */
  for(i=0, j=0; i<strlen(buf); ++i) { 
    if(buf[i] != ' ' && buf[i] != '\t') {
      squeezed[j++] = buf[i];
      flag=0;
    }
    else if(!flag) {
      squeezed[j++] = buf[i];
      flag=1;
    }
  }
  squeezed[i]='\0';
  /* Get a list of potential matches. */
  for(i=0, j=0, count=0; i<commands_list_length; ++i) {
    if(!strncmp(commands[i], squeezed, strlen(squeezed))) {
      pointers[j++] = commands[i];
      count++;
    }
  }
  for(i=0; i< strlen(buf); ++i)
    code += (int)buf[i];
  /* Has the user hit tab twice? */
  if(last_length == strlen(buf) &&
     code   == last_code &&
     count > 1 ) {
    /* Print the list of potential matches. */
    printf("\n");
    for(i=0; i<count; ++i)
      printf("%s\n", pointers[i]);
    printf("%s%s", prompt_ptr, buf);
    fflush(stdout);
    return 1;
  }
  /* Find the shortest prefix that all of the candidates share in
     common. */
  common_prefix = (char*) alloca(sizeof(char)*longest_command+1);
  memset(common_prefix, 0, longest_command+1);
  shortest_common_prefix(pointers,  count, common_prefix);

  /* Update the input buffer. */
  if(strlen(common_prefix) > strlen(buf)) {
    strncpy(buf, common_prefix, strlen(common_prefix));
    buf[strlen(common_prefix)] = '\0';
    *cursor_loc = strlen(common_prefix);
  }
  last_length = strlen(buf);
  last_code = 0;
  for(i=0; i< strlen(buf); ++i)
    last_code += (int)buf[i];
  return 1;
}

/* A function that doesn't attempt to complete when the user sends a
   tab.  In fact, it doesn't even insert the tab into the buffer. */
int getaline_ignore_tabs(char *buf __attribute__((unused)), 
                             int prompt_width __attribute__((unused)), 
                             int *cursor_loc __attribute__((unused))) {
  return 1;
}

/* Set the completion function to ignore tabs. */
void detach_completer(void) {
  gl_tab_hook = getaline_ignore_tabs;
  recording_history = OFF;
  gl_do_history(OFF);
}

/* Set the usual completion function. */
void attach_completer(void) {
  gl_tab_hook = getaline_tab_action;
  recording_history = ON;
  gl_do_history(ON);
}

#define HISTORY_FILENAME "/.cluadmin_history"

/* Attempt to get the user's home directory from the environment.
   Check the environment variables HOME, and home, in that order.  If
   neither variable is set, return 0, else copy the returned string
   into where, or up to max_size characters of it.  then return 1. */
static int user_home_directory(char *where, int max_size) {
  char *home_dir = getenv("HOME");
  if(NULL == home_dir)
    home_dir = getenv("home");
  if(NULL == home_dir)
    return 0;                     
  strncpy(where, home_dir, max_size);
  where[strlen(home_dir)] = '\0';
  return 1;
}

/* Open and read a file containing history of commands typed in prior
   sessions.  Put the commands into the getline history list.
   Intended for use at startup time. */
void load_history(void) {
  char home_dir[BUFFER_SIZE];
  char buffer[BUFFER_SIZE];
  FILE *f;

  if(!user_home_directory(home_dir,BUFFER_SIZE)) {
    return;                     /* Couldn't find user's home directory
                                   in environment, don't update the
                                   history list. */
  }
  strncpy(buffer, home_dir, BUFFER_SIZE);
  strncat(buffer, HISTORY_FILENAME, strlen(HISTORY_FILENAME));
  buffer[strlen(home_dir)+strlen(HISTORY_FILENAME)] = '\0';
  f = fopen(buffer,"r");
  if(NULL != f) {
    buffer[0] = '\0';
    while(!feof(f)) {
      fgets(buffer, BUFFER_SIZE, f);
      gl_histadd(buffer);
      buffer[0] = '\0';
    }
    fclose(f);
  }
}

#define MAX(x,y) ((x)>(y)?(x):(y))
#define MIN(x,y) ((x)<(y)?(x):(y))
/* Maximum number of history entries is defined as a static var in
   getline.c.  Must be kept in sync manually. */
#define HIST_MAX 100

/* Write out the top-level command history to a history file in the
   user's home directory.  The name of the file could be configurable,
   but it is not.*/
void save_history(void) {
  char home_dir[BUFFER_SIZE];
  char buffer[BUFFER_SIZE];
  FILE *f;

  if(!user_home_directory(home_dir,BUFFER_SIZE)) {
    return;                     /* Couldn't find user's home directory
                                   in environment, don't try to save
                                   any history. */
  }
  strncpy(buffer, home_dir, BUFFER_SIZE);
  strncat(buffer, HISTORY_FILENAME, 
          MIN(BUFFER_SIZE-strlen(home_dir),strlen(HISTORY_FILENAME)));
  buffer[strlen(home_dir) + strlen(HISTORY_FILENAME)] = '\0';
  f = fopen(buffer,"w");
  if(NULL != f) {
    int i;
    for(i = 0; i < HIST_MAX; ++i) {
      if(hist_get(i, buffer, BUFFER_SIZE)) {
        fputs(buffer, f);
        fputc('\n', f);
      }
      else {
        break;
      }
    }
    fclose(f);
  }
}

/* A wrapper around getline.  Manages the initial assignment of the
   tabbing function to be the completion function.  Also, ensures that
   the result from getline is copied out to the buffer provided from
   tcl. Stores a command into history. */
void getaline(char* prompt, 
              char* buffer, 
              unsigned int size) {
  static int initialized = 0;
  char* response;

  if(!initialized) {
    gl_tab_hook = getaline_tab_action;
    recording_history = ON;
    initialized=1;
    load_history();
  }
  prompt_ptr = prompt;
  response = getline(prompt);
  response[strlen(response)-1] = '\0'; /* Remove the trailing newline. */
  strncpy(buffer, response, size); /* Copy the response into the provided buffer. */

  if( ON == recording_history ) 
    gl_histadd(response);
  return;
}
