/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
#ifndef POWER_H
#define POWER_H
#ifdef __cplusplus
extern "C" {
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif                   

/* power.h */

/* An interface to routines that allow for high level control of the
   power management functions of the missioncritical highly available
   cluster system. 

   Author: Ron Lawrence <lawrence@missioncriticallinux.com>
   Version: $Revision: 1.7 $
*/

/* Assumptions:

   1. It is possible to query the switch to determine its status in
   detail.

   2. There will only be one switch managed by this interface.  That
   switch will control power to one machine.

   4. Only one process will ever attempt to communicate with the
   switch at any given time, i.e. there are no contention issues
   related to which process controls the switch.

   5. Nothing is assumed in the interface about the nature of the
   communication channel to the switch.  The implementation hides the
   details of the communication channel.

   Discussion:

   1. All commands are synchronous, i.e. blocking.

   2. Initialization.  Assuming that processes will not contend for
   access to the remote switch, and that there is only one such switch
   per machine, it is assumed that initialization involves opening the
   serial device, creating a lock file for the device, setting
   parameters on the serial device, and sending and receiving an
   initialization string to/from the switch.

   4. Status and results.  Certain functions return a PWR_result,
   which is a set of flag bits whose positions are taken from the
   PWR_result_codes enumeration.  For each flag position, the status
   is to be taken to be true, available, or on, as appropriate, for
   that flag if its bit value is one, otherwise the status is to be
   taken to be false, unavailable, or off.

   5. Sequencing. After PWR_init has been called and returned
   PWR_TRUE, any of the other functions may be called in any order.

   Redundant calls to functions will return the status after the call
   regardless of whether the status of the switch changed as a result
   of the call.  For example, a call to PWR_off, followed after some
   interval by a call to PWR_off will return true in each case if
   after the call, the switch is in the open condition.

   6. Timeouts.  Synchronous calls block until they fail, succeed, or
   timeout.  Currently, the default timeout for initialization is 10
   sec.; the default timeout for other commands is 1 sec.  Timeouts
   may be altered through configuration file entries.

   The RPS-10 Switch

   The RPS-10 can be controlled over a serial line.  It responds with
   a string "RPS-10 ready\n", when the corresponding serial port file,
   e.g. /dev/ttyS0 is first opened.  There is a delay before the
   initial response string is displayed.  It is on the order of 10
   sec.  Normally the switch doesn't respond when the serial port is
   opened, and must be queried to see if it is alive.

   When no power is supplied to the switch, it does not respond to
   commands over the serial port.  There is no other indication.  If
   there is no response to a command or query then this can be taken
   to be a possible indication that power is not supplied to the
   switch.

   When power is removed from the switch, the condition of the switch
   after power is re-suppled is dependent upon the position of a dip
   switch.  Depending on the position of the switch, the device will
   power on in the condition that it was in when powered off, or will
   power on closed.  For our purposes the dip switch will be
   positioned so that the switch is closed, i.e. supplying power to
   its load, when power is initially supplied to the switch.

   When the serial line is disconnected, the switch remains in the
   position that it was in at the time of the disconnect.  The switch
   remains in that same position after the serial line is reconnected.

   The power toggle delay can be configured to be 5 sec. or 10 sec.

   After a command is sent to the switch, the switch will respond with
   the current switch position, i.e. the position after the command
   has completed.  Then the switch will send "Complete\n" through the
   serial port.

   Since the only indication of failure is that the switch does not
   respond in a timely way, timeouts must be supported in software.

*/

typedef int PWR_result;
enum PWR_result_bits { 
  PWR_SWITCH      = 1,          /* Switch is closed=1/open=0. */
  PWR_ERROR       = 2,          /* A command has put the interface
                                   into an error condition.  */
  PWR_TIMEOUT     = 4,          /* The most recent command sent to the
                                   switch timed out without a
                                   response. */
  PWR_INIT        = 8           /* 1 if interface has been initialized */
};

enum PWR_boolean { PWR_FALSE, PWR_TRUE };

/* Configure the power interface. 

   Currently there are three options recognized. 

   1. power.device -- The pathname of the serial port character
   special device file.

   2. power.init_timeout -- The timeout, in sec., fof initialization.

   3. power.timeout -- The timeout, in sec., for non-initialization
   calls through the interface.

   PWR_configure can be called with: 1. The name of a configuration
   file to read for parameters.  2. NULL, or 3. a null string.  */

enum PWR_boolean PWR_configure(char *config_file_name);

/* These are wrappers around calls to CFG_Get, and CFG_set.  Should
   they be retained? */

enum PWR_boolean PWR_set(char* option, char* value);
enum PWR_boolean PWR_get(char* option, char *dflt, char** value);

/* Initialize the power switch, and the communication paths to it. */

enum PWR_boolean PWR_init(PWR_result *status);

/* Release resources associated with the module, and close the serial
   device file.  PWR_release should be called when the connection to
   the power switch is no longer needed so that the serial port lock
   file can be released. */

enum PWR_boolean PWR_release(void);


/* Query the switch to determine its current status, and its readiness
   to carry out additional commands.  Returns a PWR_result that
   contains flags that indicate whether the switch is open or closed,
   and whether the last command to the switch timed-out. */

PWR_result  PWR_status(void);

/* Send a command to the switch that will cause the supplied system to
   reboot.  Normally, this means that the switch will be opened for
   some period of time, after which it will be closed again.  If the
   switch is already open when the reboot command is issued, the
   switch remains open for 5 or ten seconds, then closes. */

PWR_result  PWR_reboot(void);

/* Send a command to the switch that will cause the switch to open,
   removing power from the effected system. */

PWR_result  PWR_off(void);

/* Send a command to the switch that will cause the switch to close,
   supplying power to the effected system. */

PWR_result  PWR_on(void);

/* Returns a number representing the type of power switch in use. */

int PWR_type(void);

/*
 * Supported power switch types.  These are the strings which get
 * specified in configuration files.
 * Following the string names for the switch types are the corresponding
 * numeric constants.
 */
#define PWR_TYPE_RPS10 "RPS10"
#define PWR_TYPE_APC   "APC"
#define PWR_TYPE_NONE  "NONE"
#define PWR_TYPE_NW_RPC100S   "NW_RPC100S"

#define SWT_UNSPECIFIED 0
#define SWT_RPS10       1
#define SWT_APC         2
#define SWT_NW_RPC100S  3
#define SWT_NONE        4

#ifdef __cplusplus
}
#endif
#endif

