/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/*
 *  $Revision: 1.6 $
 *  Copyright (C) 2000 Mission Critical Linux, LLC
 *  author: Jeff Moyer <moyer@missioncriticallinux.com>
 */

#ifndef __CLUSTER_LOG_H
#define __CLUSTER_LOG_H

#ifdef __cplusplus
extern "C" {
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <msgsvc.h>

#define LOG_MESSAGE           0x0
#define LOG_CMD_STATUS        0x1
#define LOGGER_TERM           0x2

#define LOGLEVEL_DFLT         LOG_WARNING
#define MAX_LOGMSG_LEN        255

/*
 *  Log level.  Based on values found in /usr/include/sys/syslog.h.
 *  This has to be set by the process using the logging library.  The
 *  default it LOG_WARNING, as dictated by LOGLEVEL_DFLT.
 */
extern int loglevel;

struct log_msg {
    generic_msg_hdr hdr;
    union {
	int        cmd_arg;
	struct msg {
	    int    severity;
	    int    dosyslog;
	    size_t count;
	}m;
    } data;
};

/*
 * int clu_set_loglevel(int severity)
 *
 * DESCRIPTION
 *   Set the logging level for this daemon.  This is not a 
 *   system-wide setting.
 *
 * ARGUMENTS
 *   severity  Severity as documented in sys/syslog.h (i.e. LOG_ERROR)
 *
 * RETURN VALUES
 *   On success, the previous loglevel is returned.  On error -1 is returned.
 *
 * NOTES
 *   The only way of generating errors for this call is to give a negative
 *   value for severity.  Currently, syslog lists severities up to 8, but
 *   I see no reason for this restriction if, in the future, we decided to
 *   add more levels.  Thus, any number up to MAXINT will be supported.
 */
int clu_set_loglevel(int severity);

/*
 * int clu_get_loglevel(void)
 *
 * DESCRIPTION
 *   Get the current logging level.
 *
 * ARGUMENTS
 *   none
 *
 * RETURN VALUES
 *   The current logging level is returned.
 */
int clu_get_loglevel(void);

/*
 * DESCRIPTION
 *   Cluster logging facility.  This is the actual function that does the
 *   logging.  No one should call this, you should call the wrappers provided.
 *   i.e. clulog and clulog_and_print.
 */
int do_clulog(int severity, int write_to_cons, pid_t pid, 
	      char *prog, const char *fmt, ...);
/*
 * int clulog(int severity, const char *fmt, ...)
 *
 * DESCRIPTION
 *   Cluster logging facility.  This is a library routine which sets up
 *   a connection to the logging daemon and sends the supplied parameters to
 *   said daemon.  If the supplied severity is numerically larger than the
 *   current loglevel, the message is never sent to the logging daemon.
 *
 * ARGUMENTS
 *   severity  Severity as documented in sys/syslog.h (i.e. LOG_ERROR)
 *   fmt       Format string as used with printf.
 *
 * RETURN VALUES
 *   On success, 0 is returned.  On error, -1 is returned.
 *
 * NOTES
 *   Inability to contact the logging daemon is the only source of error
 *   for this function.  Thus, it would behoove you to try a clulog before
 *   daemonizing your process.  If it fails, print a message to stderr
 *   explaining that the cluster logging daemon should probably be started.
 *   If you really want your message to be heard by someone, use
 *   clulog_and_print().
 */
#define clulog(x,fmt,args...)    do_clulog(x,0,0,NULL,fmt,##args)
#define clulog_pid(x,pid,prog,fmt,args...) do_clulog(x,0,pid,prog,fmt,##args)

/*
 * int clulog_and_print(int severity, int write_to_cons, const char *fmt, ...)
 *
 * DESCRIPTION
 *   Cluster logging facility.  This is a library routine which sets up
 *   a connection to the loggin daemon and sends the supplied parameters to
 *   said daemon.  If the supplied severity is numerically larger than the
 *   current loglevel, the message is never sent to the logging daemon.
 *
 * ARGUMENTS
 *   severity       Severity as documented in sys/syslog.h (i.e. LOG_ERROR)
 *   write_to_cons  set if the caller wants the message to go to the console
 *   fmt            Format string as used with printf.
 *
 * RETURN VALUES
 *   On success, 0 is returned.  On error, -1 is returned.
 */
#define clulog_and_print(x,fmt,args...)   do_clulog(x,1,0,NULL,fmt,##args)


/*
 * void clulog_close(void)
 *
 * DESCRIPTION
 *   This function closes the TCP connection to the logger daemon.  This is
 *   necessary to avoid pesky TIME_WAIT states upon stopping the cluster 
 *   software.
 *
 * ARGUMENTS
 *   none
 *
 * RETURN VALUES
 *   This function does not return anything.
 */
void clulog_close(void);


#ifdef __cplusplus
}
#endif
#endif /* __CLUSTER_LOG_H */
/*
 * Local variables:
 *  c-basic-offset: 4
 *  c-indent-level: 4
 *  tab-width: 8
 * End:
 */
