/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2006 by the KFTPGrabber developers
 * Copyright (C) 2003-2006 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#include "widgetlister.h"

#include <klocale.h>
#include <kguiitem.h>
#include <kpushbutton.h>
#include <kdialog.h>

#include <qpushbutton.h>
#include <qlayout.h>
#include <qhbox.h>

namespace KFTPWidgets {

WidgetLister::WidgetLister(QWidget *parent, int minWidgets, int maxWidgets)
  : QWidget(parent)
{
  m_widgetList.setAutoDelete(true);

  m_minWidgets = QMAX(minWidgets, 0);
  m_maxWidgets = QMAX(maxWidgets, m_minWidgets + 1);

  // The button box
  m_layout = new QVBoxLayout(this, 0, 4);
  m_buttonBox = new QHBox(this);
  m_buttonBox->setSpacing(KDialog::spacingHint());
  m_layout->addWidget(m_buttonBox);

  m_buttonMore = new KPushButton(KGuiItem(i18n("more widgets", "More"), "button_more"), m_buttonBox);
  m_buttonBox->setStretchFactor(m_buttonMore, 0);

  m_buttonFewer = new KPushButton(KGuiItem(i18n("fewer widgets", "Fewer"), "button_fewer"), m_buttonBox);
  m_buttonBox->setStretchFactor(m_buttonFewer, 0);

  QWidget *spacer = new QWidget(m_buttonBox);
  m_buttonBox->setStretchFactor(spacer, 1);

  m_buttonClear = new KPushButton(KGuiItem(i18n("clear widgets", "Clear"), "locationbar_erase"), m_buttonBox);
  m_buttonBox->setStretchFactor(m_buttonClear, 0);

  // Connect signals
  connect(m_buttonMore, SIGNAL(clicked()), this, SLOT(slotMore()));
  connect(m_buttonFewer, SIGNAL(clicked()), this, SLOT(slotFewer()));
  connect(m_buttonClear, SIGNAL(clicked()), this, SLOT(slotClear()));

  enableControls();
}

WidgetLister::~WidgetLister()
{
}

void WidgetLister::slotMore()
{
  addWidget();
  enableControls();
}

void WidgetLister::slotFewer()
{
  removeWidget();
  enableControls();
}

void WidgetLister::clear()
{
  setNumberShown(m_minWidgets);

  // Clear remaining widgets
  QPtrListIterator<QWidget> i(m_widgetList);
  for (i.toFirst(); i.current(); ++i)
    clearWidget((*i));

  enableControls();
  emit clearWidgets();
}

void WidgetLister::slotClear()
{
  clear();
}

void WidgetLister::addWidget(QWidget *widget)
{
  if (!widget)
    widget = createWidget(this);

  m_layout->insertWidget(m_layout->findWidget(m_buttonBox), widget);
  m_widgetList.append(widget);
  widget->show();
  
  enableControls();
  emit widgetAdded(widget);
}

void WidgetLister::removeWidget()
{
  m_widgetList.removeLast();
  
  enableControls();
  emit widgetRemoved();
}

void WidgetLister::clearWidget(QWidget *widget)
{
  Q_UNUSED(widget)
}

QWidget *WidgetLister::createWidget(QWidget* parent)
{
  return new QWidget(parent);
}

void WidgetLister::setNumberShown(int number)
{
  int superfluousWidgets = QMAX((int) m_widgetList.count() - number, 0);
  int missingWidgets = QMAX(number - (int) m_widgetList.count(), 0);

  // Remove superfluous widgets
  for (; superfluousWidgets; superfluousWidgets--)
    removeWidget();

  // Add missing widgets
  for (; missingWidgets; missingWidgets--)
    addWidget();
}

void WidgetLister::enableControls()
{
  int count = m_widgetList.count();
  bool isMaxWidgets = (count >= m_maxWidgets);
  bool isMinWidgets = (count <= m_minWidgets);

  m_buttonMore->setEnabled(!isMaxWidgets);
  m_buttonFewer->setEnabled(!isMinWidgets);
}

}
