/* This file is a whole module unto itself.
 * It's like an "object", that will store up to MAXTRANS different kanji
 *  translations.
 *    (or at least pointers thereof)
 *
 * User can pick whether these translations will be displayed in terms
 * of kanji, kana, or english.
 * Clicking on one will set the search window to display that entry.
 *
 * Basic functions from the outside:
 * MakeMulti();		[create X Widget]
 * void ShowMulti(){	[ make window visible ]
 * AddMultiTranslation(TRANSLATION); [ add to current list]
 * ClearAllMulti();
 * int getMultiMax(); [ return MAXTRANS define]
 * int getMultiCount(); [ return number of translations in current list]
 *
 *
 * And one PRIVATE function that is public because of callbacks:
 *   multicallback(int)
 */

#include <Xos.h>
#include <Intrinsic.h>
#include <StringDefs.h>
#include <Xlib.h>
#include <Xutil.h>
#include <Xfuncs.h>

#include <Shell.h>
#include <Xaw/Command.h>
#include <Xaw/Label.h>
#include <Xaw/Form.h>
#include <Xaw/Box.h>
#include <Xaw/Viewport.h>

#include <stdio.h> /* shouldnt be neccessary, but it is. sigh. */
#include "defs.h"
#include "externs.h"
#include "game.h"
#include "init.h"
#include "utils.h"

#include "searchwidgets.h"


/* This is purely internal, so not in multikanji.h */
/* This is the maximum translations we will store up and display */
#define MAXTRANS 200


static Widget      transwin[MAXTRANS];       /* user clicks to select translation*/
static TRANSLATION transptrs[MAXTRANS];	     /* info in each transwin */
static Widget      usefiletoggles[MAXTRANS]; /* is each one in usefile? */

static int transcount=0;
static int overflow=0; /* number of added translations over MAXTRANS */


Widget multikanab, multikanjib, multienglishb;
Widget multiclose;
 
static int displaymode=GUESS_ENGLISH;

static Widget multi_popup;
static Widget multi_form;
static Widget multi_status;


void setmultilabels();
void AddMultiTranslation(TRANSLATION nt);


/************************************************************/


void ShowMulti(){
	static int doneinit=0;
	XtPopup(multi_popup, XtGrabNone);
	if(doneinit==0){
		setup_deletewindow(multi_popup);
		doneinit=1;
	}
}

void HideMulti(){
	XtPopdown(multi_popup);
}

/* resizing isn't behaving. so I'll force it to never go too small.
 * Hopefully.
 * Always have string more than 30 chars long. space-pad the end.
 */
static void setmultistatus(char *string){

#define FORCEWIDE

#ifdef FORCEWIDE
	char statusbuff[50];
	int stringlen=strlen(string);
	if(stringlen>49)
		stringlen=49;
		
	if(multi_status==NULL)
		return;

	strcpy(statusbuff,
	       "                                            ");
	bcopy(string, statusbuff, stringlen);
	XtVaSetValues(multi_status, XtNlabel,statusbuff, NULL);
#else /* STUPID */
	XtVaSetValues(multi_status, XtNlabel,string, NULL);
#endif /* STUPID */
}


void ClearAllMulti(){
	int tc;

	overflow=0;

	for(tc=0; tc<transcount; tc++){

		XtDestroyWidget(transwin[tc]);
		XtDestroyWidget(usefiletoggles[tc]);
	}

	bzero(transptrs, sizeof(transptrs));
	bzero(transwin, sizeof(transwin));
	bzero(usefiletoggles, sizeof(usefiletoggles));

	transcount=0;
	setmultistatus("No matches");
}

/* This callback handles both "Change what mode you display", and
 * 'Hey, show me this one in detail'
 */
void multicallback(Widget button, XtPointer data, XtPointer call_data){

	int newmode=-1;
	int buttoncount;

	if(button==multiclose){
		HideMulti();
		return;
	}
	if(button==multikanab){
		newmode=GUESS_KANA;
	}
	if(button==multikanjib){
		newmode=GUESS_KANJI;
	}
	if(button==multienglishb){
		newmode=GUESS_ENGLISH;
	}

	if(newmode!=-1){
		/*
		 * disable this optimization, becuase we want
		 * romaji/kana to switch more easily
		if(newmode==displaymode)
			return;
		*/
		displaymode = newmode;
		setmultilabels();
		return;
	}

	for(buttoncount=0;buttoncount<transcount; buttoncount++){
		if(button== transwin[buttoncount]){
			printsearch(transptrs[buttoncount]);
			return;
		}
	}

	puts("Error: multicallback got click on unrecognized button?");
	
}

/* This handles a click on a usefile toggle button, 
 * in the multikanji popup.
 */
void multiUcallback(Widget button, XtPointer data, XtPointer call_data){
	Pixel fg,bg;
	int kindex=(int)data;

	SetUseKanji(kindex,!InUsefile(kindex));
	ReverseButton(button);

	/* Just in case this is for the currently displayed search line*/
	updatesearchlabels();

}

#define VWIDTH 350
#define VHEIGHT 300

/* called ONCE at start of program*/
void MakeMulti(){
	Widget multitop, buttonform;
	Widget viewport;
					       

	multi_popup=XtVaCreatePopupShell("multi_popup",
		transientShellWidgetClass,
		search_popup,
		XtNwidth, 350,
		XtNheight, 500,
		NULL);


	multitop=XtVaCreateManagedWidget("multitopform",
					     formWidgetClass,
					     multi_popup,
					     NULL);

	viewport=XtVaCreateManagedWidget("multiport",
					viewportWidgetClass,
					multitop,
					XtNforceBars, True,
					XtNallowHoriz,False,
					XtNallowVert,True,
					XtNwidth, VWIDTH,
					XtNheight, VHEIGHT,
					NULL);

	/* This is what holds the multiple buttons, that show
	 * all the different matches to a search
	 * YES, we DO need to seed this with an initial height, otherwize
	 * we get irritating behavior the first time we are used.
	 */
	multi_form=XtVaCreateManagedWidget("multikanjiform",
					formWidgetClass,
					/*boxWidgetClass,*/
					viewport,
					     NULL);

	multi_status=XtVaCreateManagedWidget("multistatus",
			labelWidgetClass,
			multitop,
			XtNfromVert, viewport,
			XtNlabel, "Status  ",
			XtNwidth, VWIDTH,
			NULL);

	buttonform=XtVaCreateManagedWidget("multibform",
			formWidgetClass, multitop,
			XtNfromVert, multi_status,
			XtNresize, False,
			NULL);

	multikanjib=XtVaCreateManagedWidget("multikanjib",
			commandWidgetClass,
			buttonform,
			XtNshapeStyle,XawShapeEllipse,
			XtNlabel,"Kanji",
			XtNresize, False,
			NULL);
	multienglishb=XtVaCreateManagedWidget("multienglishb",
			commandWidgetClass,
			buttonform,
			XtNshapeStyle,XawShapeEllipse,
			XtNfromHoriz, multikanjib,
			XtNlabel,"English",
			XtNresize, False,
			NULL);
	multikanab=XtVaCreateManagedWidget("multikanab",
			commandWidgetClass,
			buttonform,
			XtNshapeStyle,XawShapeEllipse,
			XtNfromHoriz, multienglishb,
			XtNlabel,"Kana meaning",
			XtNresize, False,
			NULL);


	multiclose=XtVaCreateManagedWidget("multiclose",
			commandWidgetClass,
			multitop,
			XtNfromVert, multi_status,
			XtNfromHoriz, buttonform,
			XtNlabel,"Close",
			XtNhorizDistance, 40,
			XtNvertDistance, 20,
			XtNshapeStyle,XawShapeRoundedRectangle,
			XtNcornerRoundPercent,10,
			XtNborderWidth, 2,
			NULL);
	XtAddCallback(multiclose, XtNcallback, multicallback, NULL);

	ClearAllMulti();

	XtAddCallback(multikanab, XtNcallback, multicallback, NULL);
	XtAddCallback(multikanjib, XtNcallback, multicallback, NULL);
	XtAddCallback(multienglishb, XtNcallback, multicallback, NULL);


	/* HAVE TO DO THIS NOW, otherwise layout doesn't happen the first
	 * time
	 */

	XtRealizeWidget(multi_popup);
}


/* run through all the widgets we have, and set labels appropriately */
/* kinda like printallchoices() */
/* I just LOOOOVE code reuse :-) */
void setmultilabels(){
	int lc;
	for(lc=0;lc<transcount;lc++){
		switch(displaymode){
		   case GUESS_ENGLISH:
			setenglabel(transwin[lc], transptrs[lc]);
			break;
		   case GUESS_KANA:
			setkanalabel(transwin[lc], transptrs[lc]);
			break;
		   case GUESS_KANJI:
			setkanjilabel(transwin[lc], transptrs[lc]);
			break;
		}
	}
}


/* We ignore attempts to add more than our MAXTRANS value */
void AddMultiTranslation(TRANSLATION nt){
	int kindex;

	Widget newbutton,newu;
	Widget prevbutton=NULL;
	int formwidth=GetWidgetWidth(multi_popup);
	char statusprompt[20];
	char buttname[10],uname[10];

	kindex=trans_to_index(nt);

	sprintf(buttname, "multi%x\n", (unsigned int)nt);
	sprintf(uname, "multiU%x\n", (unsigned int)nt);

	if(formwidth>50)
		formwidth-=50;
		

	if(transcount>=MAXTRANS){
		char msgbuff[100];
		overflow++;
		
		sprintf(msgbuff, "(only showing %d of %d)",
			MAXTRANS, MAXTRANS+overflow);
		setmultistatus(msgbuff);
		return;
	}

	transptrs[transcount]=nt;

	if(transcount>0){
		prevbutton=transwin[transcount-1];
	}
	newbutton=XtVaCreateManagedWidget(buttname, commandWidgetClass,
			multi_form,
			XtNfromVert,prevbutton,
			XtNwidth, formwidth,
			XtNshapeStyle,XawShapeRoundedRectangle,
			XtNcornerRoundPercent,20,
			/* this is to get the height always right*/
			XtNencoding,XawTextEncodingChar2b,
			XtNfont,largekfont,

			/*XtNresizable, False,*/
			NULL);

	XtAddCallback(newbutton, XtNcallback, multicallback, NULL);
	transwin[transcount]=newbutton;

	newu=XtVaCreateManagedWidget(buttname, commandWidgetClass,
			multi_form,
			XtNfromVert,prevbutton,
			XtNfromHoriz,newbutton,
			XtNwidth, formwidth,
			/*
			XtNencoding,XawTextEncodingChar2b,
			XtNfont,largekfont,
			XtNlabel,Ulabel,
			*/
			XtNencoding,XawTextEncoding8bit,
			XtNfont,englishfont,
			XtNlabel,"u",
			NULL);

	XtAddCallback(newu, XtNcallback, multiUcallback, (XtPointer)kindex);

	if(InUsefile(kindex)){
		ReverseButton(newu);
	}

	usefiletoggles[transcount]=newu;

	switch(displaymode){
	   case GUESS_ENGLISH:
		setenglabel(transwin[transcount], transptrs[transcount]);
		break;
	   case GUESS_KANA:
		setkanalabel(transwin[transcount], transptrs[transcount]);
		break;
	   case GUESS_KANJI:
		setkanjilabel(transwin[transcount], transptrs[transcount]);
		break;
	}
	transcount++;

	sprintf(statusprompt,"%d matches",transcount);
	setmultistatus(statusprompt);

	XawFormDoLayout(multi_form, True);

}


int getMultiMax(){
	return MAXTRANS;
}

int getMultiCount(){
	return transcount;
}


