/***************************************************************************
 *   Copyright (C) 2004, 2005 Thomas Nagy                                  *
 *   tnagy2^8@yahoo.fr                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qpainter.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qmultilineedit.h>
#include <qwmatrix.h>

#include <kiconloader.h>
#include <kaboutdata.h>
#include <klocale.h>
#include <kinstance.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kfiledialog.h>
#include <kio/netaccess.h>
#include <kurl.h>
#include <krun.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <kdebug.h>
#include <klibloader.h>
#include <ktempfile.h>
#include <kparts/genericfactory.h>
#include <kmacroexpander.h>

#include "generatorwizard.h"

#include "aux.h"
#include "DDataItem.h"
#include "docsettingsdlg.h"
#include "DGuiView.h"
#include "DDataControl.h"
#include "DCanvasView.h"
#include "DTreeListView.h"
#include "DGenerator.h"

#include "KDissertPart.h"

typedef KParts::GenericFactory<KDissertPart> KDissertPartFactory;
K_EXPORT_COMPONENT_FACTORY( libkdissertpart, KDissertPartFactory );

KDissertPart::KDissertPart( QWidget *parentWidget, const char *widgetName, 
		QObject *parent, const char *name, const QStringList & /*args*/ )
: KParts::ReadWritePart(parent, name)
{
	m_caption = QString::null;
	m_lastid = DItem::NOITEM;

	// we need an instance
	setInstance( KDissertPartFactory::instance() );

	// add the translations
	KGlobal::locale()->insertCatalogue( "kdissert" );

	// create the data and the views
	m_data = new DDataControl();

	m_canvasview = new DCanvasView( parentWidget, widgetName );
	m_canvasview->setDataTree(m_data);

	m_treelistview = new DTreeListView( NULL, "tree view" );
	m_treelistview->setDataTree(m_data);

	m_canvasview->setFocusPolicy( QWidget::ClickFocus );
	setWidget( m_canvasview );

	// set our XML-UI resource file
	setXMLFile("kdissertpart.rc");

	// do this *after* calling setWidget
	KActionCollection *ac = actionCollection();
	//ac->setWidget( m_canvasview );

	new KAction(i18n("&Clear"), 0,  this, SLOT(clearDocument()), 
			ac, "kdissert_clear");
	new KAction(i18n("&Document properties..."), 0,  this, SLOT(docProperties()),
			ac, "kdissert_docproperties");
	new KAction(i18n("&Generate Documents..."), 0, this, SLOT(generateDocument()), 
			ac, "kdissert_generate");
	new KAction(i18n("&Create Picture..."), 0, this, SLOT(savePic()),
			ac, "kdissert_savepic");

	KRadioAction* select = new KRadioAction(i18n("&Select leaves"), "kdissert_point",
			"Ctrl+j", this, SLOT(setPointMode()), ac,
			"kdissert_point");
	KRadioAction* link = new KRadioAction(i18n("&Link leaves"), "kdissert_link",
			"Ctrl+k", this, SLOT(setLinkMode()), ac,
			"kdissert_link");
	KRadioAction* sort = new KRadioAction(i18n("S&ort subtrees"), "kdissert_sort",
			"Ctrl+l", this, SLOT(setSortMode()), ac,
			"kdissert_sort");
	KRadioAction* scroll = new KRadioAction(i18n("S&croll"), SmallIcon("move"),
			"Ctrl+m", this, SLOT(setScrollMode()), ac,
			"kdissert_scroll");

	select->setExclusiveGroup("hades");
	link->setExclusiveGroup("hades");
	sort->setExclusiveGroup("hades");
	scroll->setExclusiveGroup("hades");

	select->setChecked(true);

	KStdAction::zoomIn(this, SLOT(zoomIn()), ac, "kdissert_zoomin" );
	KStdAction::zoomOut(this, SLOT(zoomOut()), ac, "kdissert_zoomout" );
	//KStdAction::redisplay(this, SLOT(reloadLayout()), ac );
	
	KActionMenu *focusmenu = new KActionMenu(i18n("&Focus"), ac, "kdissert_focus");
	focusmenu->insert( new KAction(i18n("&Focus on root"), "Ctrl+h", m_canvasview, 
				SLOT(focusOnRoot()), ac, "kdissert_focusroot"));
	focusmenu->insert( new KAction(i18n("Focus on next root"), "Ctrl+,", m_canvasview, 
				SLOT(focusOnNextRoot()), ac, "kdissert_nextroot"));
	focusmenu->insert( new KAction(i18n("Go item up"), "Ctrl+Up", m_canvasview, 
				SLOT(selectObjUp()), ac, "kdissert_goup"));
	focusmenu->insert( new KAction(i18n("Go item down"), "Ctrl+Down", m_canvasview, 
				SLOT(selectObjDown()), ac, "kdissert_godown"));
	focusmenu->insert( new KAction(i18n("Cycle item next"), "Ctrl+Left", m_canvasview,
				SLOT(selectObjLeft()), ac, "kdissert_goleft"));
	focusmenu->insert( new KAction(i18n("Cycle item previous"), "Ctrl+Right", m_canvasview, 
				SLOT(selectObjRight()), ac, "kdissert_goright"));
	
	KActionMenu *movemenu = new KActionMenu(i18n("&Move selection"), ac, "kdissert_move");
	movemenu->insert( new KAction(i18n("Move objects up"), "Alt+Up", m_canvasview, 
				SLOT(moveSelObjectsUp()), ac, "kdissert_mvup"));
	movemenu->insert( new KAction(i18n("Move objects down"), "Alt+Down", m_canvasview,
				SLOT(moveSelObjectsDown()), ac, "kdissert_mvdown"));
	movemenu->insert( new KAction(i18n("Move objects left"), "Alt+Left", m_canvasview,
				SLOT(moveSelObjectsLeft()), ac, "kdissert_mvleft"));
	movemenu->insert( new KAction(i18n("Move objects right"), "Alt+Right", m_canvasview,
				SLOT(moveSelObjectsRight()), ac, "kdissert_mvright"));

	KActionMenu *addmenu = new KActionMenu(i18n("&Add objects"), ac, "kdissert_add");
	addmenu->insert( new KAction(i18n("Add a mindmap child"), "Insert", m_canvasview, 
				SLOT(addChild()), ac, "kdissert_childadd"));
	addmenu->insert( new KAction(i18n("Add a mindmap sibling"), "Return", m_canvasview,
			SLOT(addSibling()), ac, "kdissert_siblingadd"));


	new KAction(i18n("&Reorganize the map"), "Ctrl+g", m_canvasview, SLOT(tidyClean()), 
			ac, "kdissert_tidy");

	m_saveAction = KStdAction::save(this, SLOT(slotFileSave()), ac);
	KStdAction::saveAs(this, SLOT(slotFileSaveAs()), ac);
	KStdAction::spelling(m_data, SLOT(slotFileSpell()), ac);

	m_warning  = new KToggleAction(i18n("Warning"), "messagebox_warning", 0,
			this, SLOT(checkFlags()), ac, "flag_warning");
	m_good     = new KToggleAction(i18n("Good"), "bookmark", 0,
			this, SLOT(checkFlags()), ac, "flag_good");
	m_idea     = new KToggleAction(i18n("Idea"), "ktip", 0,
			this, SLOT(checkFlags()), ac, "flag_idea");
	m_work     = new KToggleAction(i18n("Work needed"), "package_settings", 0,
			this, SLOT(checkFlags()), ac, "flag_work");
	m_clarify  = new KToggleAction(i18n("Clarify"), "xmag", 0,
			this, SLOT(checkFlags()), ac, "flag_clarify");
	m_question = new KToggleAction(i18n("Question"), "help", 0,
			this, SLOT(checkFlags()), ac, "flag_question");
	m_trash    = new KToggleAction(i18n("Trash"), "trashcan_empty", 0,
			this, SLOT(checkFlags()), ac, "flag_trash");
	m_meeting  = new KToggleAction(i18n("Meeting"), "penguin", 0,
			this, SLOT(checkFlags()), ac, "flag_meeting");
	m_trouble  = new KToggleAction(i18n("Trouble there"), "ksplash", 0,
			this, SLOT(checkFlags()), ac, "flag_trouble");

	// enable the connections
	connect( m_data, SIGNAL(documentChanged(bool)),
			this, SLOT(setModified(bool)) );

	connect( m_data, SIGNAL(itemChanged(int)),
			this, SLOT(setItemChanged(int)) );

	connect( m_data, SIGNAL( itemSelected(int, DGuiView*) ),
			this, SLOT( itemChanged(int, DGuiView*) ) );

	// we are read-write by default
	setReadWrite(true);

	// we are not modified since we have not done anything yet
	setModified(false);
}

KDissertPart::~KDissertPart()
{
	//kdWarning()<<"KDissertPart::~KDissertPart"<<endl;
	actionCollection()->clear();
	//delete m_canvasview;
	delete m_treelistview;
	delete m_data;
}

KAboutData *KDissertPart::createAboutData()
{
	// The non-i18n name here must be the same as the directory in
	// which the part's rc file is installed
	KAboutData *aboutData = new KAboutData("kdissertpart", I18N_NOOP("KDissertPart"), "0.3.8");
	aboutData->addAuthor("Thomas Nagy", 0, "tnagy2^8@yahoo.fr");
	return aboutData;
}

void KDissertPart::slotFileSave()
{
	saveFile();
}

void KDissertPart::slotFileSaveAs()
{
	saveFileAs();
}

void KDissertPart::settingsChanged()
{
	m_data->updateSettings();
}

void KDissertPart::setReadWrite(bool rw)
{
	// notify the views of the read-write state
	m_canvasview->setReadWrite(rw);
	m_treelistview->setReadWrite(rw);

	ReadWritePart::setReadWrite(rw);
}

void KDissertPart::setModified(bool modified)
{
	// we either enable or disable it based on the current state
	if (modified) m_saveAction->setEnabled(true);
	else m_saveAction->setEnabled(false);

	// in any event, we want our parent to do it's thing
	ReadWritePart::setModified(modified);

	emit stateChanged();
}

void KDissertPart::slotFileOpen()
{
	KURL url = KFileDialog::getOpenURL(QString::null,
			i18n("*.kdi|kdissert project (*.kdi)") +"\n"+
			i18n("*.mm|Freemind files (*.mm)") +"\n"+
			i18n("*.kno|Knowit files (*.kno)") ,
			m_canvasview, i18n("Open a mindmap project"));
	if (url.isEmpty())
		return;

	openURL(url);
}

bool KDissertPart::openURL(KURL url)
{
	if (! url.isValid())
		return false;

	bool result = m_data->loadFromFile(url);
	if (result)
	{
		m_canvasview->focusOnRoot();
		setModified( false );
		m_url = url;
		emit accessed(url);

		setWindowCaption( url.prettyURL() );
	}
	return result;
}

bool KDissertPart::openFile()
{
	// TODO : never tried as a standalone kpart
	//    kdWarning()<<"KDissertPart::openFile"<<endl;

	// standard filedialog
	KURL url = KFileDialog::getOpenURL(QString::null,
			i18n("*.kdi|kdissert project (*.kdi)") +"\n"+
			i18n("*.mm|Freemind files (*.mm)") +"\n"+
			i18n("*.kno|Knowit files (*.kno)"),
			m_canvasview, i18n("Open a mindmap project"));
	if (url.isEmpty())
		return false;

	return openURL(url);
}

bool KDissertPart::saveFileAs()
{
	// this slot is called whenever the File->Save As menu is selected,
	m_url = KFileDialog::getSaveURL(QString::null, i18n("*.kdi|kdissert project (*.kdi)"), 
			m_canvasview, i18n("Save kdissert project"));
	if (!m_url.isEmpty() && m_url.isValid())
		return saveFile();

	return false;
}

bool KDissertPart::saveFile()
{
	// if we aren't read-write, return immediately
	if (isReadWrite() == false)
		return false;

	if ( m_url.isEmpty() || !m_url.isValid() )
	{
		return saveFileAs();
	}
	else
	{
		bool savestatus = m_data->saveToFile(m_url);
		if ( savestatus )
		{
			emit accessed(m_url);
			emit setStatusBarText(i18n("The project was saved")+" "+m_url.prettyURL());
			return savestatus;
		}
		else
		{
			return saveFileAs();
		}
	}
	return true;
}

void KDissertPart::clearDocument()
{
	if (!m_data)
		return;

	int result = KMessageBox::warningContinueCancel(
			m_canvasview,
			i18n("You are about to clear the entire document. Are you sure ?"),
			i18n("Clear the entire document"),
			i18n("Clear"));

	if (result == KMessageBox::Continue)
		m_data->clearDocument();
}

void KDissertPart::generateDocument()
{
	if (!m_data)
		return;

	if (!m_data->canGenerate())
	{
		KMessageBox::sorry(m_canvasview,
				i18n("Your document does not have any root connected to children\n"
					"kdissert will not generate any document"),
				i18n("Invalid document") );
		return;
	}

	QStringList lst = KGlobal::instance()->dirs()->findAllResources("module", "*kdiss*.la");
	//kdWarning()<<" *** kdissert generators found *** \n"<<lst.join("\n")<<endl;

	// load the libraries
	QValueList<DGenerator*> liblist;
	for (unsigned int i=0; i<lst.count(); i++)
	{
		QString libname = lst[i].section( '/', -1 );

		libname.truncate(libname.length()-3);
		//kdWarning()<<"loading : "<<libname<<endl;

		KLibFactory* factory = KLibLoader::self()->factory(libname);
		if (factory)
		{
			//kdWarning()<<"library successfully dlopened!!"<<endl;

			DGenerator* lib = static_cast<DGenerator*> (factory->create(this, "alpha", "omega"));
			//kdWarning()<<"library full name is  : "<<lib->fullName()<<endl;
			//kdWarning()<<"library group is : "<<lib->group()<<endl;
			liblist.push_back(lib);
		}
		else
		{
			kdWarning()<<"library "<<lst[i]<<" is invalid "<<KLibLoader::self()->lastErrorMessage()<<endl;
		}
	}

	if (liblist.count() > 0)
	{
		generatorwizard *wiz = new generatorwizard(this);
		wiz->setgeneratorlist(liblist);
		wiz->setgeneratordata(m_data);
		wiz->exec();

		delete wiz;
	}
	else
	{
		KMessageBox::sorry(m_canvasview, i18n("No templates were found, please check your installation"), 
				i18n("Templates are missing") );
	}

	// clean after use
	for (unsigned int i=0; i<liblist.count(); i++)
	{
		delete liblist[i];
	}
	KLibLoader::self()->cleanUp();
}

void KDissertPart::savePic()
{
	if (!m_data)
	{
		return;
	}

	KURL url = KFileDialog::getSaveURL(QString::null, i18n("*.png|Png files (*.png)"), 
			m_canvasview, i18n("Grab picture of mindmap"));

	if (url.isEmpty() || !url.isValid())
	{
		//kdWarning()<<"invalid url !"<<endl;
		return;
	}

	QCanvas * canvas = m_canvasview->canvas();

	QRect enclosingsize = m_canvasview->canvasSize();

	QPixmap pix(enclosingsize.width(), enclosingsize.height());
	QPainter p(&pix);
	p.translate(-enclosingsize.x(), -enclosingsize.y());
	canvas->drawArea( enclosingsize, &p );

	KTempFile tf;
	tf.close();
	tf.setAutoDelete(true);

	pix.save(tf.name(), "PNG");
	KIO::NetAccess::upload(tf.name(), url, NULL);
}

void KDissertPart::setLinkMode()
{
	m_canvasview->setActionType( DCanvasView::act_link );
}

void KDissertPart::setPointMode()
{
	m_canvasview->setActionType( DCanvasView::act_point );
}

void KDissertPart::setSortMode()
{
	m_canvasview->setActionType( DCanvasView::act_sort );
}

void KDissertPart::setScrollMode()
{
	m_canvasview->setActionType( DCanvasView::act_scroll );
}

void KDissertPart::itemChanged(int val, DGuiView*)
{
	m_lastid = val;
	DDataItem* item = (DDataItem*) m_data->Item(val);
	emit itemSelected( item );

	if (item)
	{
		m_warning->setEnabled(true);
		m_idea->setEnabled(true);
		m_good->setEnabled(true);
		m_work->setEnabled(true);
		m_clarify->setEnabled(true);
		m_question->setEnabled(true);
		m_trash->setEnabled(true);
		m_meeting->setEnabled(true);
		m_trouble->setEnabled(true);

		
		m_warning->setChecked( (bool) item->m_flags.contains(DDataControl::e_warning) );
		m_good->setChecked( (bool) item->m_flags.contains(DDataControl::e_good) );
		m_idea->setChecked( (bool) item->m_flags.contains(DDataControl::e_idea) );
		m_work->setChecked( (bool) item->m_flags.contains(DDataControl::e_work) );
		m_clarify->setChecked( (bool) item->m_flags.contains(DDataControl::e_clarify) );
		m_question->setChecked( (bool) item->m_flags.contains(DDataControl::e_question) );
		m_trash->setChecked( (bool) item->m_flags.contains(DDataControl::e_trash) );
		m_meeting->setChecked( (bool) item->m_flags.contains(DDataControl::e_meeting) );
	}
	else
	{
		m_warning->setEnabled(false);
		m_idea->setEnabled(false);
		m_good->setEnabled(false);
		m_work->setEnabled(false);
		m_clarify->setEnabled(false);
		m_question->setEnabled(false);
		m_trash->setEnabled(false);
		m_meeting->setEnabled(false);
		m_trouble->setEnabled(false);

		m_warning->setChecked(false);
		m_idea->setChecked(false);
		m_good->setChecked(false);
		m_work->setChecked(false);
		m_clarify->setChecked(false);
		m_question->setChecked(false);
		m_trash->setChecked(false);
		m_meeting->setChecked(false);
		m_trouble->setChecked(false);
	}
}

void KDissertPart::checkFlags()
{
	DDataItem* item = (DDataItem*) m_data->Item(m_lastid);
	if (!item) return;

	item->m_flags.clear();
	if (m_warning->isChecked()) item->m_flags.append( DDataControl::e_warning );
	if (m_good->isChecked()) item->m_flags.append( DDataControl::e_good );
	if (m_idea->isChecked()) item->m_flags.append( DDataControl::e_idea );
	if (m_work->isChecked()) item->m_flags.append( DDataControl::e_work );
	if (m_clarify->isChecked()) item->m_flags.append( DDataControl::e_clarify );
	if (m_question->isChecked()) item->m_flags.append( DDataControl::e_question );
	if (m_trash->isChecked()) item->m_flags.append( DDataControl::e_trash );
	if (m_meeting->isChecked()) item->m_flags.append( DDataControl::e_meeting );
	if (m_trouble->isChecked()) item->m_flags.append( DDataControl::e_trouble );

	m_data->notifyChildChange(m_lastid);
}

void KDissertPart::setItemChanged(int id)
{
	emit itemChanged(id);
}

void KDissertPart::zoomIn()
{
	QWMatrix wm;
	wm.setMatrix(1.25, 0, 0, 1.25, 0, 0);
	m_canvasview->setWorldMatrix( wm*= m_canvasview->worldMatrix());
	m_canvasview->focusOnRoot();
}

void KDissertPart::zoomOut()
{
	QWMatrix wm;
	wm.setMatrix(.8, 0, 0, 0.8, 0, 0);
	m_canvasview->setWorldMatrix( wm*= m_canvasview->worldMatrix());
	m_canvasview->focusOnRoot();
}

void KDissertPart::setZoom(int zoom)
{
	// TODO
}

void KDissertPart::docProperties()
{
	docsettingsdlg* dlg = new docsettingsdlg(this->widget(), m_data);
	dlg->exec();
}

void KDissertPart::launchURL(const QString& url, bool usekfmclient)
{
	QMap<QChar,QString> map;
	map.insert('s', url);

	QString command = Settings::launchCommand();
	if (usekfmclient) command = "kfmclient exec %s";
	
	KRun::runCommand( KMacroExpander::expandMacrosShellQuote( command, map ) );
}

#include "KDissertPart.moc"

