/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-stop-reload-action.h"

#include <glib/gi18n.h>
#include "kazehakase.h"

typedef struct _KzStopReloadActionPrivate KzStopReloadActionPrivate;
struct _KzStopReloadActionPrivate 
{
	KzWindow *kz;
	gint      state;
};

#define KZ_STOP_RELOAD_ACTION_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_STOP_RELOAD_ACTION, KzStopReloadActionPrivate))

enum {
	PROP_0,
	PROP_KZ_WINDOW,
	PROP_STATE
};


static void dispose          (GObject *object);
static void set_property     (GObject           *object,
                              guint              prop_id,
                              const GValue      *value,
                              GParamSpec        *pspec);
static void get_property     (GObject           *object,
                              guint              prop_id,
                              GValue            *value,
                              GParamSpec        *pspec);

static void activate         (GtkAction         *action);
static void connect_proxy    (GtkAction         *action,
                              GtkWidget         *proxy);
static void disconnect_proxy (GtkAction         *action,
                              GtkWidget         *proxy);

static void kz_stop_reload_action_sync_state       (GtkAction  *action,
				  		    GParamSpec *pspec,
				  		    GtkWidget  *proxy);

G_DEFINE_TYPE(KzStopReloadAction, kz_stop_reload_action, GTK_TYPE_ACTION)

static void
kz_stop_reload_action_class_init (KzStopReloadActionClass *klass)
{
	GObjectClass *object_class;
	GtkActionClass *action_class;

	object_class = G_OBJECT_CLASS(klass);
	action_class = GTK_ACTION_CLASS(klass);

	object_class->set_property     = set_property;
	object_class->get_property     = get_property;
	object_class->dispose          = dispose;

	action_class->activate         = activate;
	action_class->connect_proxy    = connect_proxy;
	action_class->disconnect_proxy = disconnect_proxy;

	g_object_class_install_property
		(object_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object ("kz-window",
				      _("KzWindow"),
				      _("The KzWindow to add a stop_reload button"),
				      KZ_TYPE_WINDOW,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property
		(object_class,
		 PROP_STATE,
		 g_param_spec_int ("state",
				 _("The state"),
				 _("The state of reload of stop"),
				   G_MININT,
				   G_MAXINT,
				   KZ_STOP_RELOAD_ACTION_STATE_RELOAD,
				   G_PARAM_READWRITE));
	g_type_class_add_private (object_class, sizeof(KzStopReloadActionPrivate));
}


static void
kz_stop_reload_action_init (KzStopReloadAction *action)
{
	KzStopReloadActionPrivate *priv = KZ_STOP_RELOAD_ACTION_GET_PRIVATE (action);

  	priv->kz = NULL;
  	priv->state = KZ_STOP_RELOAD_ACTION_STATE_RELOAD;
}


static void
dispose (GObject *obj)
{
	KzStopReloadActionPrivate *priv = KZ_STOP_RELOAD_ACTION_GET_PRIVATE (obj);

	if (priv->kz)
		g_object_unref(priv->kz);
	priv->kz = NULL;

	if (G_OBJECT_CLASS(kz_stop_reload_action_parent_class)->dispose)
		G_OBJECT_CLASS(kz_stop_reload_action_parent_class)->dispose(obj);
}


static void
set_property (GObject         *object,
              guint            prop_id,
              const GValue    *value,
              GParamSpec      *pspec)
{
	KzStopReloadActionPrivate *priv = KZ_STOP_RELOAD_ACTION_GET_PRIVATE (object);
  
	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		priv->kz = g_object_ref(g_value_get_object(value));
		break;
	case PROP_STATE:
		priv->state = g_value_get_int(value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
get_property (GObject         *object,
              guint            prop_id,
              GValue          *value,
              GParamSpec      *pspec)
{
	KzStopReloadActionPrivate *priv = KZ_STOP_RELOAD_ACTION_GET_PRIVATE (object);

	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		g_value_set_object(value, priv->kz);
		break;
    	case PROP_STATE:
      		g_value_set_int (value, priv->state);
      		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


KzStopReloadAction *
kz_stop_reload_action_new(KzWindow *kz)
{
	KzStopReloadAction *action;

	action = g_object_new(KZ_TYPE_STOP_RELOAD_ACTION,
			     "name",     "StopReload",
			     "label",    _("_Reload"),
			     "tooltip",  _("Display the latest content of the current page"),
			     "stock_id",  GTK_STOCK_REFRESH,
			     "kz-window", kz,
			     NULL);

	return action;
}


static void
activate (GtkAction *action)
{
	KzWindow *kz;
	GdkModifierType state = (GdkModifierType)0;
	gint x, y;
	KzWeb *web;
	KzStopReloadActionPrivate *priv = KZ_STOP_RELOAD_ACTION_GET_PRIVATE (action);
	
	kz = priv->kz;
	if (!KZ_IS_WINDOW(kz)) return;

	gdk_window_get_pointer(NULL, &x, &y, &state);

	web = KZ_WINDOW_CURRENT_WEB(kz);
	if (!web)
		return;

	switch (priv->state)
	{
	case KZ_STOP_RELOAD_ACTION_STATE_STOP:	
		kz_web_stop_load(web);
		break;
	case KZ_STOP_RELOAD_ACTION_STATE_RELOAD:
		kz_web_reload(web,
				(state & GDK_SHIFT_MASK) ?
				KZ_WEB_RELOAD_BYPASS_PROXY_AND_CACHE :
				KZ_WEB_RELOAD_NORMAL);
		break;
	default:
		break;
	}
}


static void
connect_proxy (GtkAction *action, GtkWidget *proxy)
{
	g_signal_connect_object(action, "notify::state",
				G_CALLBACK(kz_stop_reload_action_sync_state),
				proxy, 0);
	(* GTK_ACTION_CLASS (kz_stop_reload_action_parent_class)->connect_proxy) (action, proxy);
}


static void
disconnect_proxy (GtkAction *action, GtkWidget *proxy)
{
	GTK_ACTION_CLASS (kz_stop_reload_action_parent_class)->disconnect_proxy (action, proxy);

	g_signal_handlers_disconnect_by_func
			(proxy,
			 G_CALLBACK(kz_stop_reload_action_sync_state),
			 action);
}


static void
kz_stop_reload_action_sync_state (GtkAction  *action,
				  GParamSpec *pspec,
				  GtkWidget  *proxy)
{
	gchar *stock_id;
	KzStopReloadActionPrivate *priv = KZ_STOP_RELOAD_ACTION_GET_PRIVATE (action);
	
	switch(priv->state)
	{
	case KZ_STOP_RELOAD_ACTION_STATE_STOP:
		g_object_set(action,
			     "label",   _("_Stop"),
			     "tooltip", _("Stop current data transfer"),
			     "stock_id",  GTK_STOCK_STOP,
			     NULL);
		break;
	case KZ_STOP_RELOAD_ACTION_STATE_RELOAD:
	default:
		g_object_set(action,
			     "label", _("_Reload"),
			     "tooltip", _("Display the latest content of the current page"),
			     "stock_id", GTK_STOCK_REFRESH,
			     NULL);
		break;
	}

	/* 
	 * GtkAction does not redraw proxy widget 
	 * when the stock_id changes 
	 * so we redraw wproxy widget explicitly.
	 */
	g_object_get(action,
		     "stock_id", &stock_id,
		     NULL);

	if (GTK_IS_IMAGE_MENU_ITEM(proxy))
	{
		GtkWidget *image;
		image = gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_MENU);
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(proxy), 
					      image);
	}
	else if (GTK_IS_TOOL_BUTTON(proxy))
	{
		GtkWidget *image;
		image = gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_BUTTON);
		gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(proxy),
						image);
		gtk_widget_show(image);
	}
	g_free(stock_id);
}


void
kz_stop_reload_action_set_state (KzStopReloadAction *action,
				 KzStopReloadActionState state)
{
	g_return_if_fail(KZ_IS_STOP_RELOAD_ACTION(action));

	g_object_set(action,
		     "state", state, NULL);
}

