/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
 * These codes originated in gtkhpaned.c and gtkvpaned.c in GTK+.
 */

#include "kz-paned.h"

#include "gobject-utils.h"
#include "gtk-utils.h"
#include <glib/gi18n.h>

enum {
	CHANGED_POSITION,
	LAST_SIGNAL
};

/* object class */
static void kz_paned_class_init     (KzPanedClass   *klass);
static void kz_paned_init           (KzPaned        *kzpaned);
static void kz_paned_size_request   (GtkWidget      *widget,
				     GtkRequisition *requisition);
static void kz_paned_size_allocate  (GtkWidget      *widget,
				     GtkAllocation  *allocation);
static void kz_paned_hsize_request  (GtkWidget      *widget,
				     GtkRequisition *requisition);
static void kz_paned_hsize_allocate (GtkWidget      *widget,
				     GtkAllocation  *allocation);
static void kz_paned_vsize_request  (GtkWidget      *widget,
				     GtkRequisition *requisition);
static void kz_paned_vsize_allocate (GtkWidget      *widget,
				     GtkAllocation  *allocation);

static void kz_paned_rotate_position      (KzPaned *kzpaned);
static void kz_paned_flip_position        (KzPaned *kzpaned);
static void kz_paned_set_divider_position (KzPaned *kzpaned);

#define KZ_PANED_SET_VERTICAL(kzpaned)					\
{									\
	GTK_PANED(kzpaned)->cursor_type = GDK_SB_V_DOUBLE_ARROW;	\
	GTK_PANED(kzpaned)->orientation = GTK_ORIENTATION_HORIZONTAL;	\
}

#define KZ_PANED_SET_HORIZONTAL(kzpaned)				\
{									\
	GTK_PANED(kzpaned)->cursor_type = GDK_SB_H_DOUBLE_ARROW;	\
	GTK_PANED(kzpaned)->orientation = GTK_ORIENTATION_VERTICAL;	\
}

static GtkPanedClass *parent_class = NULL;

static gint kz_paned_signals[LAST_SIGNAL] = {0};
 
KZ_OBJECT_GET_TYPE(kz_paned, "KzPaned", KzPaned,
		   kz_paned_class_init, kz_paned_init,
		   GTK_TYPE_PANED)

static void
kz_paned_class_init (KzPanedClass *klass)
{
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (klass);

	widget_class  = (GtkWidgetClass *) klass;

	widget_class->size_request  = kz_paned_size_request;
	widget_class->size_allocate = kz_paned_size_allocate;
	
	klass->changed_position = NULL;

	kz_paned_signals[CHANGED_POSITION]
		= g_signal_new ("changed_position",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzPanedClass, changed_position),
				NULL, NULL,
				g_cclosure_marshal_VOID__INT,
				G_TYPE_NONE, 1,
				G_TYPE_INT);
}


static void
kz_paned_init (KzPaned *kzpaned)
{
	KZ_PANED_SET_HORIZONTAL(kzpaned);	
	kzpaned->position  = GTK_POS_LEFT;
}


GtkWidget *
kz_paned_new (void)
{
	KzPaned *kzpaned;

	kzpaned = g_object_new (KZ_TYPE_PANED, NULL);

	return GTK_WIDGET (kzpaned);
}


void
kz_paned_set_position(KzPaned *kzpaned, GtkPositionType position)
{
	GtkPositionType prev_pos;
	prev_pos = kzpaned->position;

	kzpaned->position = position;
	switch(position)
	{
	 case GTK_POS_LEFT:
		KZ_PANED_SET_HORIZONTAL(kzpaned);
		if (prev_pos == GTK_POS_BOTTOM)
			kz_paned_rotate_position(kzpaned);
		else if (prev_pos == GTK_POS_RIGHT)
			kz_paned_flip_position(kzpaned);
		break;
	 case GTK_POS_RIGHT:
		KZ_PANED_SET_HORIZONTAL(kzpaned);
		if (prev_pos == GTK_POS_TOP)
			kz_paned_rotate_position(kzpaned);
		else if (prev_pos == GTK_POS_LEFT)
			kz_paned_flip_position(kzpaned);
		else if (prev_pos == GTK_POS_BOTTOM)
			kz_paned_set_divider_position(kzpaned);
		break;
	 case GTK_POS_TOP:
		KZ_PANED_SET_VERTICAL(kzpaned);
		if (prev_pos == GTK_POS_RIGHT)
			kz_paned_rotate_position(kzpaned);
		else if (prev_pos == GTK_POS_BOTTOM)
			kz_paned_flip_position(kzpaned);
		break;
	 case GTK_POS_BOTTOM:
		KZ_PANED_SET_VERTICAL(kzpaned);
		if (prev_pos == GTK_POS_LEFT)
			kz_paned_rotate_position(kzpaned);
		else if (prev_pos == GTK_POS_TOP)
			kz_paned_flip_position(kzpaned);
		else if (prev_pos == GTK_POS_RIGHT)
			kz_paned_set_divider_position(kzpaned);
		break;
	 default:
		break;
	}	
	gtk_widget_queue_resize(GTK_WIDGET(kzpaned));

	g_signal_emit(kzpaned,
		      kz_paned_signals[CHANGED_POSITION],
		      0,
		      position);
}


static void
kz_paned_set_divider_position(KzPaned *kzpaned)
{
	GtkPaned *paned;
	GtkWidget *child1, *child2;
	gint child1_width, child1_height;
	gint child2_width, child2_height;
	
	paned = GTK_PANED(kzpaned);

	child1 = paned->child1;
	child2 = paned->child2;
	child1_width  = child1->allocation.width;
	child1_height = child1->allocation.height;
	child2_width  = child2->allocation.width;
	child2_height = child2->allocation.height;

	switch(kzpaned->position)
	{
	 case GTK_POS_RIGHT:/* only from GTK_POS_BOTTOM */
		gtk_paned_set_position(paned, child1_width - child2_height);
		break;
	 case GTK_POS_BOTTOM: /* only from GTK_POS_RIGHT */
		gtk_paned_set_position(paned, child1_height - child2_width);
		break;
	 default:
		break;
	}
}


static void
kz_paned_rotate_position(KzPaned *kzpaned)
{
	GtkPaned *paned;
	GtkWidget *child1, *child2;
	gint child1_width, child1_height;
	gint child2_width, child2_height;
	
	paned = GTK_PANED(kzpaned);

	child1 = paned->child1;
	child2 = paned->child2;
	child1_width  = child1->allocation.width;
	child1_height = child1->allocation.height;
	child2_width  = child2->allocation.width;
	child2_height = child2->allocation.height;

	paned->child1 = child2;
	paned->child2 = child1;
	
	/*
	 * When pane is divided in horizontal, child1_height always equals child2_height.
	 * When pane is divided in vertical, child1_width always equals child2_width.
	 */

	switch(kzpaned->position)
	{
	 case GTK_POS_LEFT: /* only from GTK_POS_BOTTOM */
		gtk_paned_set_position(paned, child2_height);
		break;
	 case GTK_POS_RIGHT:/* only from GTK_POS_TOP */
		gtk_paned_set_position(paned, child1_width - child1_height);
		break;
	 case GTK_POS_TOP: /* only from GTK_POS_RIGHT */
		gtk_paned_set_position(paned, child2_width);
		break;
	 case GTK_POS_BOTTOM: /* only from GTK_POS_LEFT */
		gtk_paned_set_position(paned, child1_height - child1_width);
		break;
	 default:
		break;
	}
}


static void
kz_paned_flip_position(KzPaned *kzpaned)
{
	GtkPaned *paned;
	GtkWidget *child1, *child2;
	gint child2_width, child2_height;
	
	paned = GTK_PANED(kzpaned);

	child1 = paned->child1;
	child2 = paned->child2;
	child2_width  = child2->allocation.width;
	child2_height = child2->allocation.height;

	paned->child1 = child2;
	paned->child2 = child1;
	switch(kzpaned->position)
	{
	 case GTK_POS_LEFT:
	 case GTK_POS_RIGHT:
		gtk_paned_set_position(paned, child2_width);
		break;
	 case GTK_POS_TOP:
	 case GTK_POS_BOTTOM:
		gtk_paned_set_position(paned, child2_height);
		break;
	 default:
		break;
	}
}


static void
kz_paned_size_request (GtkWidget      *widget,
		       GtkRequisition *requisition)
{
	KzPaned *kzpaned;
	g_return_if_fail (KZ_IS_PANED (widget));
	kzpaned = KZ_PANED (widget);

	switch(kzpaned->position)
	{
	 case GTK_POS_LEFT:
	 case GTK_POS_RIGHT:
		kz_paned_hsize_request(widget, requisition);
		break;
	 case GTK_POS_TOP:
	 case GTK_POS_BOTTOM:
		kz_paned_vsize_request(widget, requisition);
		break;
	 default:
		break;
	}
}


static void
kz_paned_size_allocate (GtkWidget     *widget,
			GtkAllocation *allocation)
{
	KzPaned *kzpaned;
	g_return_if_fail (KZ_IS_PANED (widget));
	kzpaned = KZ_PANED (widget);
	
	switch(kzpaned->position)
	{
	 case GTK_POS_LEFT:
	 case GTK_POS_RIGHT:
		kz_paned_hsize_allocate(widget, allocation);
		break;
	 case GTK_POS_TOP:
	 case GTK_POS_BOTTOM:
		kz_paned_vsize_allocate(widget, allocation);
		break;
	 default:
		break;
	}
}


static void
kz_paned_vsize_request (GtkWidget      *widget,
			GtkRequisition *requisition)
{
	GtkPaned *paned = GTK_PANED (widget);
	GtkRequisition child_requisition;

	requisition->width = 0;
	requisition->height = 0;

	if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1))
	{
		gtk_widget_size_request (paned->child1, &child_requisition);

		requisition->height = child_requisition.height;
		requisition->width = child_requisition.width;
	}

	if (paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
	{
		gtk_widget_size_request (paned->child2, &child_requisition);

		requisition->width = MAX (requisition->width, child_requisition.width);
		requisition->height += child_requisition.height;
	}

	requisition->height += GTK_CONTAINER (paned)->border_width * 2;
	requisition->width += GTK_CONTAINER (paned)->border_width * 2;
  
	if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1) &&
	    paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
	{
		gint handle_size;

		gtk_widget_style_get (widget, "handle_size", &handle_size, NULL);
		requisition->height += handle_size;
	}
}

static void
kz_paned_vsize_allocate (GtkWidget     *widget,
			 GtkAllocation *allocation)
{
	GtkPaned *paned = GTK_PANED (widget);
	gint border_width = GTK_CONTAINER (paned)->border_width;

	widget->allocation = *allocation;

	if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1) &&
	    paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
	{
		GtkRequisition child1_requisition;
		GtkRequisition child2_requisition;
		GtkAllocation child1_allocation;
		GtkAllocation child2_allocation;
		gint handle_size;
      
		gtk_widget_style_get (widget, "handle_size", &handle_size, NULL);

		gtk_widget_get_child_requisition (paned->child1, &child1_requisition);
		gtk_widget_get_child_requisition (paned->child2, &child2_requisition);
# warning FIXME! this function is deprecated in GTK+-2.4.0.    
		gtk_paned_compute_position (paned,
					    MAX (1, widget->allocation.height
						 - handle_size
						 - 2 * border_width),
					    child1_requisition.height,
					    child2_requisition.height);

		paned->handle_pos.x = widget->allocation.x + border_width;
		paned->handle_pos.y = widget->allocation.y + paned->child1_size + border_width;
		paned->handle_pos.width = MAX (1, (gint) widget->allocation.width - 2 * border_width);
		paned->handle_pos.height = handle_size;
      
		if (GTK_WIDGET_REALIZED (widget))
		{
			if (GTK_WIDGET_MAPPED (widget))
				gdk_window_show (paned->handle);
			gdk_window_move_resize (paned->handle,
						paned->handle_pos.x,
						paned->handle_pos.y,
						paned->handle_pos.width,
						handle_size);
		}

		child1_allocation.width = child2_allocation.width = MAX (1, (gint) allocation->width - border_width * 2);
		child1_allocation.height = MAX (1, paned->child1_size);
		child1_allocation.x = child2_allocation.x = widget->allocation.x + border_width;
		child1_allocation.y = widget->allocation.y + border_width;
      
		child2_allocation.y = child1_allocation.y + paned->child1_size + paned->handle_pos.height;
		child2_allocation.height = MAX (1, widget->allocation.y + widget->allocation.height - child2_allocation.y - border_width);
      
		/* Now allocate the childen, making sure, when resizing not to
		 * overlap the windows
		 */
		if (GTK_WIDGET_MAPPED (widget) &&
		    paned->child1->allocation.height < child1_allocation.height)
		{
			gtk_widget_size_allocate (paned->child2, &child2_allocation);
			gtk_widget_size_allocate (paned->child1, &child1_allocation);
		}
		else
		{
			gtk_widget_size_allocate (paned->child1, &child1_allocation);
			gtk_widget_size_allocate (paned->child2, &child2_allocation);
		}
	}
	else
	{
		GtkAllocation child_allocation;

		if (GTK_WIDGET_REALIZED (widget))      
			gdk_window_hide (paned->handle);

		if (paned->child1)
			gtk_widget_set_child_visible (paned->child1, TRUE);
		if (paned->child2)
			gtk_widget_set_child_visible (paned->child2, TRUE);

		child_allocation.x = widget->allocation.x + border_width;
		child_allocation.y = widget->allocation.y + border_width;
		child_allocation.width = MAX (1, allocation->width - 2 * border_width);
		child_allocation.height = MAX (1, allocation->height - 2 * border_width);

		if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1))
			gtk_widget_size_allocate (paned->child1, &child_allocation);
		else if (paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
			gtk_widget_size_allocate (paned->child2, &child_allocation);
	}
}

static void
kz_paned_hsize_request (GtkWidget      *widget,
			GtkRequisition *requisition)
{
	GtkPaned *paned = GTK_PANED (widget);
	GtkRequisition child_requisition;

	requisition->width = 0;
	requisition->height = 0;

	if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1))
	{
		gtk_widget_size_request (paned->child1, &child_requisition);

		requisition->height = child_requisition.height;
		requisition->width = child_requisition.width;
	}

	if (paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
	{
		gtk_widget_size_request (paned->child2, &child_requisition);

		requisition->height = MAX(requisition->height, child_requisition.height);
		requisition->width += child_requisition.width;
	}

	requisition->width += GTK_CONTAINER (paned)->border_width * 2;
	requisition->height += GTK_CONTAINER (paned)->border_width * 2;

	if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1) &&
	    paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
	{
		gint handle_size;
      
		gtk_widget_style_get (widget, "handle_size", &handle_size, NULL);
		requisition->width += handle_size;
	}
}

static void
flip_child (GtkWidget *widget, GtkAllocation *child_pos)
{
	gint x = widget->allocation.x;
	gint width = widget->allocation.width;

	child_pos->x = 2 * x + width - child_pos->x - child_pos->width;
}

static void
kz_paned_hsize_allocate (GtkWidget     *widget,
			 GtkAllocation *allocation)
{
	GtkPaned *paned = GTK_PANED (widget);
	gint border_width = GTK_CONTAINER (paned)->border_width;

	widget->allocation = *allocation;

	if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1) &&
	    paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
	{
		GtkAllocation child1_allocation;
		GtkAllocation child2_allocation;
		GtkRequisition child1_requisition;
		GtkRequisition child2_requisition;
		gint handle_size;
      
		gtk_widget_style_get (widget, "handle_size", &handle_size, NULL);

		gtk_widget_get_child_requisition (paned->child1, &child1_requisition);
		gtk_widget_get_child_requisition (paned->child2, &child2_requisition);
      
		gtk_paned_compute_position (paned,
					    MAX (1, widget->allocation.width
						 - handle_size
						 - 2 * border_width),
					    child1_requisition.width,
					    child2_requisition.width);
      
		paned->handle_pos.x = widget->allocation.x + paned->child1_size + border_width;
		paned->handle_pos.y = widget->allocation.y + border_width;
		paned->handle_pos.width = handle_size;
		paned->handle_pos.height = MAX (1, widget->allocation.height - 2 * border_width);

		child1_allocation.height = child2_allocation.height = MAX (1, (gint) allocation->height - border_width * 2);
		child1_allocation.width = MAX (1, paned->child1_size);
		child1_allocation.x = widget->allocation.x + border_width;
		child1_allocation.y = child2_allocation.y = widget->allocation.y + border_width;

		child2_allocation.x = child1_allocation.x + paned->child1_size + paned->handle_pos.width;
		child2_allocation.width = MAX (1, widget->allocation.x + widget->allocation.width - child2_allocation.x - border_width);

		if (gtk_widget_get_direction (GTK_WIDGET (widget)) == GTK_TEXT_DIR_RTL)
		{
			flip_child (widget, &(child2_allocation));
			flip_child (widget, &(child1_allocation));
			flip_child (widget, &(paned->handle_pos));
		}
      
		if (GTK_WIDGET_REALIZED (widget))
		{
			if (GTK_WIDGET_MAPPED (widget))
				gdk_window_show (paned->handle);
			gdk_window_move_resize (paned->handle,
						paned->handle_pos.x,
						paned->handle_pos.y,
						handle_size,
						paned->handle_pos.height);
		}
      
		/* Now allocate the childen, making sure, when resizing not to
		 * overlap the windows
		 */
		if (GTK_WIDGET_MAPPED (widget) &&
		    paned->child1->allocation.width < child1_allocation.width)
		{
			gtk_widget_size_allocate (paned->child2, &child2_allocation);
			gtk_widget_size_allocate (paned->child1, &child1_allocation);
		}
		else
		{
			gtk_widget_size_allocate (paned->child1, &child1_allocation);
			gtk_widget_size_allocate (paned->child2, &child2_allocation);
		}
	}
	else
	{
		GtkAllocation child_allocation;

		if (GTK_WIDGET_REALIZED (widget))      
			gdk_window_hide (paned->handle);

		if (paned->child1)
			gtk_widget_set_child_visible (paned->child1, TRUE);
		if (paned->child2)
			gtk_widget_set_child_visible (paned->child2, TRUE);

		child_allocation.x = widget->allocation.x + border_width;
		child_allocation.y = widget->allocation.y + border_width;
		child_allocation.width = MAX (1, allocation->width - 2 * border_width);
		child_allocation.height = MAX (1, allocation->height - 2 * border_width);
      
		if (paned->child1 && GTK_WIDGET_VISIBLE (paned->child1))
			gtk_widget_size_allocate (paned->child1, &child_allocation);
		else if (paned->child2 && GTK_WIDGET_VISIBLE (paned->child2))
			gtk_widget_size_allocate (paned->child2, &child_allocation);
	}
}
