{-
    Kaya - My favourite toy language.
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

module Module where

import Language
import Debug.Trace
import System.Directory
import List
import Lib
import Options
import Portability

importVersion :: Int
importVersion = 2

writeIface :: FilePath -> Program -> IO ()
writeIface fn prog
  = do let str = mkIface prog
       writeFile fn str

mkIface :: Program -> String
mkIface [] = ""
mkIface (CInclude str:xs) = mkIface xs
mkIface (Imported str:xs) = "%imported "++show str++";\n"++mkIface xs
mkIface (Linker str:xs) = "%link "++show str++";\n"++mkIface xs
mkIface ((FunBind (f,l,n,ty,fopts,Unbound) _ _):xs) 
    | elem Export fopts = mkExt n ty fopts ++ "\n" ++ mkIface xs
    | otherwise = mkExt n ty (delete Public fopts) ++ "\n" ++ mkIface xs
mkIface ((FunBind (f,l,n,ty,fopts,Defined _) _ _):xs)
   | not (Generated `elem` fopts) = mkExt n ty fopts ++ "\n" ++ mkIface xs
   | otherwise = mkExtCName n ty ++ "\n" ++ 
                 mkIface xs -- write the C name, to keep FunMap correct
--mkIface (FunBind (n,ty,DataCon i ar):xs) = 
--   mkExtCon n ty i ar ++ "\n" ++ mkIface xs
mkIface ((DataDecl f l dopts n tys cons comm):xs) 
   | DAbstract `elem` dopts && DExport `elem` dopts 
       = mkData n tys dopts [] ++ "\n" ++ mkIface xs
   | DPublic `elem` dopts && DExport `elem` dopts 
       = mkData n tys dopts cons ++ "\n" ++ mkIface xs
   | otherwise = mkIface xs
--       = mkData n tys [] [] ++ "\n" ++ mkIface xs

mkIface ((TySyn (f,l,n,ps,ty,True)):xs) =
   mkTySyn n ps ty ++ "\n" ++ mkIface xs
mkIface (_:xs) = mkIface xs

mkExtCName n ty = "%lifted \"" ++ show n ++ mangling ty ++ "\""

mkExt :: Name -> Type -> [FOpt] -> String
mkExt n (Fn defaults args ret) fopts 
   = "%extern " ++show importVersion ++ " "++ writefopts fopts ++ show ret ++ " " ++ showuser n ++ 
     (if elem NoArgs fopts then ";" else "(" ++ showargs defaults args ++ ");")
  where showargs [] [] = ""
	showargs [d] [x] = show x ++ showdef d
	showargs (d:ds) (x:xs) = show x ++ showdef d ++ "," ++ showargs ds xs

        showdef Nothing = "";
	showdef (Just a) = " = " ++ showexp a -- FIXME: Do it properly
	writefopts (Public:xs) = "public " ++ writefopts xs
	writefopts (Pure:xs) = "pure " ++ writefopts xs
	writefopts (_:xs) = writefopts xs
	writefopts [] = ""
mkExt n t fopts = "%extern " ++ show importVersion ++ " " ++ 
                  writefopts fopts ++ 
		  show t ++ " " ++ showuser n ++ 
     (if elem NoArgs fopts then ";" else "();")
   where writefopts (Public:xs) = "public " ++ writefopts xs
	 writefopts (Pure:xs) = "pure " ++ writefopts xs
	 writefopts (_:xs) = writefopts xs
	 writefopts [] = ""


mkData :: Name -> [Type] -> [DOpt] -> [ConDecl] -> String
mkData n args opts cons = "%data " ++ show importVersion ++ " " ++ 
                          showopts opts ++ showuser n ++ 
			  params args ++ " = " ++ showcons cons ++ ";"
   where params [] = ""
	 params (x:xs) = "<" ++ p' (x:xs) ++ ">"
	 showopts [] = ""
	 showopts (DPublic:xs) = "public "++showopts xs
	 showopts (DAbstract:xs) = "abstract "++showopts xs
	 showopts (x:xs) = showopts xs

         p' [] = ""
	 p' [x] = show x
	 p' (x:xs) = show x ++ "," ++ p' xs
	 showcons [] = ""
	 showcons [x] = showcon x
	 showcons (x:xs) = showcon x ++ " | " ++ showcons xs
	 showcon (Con n (Fn _ ts _) ns _) = showuser n ++ "(" ++ 
					    showargs ts ns ++ ")"
	 showargs [] [] = ""
	 showargs (t:[]) (n:[]) = showarg n t
	 showargs (t:ts) (n:ns) = showarg n t ++ "," ++ showargs ts ns
	 showarg n t = show t ++ case n of
		          None -> ""
			  x -> " " ++ showuser x

mkTySyn :: Name -> [Name] -> Type -> String
mkTySyn n ps t = "%type " ++ show importVersion ++ " " ++
                 showuser n ++ params ps ++ " = " ++ show t ++ ";"
   where params [] = ""
	 params (x:xs) = "<" ++ p' (x:xs) ++ ">"
         p' [] = ""
	 p' [x] = showuser x
	 p' (x:xs) = showuser x ++ "," ++ p' xs

{-
mkExtTy :: Name -> [Type] -> String
mkExtTy (UN n) tys = "%datatype " ++ n ++ "<" ++ showtys tys ++ ">;"
   where showtys [] = ""
	 showtys [x] = show x
	 showtys (x:xs) = show x ++ "," ++ show xs

mkExtCon :: Name -> Type -> Int -> Int -> String
mkExtCon (UN n) (Fn tvars args ret) i ar
    = "%datacon " ++ show ret ++ " " ++ n ++ 
      "(" ++ showargs args ++ ")[" ++ show i ++ "," ++ show ar ++"];"
  where showargs [] = ""
	showargs [x] = show x
	showargs (x:xs) = show x ++ "," ++ showargs xs
mkExtCon (UN n) t i ar = "%datacon " ++ show t ++ " " ++ n ++ "()[" ++ show i ++ "," ++ show ar ++"];"
-}

-- Take a list of dynamic link package files (.ddl), and return a mapping
-- from .o files to the libraries to link instead.
linkfiles :: [FilePath] -> [String] -> IO [(String,String)]
linkfiles libs [] = return []
linkfiles libs (f:fs) 
    = do ds <- linkfiles libs fs
	 libdata <- findFile libs (f++".ddl")
	 case libdata of
             (Just d) -> do
--	        putStrLn $ show d
	        let file = lines d
	        let info = words (file!!0)
		let linkinfo = (file!!1)
		let lmap = map (\x -> (x,"-l"++(head info)++" "++linkinfo)) (tail info)
--		putStrLn (show lmap)
		return $ nub (lmap++ds)
	     Nothing -> return ds
		   

getObjs :: Program -> [FilePath] -> [(String,String)] -> 
	   IO ([FilePath],[String])
getObjs p fp dls = do (fp,libs) <- go' p
		      return (nub fp, nub libs) 
    where go' [] = return ([],[])
	  go' ((Imported str):xs) = 
	      do ofile <- findFile fp (str++".o")
		 (rest,lops) <- go' xs
		 case (lookup (str++".o") dls) of
		    Nothing -> return (ofile:rest, lops)
		    (Just lib) -> return (rest,nub (lib:lops))
	  go' ((Linker str):xs) =
              do (rest,lops) <- go' xs
		 return (rest,('-':'l':str):lops)
	  go' (x:xs) = go' xs

          findFile [] path
	      = fail $ "No such module " ++ path
	  findFile (x:xs) path 
	      = do --putStrLn $ "Looking in " ++ (x++path)
		   exist <- doesFileExist (x++path)
		   if exist 
		      then return (x++path)
		      else findFile xs path


showexp :: Raw -> String
showexp (RVar _ _ n) = showuser n
showexp (RQVar _ _ n) = "@" ++ showuser n
showexp (RConst _ _ c) = showconst c
showexp (RApply _ _ r as) = showexp r ++ "(" ++ showargs as ++ ")"
     where showargs [] = ""
	   showargs [x] = showexp x
	   showargs (x:xs) = showexp x ++ "," ++ showargs xs
showexp (RArrayInit _ _ rs) = "[" ++ showargs rs ++ "]"
     where showargs [] = ""
	   showargs [x] = showexp x
	   showargs (x:xs) = showexp x ++ "," ++ showargs xs
showexp (RIndex _ _ r i) = showexp r ++ "["++showexp i++"]"
showexp (RField _ _ n f) = showexp n ++ "." ++ showuser f
showexp (RUnary _ _ op r) = showunop op ++ showexp r
   where showunop Not = "!"
	 showunop Neg = "-"
-- TODO: The rest
showexp _ = error "Invalid default argument, this probably shouldn't have parsed"

findLib :: [FilePath] -> FilePath -> IO FilePath
findLib [] path
  = fail $ "Can't find " ++ path
findLib (x:xs) path 
  = do ex <- doesFileExist (x++path)
       if ex then return (x++path)
	  else findLib xs path

findFile :: [FilePath] -> FilePath -> IO (Maybe String)
findFile [] path
  = return Nothing
findFile (x:xs) path 
  = catch
         (do --putStrLn $ "Trying " ++ x ++ path
	     f <- readFile (x++path)
	     return (Just f))
         (\e -> findFile xs path)

-- Get all the library directories, looking at the options and the
-- KAYA_LIBRARY_PATH environment variable.

getAllLibDirs :: Options -> IO [FilePath]
getAllLibDirs opts = do
     let lds = getlibdir opts ("./":
                               (map ((++"/").stripSlash) libpath) ++ 
                               (map ((++"/imports/").stripSlash) libpath))
     env <- environment "KAYA_LIBRARY_PATH"
     return $ if (noenvlibs opts) then lds else (filter (\x -> length x > 0) $ splitBy pathsep env) ++ lds 

splitBy sep (Just xs) = splitBy' sep xs []
splitBy sep _ = []
splitBy' sep [] acc = [reverse acc]
splitBy' sep (x:xs) acc | x == sep = (reverse acc):(splitBy' sep xs [])
                        | otherwise = splitBy' sep xs (x:acc)

