{-
    Kaya - My favourite toy language.
    Copyright (C) 2004-2006 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

module Chaser where

import Parser
import Language
import Options
import Module
import Lib
import Lexer

import Char
import IO
import System.Directory
import Debug.Trace
import Monad

-- File name, module name and dependencies
data ModuleTree = Mod FilePath InputType Name [ModuleTree]
   deriving Show

data ModuleData = MD { mod_inputtype :: InputType,
                       mod_name :: String,
                       mod_imports :: [String] }

instance Show ModuleData where
    show (MD int nm imp) = show int ++ " " ++ nm ++ " depends on " ++ show imp

data CompileNeeded = Comp { cn_filename :: FilePath,
                            cn_inputtype :: InputType,
                            cn_module :: Name,
                            cn_build :: Bool }
   deriving Show

-- Given a top level file, return a tree describing which modules depend 
-- on which others. Look out for circular dependencies and ignore them.
getDeps :: FilePath -> Options -> [FilePath] -> IO ModuleTree
getDeps fn opts done = 
    do libdirs <- getAllLibDirs opts
       prog <- readFile fn
--       let imppt = parse (getroot fn) libdirs prog fn
       let moddata = quickParse (getroot fn) prog (nochase opts)
       when (dumpdeps opts) $ do
               case moddata of
                        (Success x) -> do 
                           putStrLn $ show x
                           when (length done > 0) $ 
                             putStrLn $ "\tThis path contains " ++ show done
                        _ -> return ()
       -- Dig out all of the imports from imppt
       if (nochase opts)
          then case moddata of
                   (Success (MD ty mod pt)) -> return $ Mod fn ty (UN mod) []
                   (Failure _ _ _) -> return $ Mod fn Module (UN (getroot fn)) []
          else findImports moddata libdirs
  where getroot ".k" = ""
	getroot (x:xs) = x:(getroot xs)
	getroot [] = ""

        findImports (Failure _ _ _) libdirs = return $ Mod fn Module (UN "error") []
  -- fail "Can't happen (findImports)" -- actually can happen!
        findImports (Success (MD ty mod pt)) libdirs = do
            deps <- fi libdirs pt
            return $ Mod fn ty (UN mod) deps
        fi libdirs [] = return []
        fi libdirs (x:xs) = 
         do -- look in current directory only
            catch (do impfn <- findLib [""] (x++".k")
                      if (impfn `elem` (fn:done))
                         then do putStrLn $ "Warning: " ++
                                              "Circular dependency (in " 
                                              ++ show fn ++ ")"
                                 return []
                         else do
                              modtreeRec <- getDeps impfn opts (fn:done)
                              rest <- fi libdirs xs
                              return (modtreeRec:rest))
                  -- Make sure the ki exists at least!
	          (\e -> do fn <- findLib libdirs (x++".ki")
		            fi libdirs xs)

buildWhat :: Options -> ModuleTree -> IO [CompileNeeded]
buildWhat opts (Mod fn ty mod deps) = do
    if (nochase opts)
       then return [Comp fn ty mod True]
       else do
            needed <- uptodateCheck ty mod fn
            builddeps <- mapM (buildWhat opts) deps
            let alldeps = undup $ concat builddeps
            let buildit = (forcebuild opts) || needed ||
                          (or $ map cn_build alldeps)
            return $ (Comp fn ty mod buildit) : alldeps
 where undup [] = []
       undup (x:xs) | (cn_filename x) `elem` (map cn_filename xs) = undup xs
                    | otherwise = x:(undup xs)

uptodateCheck t mod infile =
    do let outfile = outputfile t mod
       ex <- doesFileExist outfile	  
       if (not ex) then return True
	  else do inmod <- getModificationTime infile
		  outmod <- getModificationTime outfile
		  return (inmod>outmod)

outputfile Module mod = showuser mod ++ ".o"
outputfile Program mod = showuser mod
outputfile Shebang mod = showuser mod
outputfile Webapp mod = showuser mod ++ ".cgi"
outputfile Webprog mod = showuser mod ++ ".cgi"

-- Extract module name and import data from a file.
-- FIXME: It would really be better to use the real parser, if we can
-- find a way of not parsing the bits we don't actually care about. I doubt
-- this is really possible, and this hack should be fine.

quickParse :: String -> String -> Bool -> Result ModuleData
quickParse root inf quick 
    = do let stripped = stripComments inf
         (inf, mty, modname) <- parseModName root stripped
         imports <- case quick of
                      True -> return []
                      _ -> parseImports inf []
         return $ MD mty modname imports

-- strip Comments and spaces (i.e. non code things)

stripComments [] = []
stripComments ('/':'/':xs) = stripComments (dropToEndLine xs)
stripComments ('/':'*':xs) = stripComments (dropToEndComment xs)
stripComments ('"':xs) = case getstr xs of
                            Just (_,rest,_) -> stripComments rest
                            _ -> ""
stripComments (x:xs) = x:(stripComments xs)

dropToEndLine [] = []
dropToEndLine l@('\n':xs) = l
dropToEndLine (x:xs) = dropToEndLine xs

dropToEndComment [] = []
dropToEndComment ('*':'/':xs) = xs
dropToEndComment (x:xs) = dropToEndComment xs

parseModName :: String -> String -> Result (String, InputType, String)
parseModName root [] = fail "Not a module"
parseModName root ('#':'!':cs) = return (cs, Shebang, root)
parseModName root (c:cs) = case span isAlpha (c:cs) of
   ("", _) -> parseModName root cs
   (x,rest) -> if x `elem` ["module","program","webprog","webapp"]
                 then do (rest, nm) <- getName rest "module"
                         return (rest, getty x, nm)
                 else parseModName root rest
  where getty "module" = Module
        getty "program" = Program
        getty "webapp" = Webapp
        getty "webprog" = Webprog

getName cs thing = case span isAllowed (stripspace cs) of
   ("", _) -> fail $ "Can't get " ++ thing ++ " name"
   ("public", rest) -> getName rest thing
   (x, rest) -> if (head (stripspace rest) == ';') 
                   then return (rest, x)
                   else fail "Not an import"
  where stripspace (x:xs) | isSpace x || x == '\n' = stripspace xs
        stripspace xs = xs

-- only chase imports at the start of a line.

parseImports :: String -> [String] -> Result [String]
parseImports [] acc = return acc
parseImports ('\n':c:cs) acc = case span isAlpha (c:cs) of
   ("",_) -> parseImports (c:cs) acc
   ("import",rest) -> case getName rest "import" of
                        (Success (rest, nextImp)) ->
                           parseImports rest (nextImp:acc)
                        _ -> parseImports rest acc
   (_,rest) -> parseImports rest acc
parseImports (c:cs) acc -- = parseImports cs acc
   | isSpace c || c==';' = parseImports cs acc
   | otherwise = return acc