/*
    systemmanager.h - Singleton class to manager loaded System plugins.

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    Portions from Kopete
    Copyright (c) 2001-2005 Kopete Developers <kopete-devel@kde.org>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#ifndef KAMEFUSYSTEMMANAGER_H
#define KAMEFUSYSTEMMANAGER_H

#include <qobject.h>
#include <qvaluelist.h>
#include <qstringlist.h>
#include <kdemacros.h>

class KPluginInfo;

namespace Kamefu {

class System;

typedef QValueList<System*> SystemList;
/**
This class manage all loaded system plugins.
This a singleton class, acces it with SystemManager::self()

@author Michaël Larouche
*/
class KDE_EXPORT SystemManager : public QObject
{
	Q_OBJECT
	
public:
	static SystemManager *self();
	~SystemManager();

	 /**
	 * Returns a list of all available system plugins.
	 *
	 * You can query all information on the plugins through the KPluginInfo
	 * interface.
	 */
	QValueList<KPluginInfo *> availableSystems() const;

	/**
	 * Return a list of all loaded system plugins.
	 */ 
	SystemList loadedSystems() const;

	/**
	 * @brief Search by plugin name. This is the key used as X-KDE-PluginInfo-Name in
	 * the .desktop file, e.g. "kamefu_snes"
	 *
	 * @return The @ref Kamefu::System object found by the search, or a null
	 * pointer if the plugin is not loaded.
	 */
	System *system(const QString &pluginName) const;

	/**
	 * @return the KPluginInfo for the specified system
	 */
	KPluginInfo *systemInfo(const Kamefu::System *system) const;

	/**
	 * Shuts down the plugin manager on shutdown, but first
	 * unloads all plugins asynchronously.
	 *
	 * After 3 seconds all plugins should be removed; what's still left
	 * by then is unloaded through a hard delete instead.
	 *
	 * Note that this call also derefs the plugin manager from the event
	 * loop, so do NOT call this method when not terminating!
	 */
	void shutdown();

	/**
	 * This method check if all the systems are loaded.
	 * @return true if all the systems are loaded.
	 */
	bool isAllSystemsLoaded() const;

	/**
	 * Return the list of all registred MIME type by the system plugins.
	 */
	QStringList getRegistredMimeTypes() const;

	/**
	 * Return a display name of a system based on the MIME type.
	 */
	QString guessSystemNameFromMimeType(const QString &mimeType) const;
	
	/**
	 *	Get a system instance from a MIME type.
	 */
	System *getSystemFromMimeType(const QString &mimeType) const;

public slots:
	/**
	 * Load a single system by plugin name. Returns an existing system
	 * if one is already loaded in memory.
	 *
	 * The plugin will be queued and loaded in
	 * the background. This method will return a null pointer. To get
	 * the loaded plugin you can track the @ref systemLoaded() signal.
	 *
	 * See also @ref system().
	 */
	void loadSystem(const QString &pluginId);

	/**
	 * @brief Unload the plugin specified by @p pluginName
	 */
	bool unloadSystem(const QString &pluginName);

	/**
	 * Load all availables system plugins.
	 */
	void loadAllSystems();

signals:
	/**
	 * Signals a new system plugin has just been loaded.
	 */
	void systemLoaded(Kamefu::System *system);

	/**
	 * All system plugins have been loaded by the system plugin manager.
	 */
	void allSystemsLoaded();

private slots:
	/**
	 * @brief Cleans up some references if the plugin is destroyed
	 */
	void slotSystemDestroyed(QObject *system);

	/**
	 * Common entry point to deref() the KApplication. Used both by the clean
	 * shutdown and the timeout condition of slotShutdownTimeout()
	 */
	void slotShutdownDone();

	/**
	 * Load a system plugin from the queue.
	 */
	void slotLoadNextSystem();

private:
	SystemManager(QObject *parent = 0, const char *name = 0);

	/**
	 * @internal
	 *
	 * The internal method for loading system.
	 * Called by @ref loadPlugin directly or through the queue for async plugin
	 * loading.
	 */
	System *loadSystemInternal(const QString &pluginId);

	/**
	 * @internal
	 *
	 * Find the KPluginInfo structure by key. Reduces some code duplication.
	 *
	 * Returns a null pointer when no plugin info is found.
	 */
	KPluginInfo *infoForPluginId(const QString &pluginId) const;

private:
	class Private;
	Private *d;

	static SystemManager *s_self;
};

}

#endif
